/****************************************************************************
 * A TreeTableModel that can display Refinement and Node objects.
 *
 * @author Jussi Stader
 * Updated: Thu Mar 15 12:37:54 2007
 * @version 2.0
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview.table;

import ix.util.*;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.iface.ui.FourField;
import ix.iface.ui.table.*;
import ix.iface.ui.util.*;

import javax.swing.table.*;
import javax.swing.tree.*;
import java.util.*;


/****************************************************************************
 * A TreeTableModel that can display Refinement and Node objects.
 *****************************************************************************
 */
public class ActionTreeTableModel extends AbstractTreeTableModel 
  implements UIDomainListener
{
  protected UIDomain uiDomain;
  protected int PATTERN_COL = 1;
  private boolean sort = false;

  public ActionTreeTableModel(boolean markable, UIDomain uiDomain) { 
    super(markable);
    setColumnNames();
    setDomain(uiDomain);
  }


  public void setDomain(UIDomain domain) {
    uiDomain = domain;
    uiDomain.addUIDomainListener(this);
    List constructs = uiDomain.getAllRefinements();
    //Debug.noteln("ATTM: Refinements are:", UIUtil.listToDisplay(constructs));
    if (sort)
      constructs = 
	(List)IVUtil.sortNamedCollection(new ArrayList(constructs));   
    //Debug.noteln(" after sort:", UIUtil.listToDisplay(constructs));
    //setData(constructs);
    updateData(constructs);
  }

  /**
   * Makes an IXTreeTableNode from the given object.
   * If the given object is not a TreeNode already, uses the domain to get
   * the object's UIObject.
   */
  public IXTreeTableNode makeIXTreeTableNode(Object theNode){
    TreeNode tNode;
    IXTreeTableNode ixtNode = null;
    if (theNode == null) tNode = null;
    else if (theNode instanceof TreeNode)
      tNode = (TreeNode)theNode;
    else if (theNode instanceof IXObject)
      tNode = (TreeNode)uiDomain.getUIObject((IXObject)theNode);
    else tNode = null;
    if (tNode == null) ixtNode = null;
    else //make a new eNode using the tNode
      ixtNode = new IXTNTreeTableNode(this, tNode);
    return ixtNode;
  }



  /**
   * Gets the cell value for normal cells (only one - the pattern).
   * The name cell is done in the AbstractTreeTableModel method getTreeValueAt.
   * @return the value as an object, null for illegal columns.
   */
  public Object getCellValueAt(Object o, int columnIndex) {
    IXTNTreeTableNode iNode = (IXTNTreeTableNode)o;
    if (columnIndex == PATTERN_COL) {
      LList pattern = getPattern(iNode);
      return UIUtil.listToDisplay(pattern);
    }
    else return null;
  }

  /**
   * Gets the name string for the given row object.
   */
  public String getNameString(TreeNode node) {
    IXTNTreeTableNode iNode = (IXTNTreeTableNode)node;
    return ((Named)iNode.node).getName();
  }
  /**
   * Gets the strings that are to be used as column names.
   */
  void setColumnNames() {
    Class[] classes = {FourField.class, String.class};
    String[] names = {"Name", "Pattern"};
    super.setColumnClasses(classes);
    super.setColumnNames(names);
  }


  protected LList getPattern(IXTNTreeTableNode iNode) {
    Object node = iNode.node;
    LList pattern = null;
    if (node instanceof UIRefinement)
      pattern = ((UIRefinement)node).getPattern();
    else if (node instanceof Refinement)
      pattern = ((Refinement)node).getPattern();
    else if (node instanceof NodeSpec)
      pattern = ((NodeSpec)node).getPattern();
    else Debug.noteln("ActTTM: found node with odd class", node.getClass());
    //Debug.noteln("ActTTM: pattern is", pattern.toString());
    return pattern;
  }


  /**
   * Looks up all refinements and updates the tree model from
   * them. This method retains the expansion information.
   */
  public void reloadData() {
    //Debug.noteln("AcTTM: reloadData");
    //(new Throwable()).printStackTrace();
    List constructs = uiDomain.getAllRefinements();
    //Debug.noteln("ActTTM: Refinements:", UIUtil.listToDisplay(constructs));
    if (sort)
      constructs = 
	(List)IVUtil.sortNamedCollection(new ArrayList(constructs));   
    //Debug.noteln(" after sort:", UIUtil.listToDisplay(constructs));
    List data = new LinkedList(constructs);
    //Debug.noteln("ATTM: as arrayList:", UIUtil.listToDisplay(data));
    updateData(data);
  }

  /**
   * Looks up all refinements and builds the tree model from them,
   * discarding previous data. This method loses the expansion
   * information and will collapse the whole tree. Use reloadData to
   * retain expansion information.
   */
  public void resetData() {
    reloadData();
  }

  public void setSort(boolean sortIt) {
    sort = sortIt;
    reloadData();
  }
  public boolean getSort() {return sort;}


  public boolean isOwnObject(Object o) {
    return ((o instanceof UIRefinement) || (o instanceof Refinement));
  }

  //----------------------------UIDomainListening things--------------------

  public void refinementAdded(RefinementEvent e) {
    if (!uiDomain.isLoading()) addData(e.getRefinement());
  }
  public void refinementRemoved(RefinementEvent e) {
    removeData(e.getRefinement());
  }
  public void constructAdded(UIDomainEvent e) {
    if (!uiDomain.isLoading() && isOwnObject(e.getObject())) 
      addData(e.getObject());    
  }
  public void constructAdded(DomainEvent e) {
    if ((!uiDomain.isLoading()) && (e instanceof RefinementEvent))      
      addData(((RefinementEvent)e).getRefinement());   
  }
  public void constructRemoved(UIDomainEvent e) {
    reloadData(); //no real reference in the event!
  }
  public void constructEdited(UIDomainEvent e) {
    fireObjectChanged(e.getObject());
  }
  public void domainCleared(UIDomainEvent e) {
    clearData();
    fireTableDataChanged();
  }
  //last two done in editor
  public void domainEdited(UIDomainEvent e) {
    //Debug.noteln("AcTTM: domain edited", e.getDomain().getName());
    //Debug.noteln(" tableModel is", this);
    setDomain(e.getDomain());
  }
  public void domainSet(UIDomainEvent e) {
    //Debug.noteln("AcTTM: domain set", e.getDomain().print());
    setDomain(e.getDomain());
  }


}
