/****************************************************************************
 * Mirrors the domain's Refinement and adds UI, Graph, and TreeNode things
 *
 * File: UIRefinement.java
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Mar  1 12:45:27 2007
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview.domain;

import java.util.*;
import javax.swing.tree.*;

import ix.util.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iview.igraph.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iface.ui.tree.*;

/**
 * Mirrors the domain's Refinement and adds UI, Graph, and TreeNode things
 */

public class UIRefinement 
implements UIObject, Named, SchemaTerms, TreeNode {
  public String name;
  public LList pattern = Lisp.NIL;
  public List nodes = new ArrayList();	// List of NodeSpecs
  public List orderings = new ArrayList();	// List of Orderings
  public List constraints = new ArrayList();
  public List issues = new ArrayList();
  public String comments = "";    // annotations
  public List variableDeclarations;    // may be null!!!
  public Annotations annotations;
    
  protected UIDomain uiDomain;
  protected Refinement baseObject; //used to note draft
  protected String baseReference = ""; /** name of the original refinement */
  protected IGraph graph;

  protected List autoPoints;


  private boolean initing = false;

  public UIRefinement(UIDomain uiDomain) {
    super();
    this.uiDomain = uiDomain;
    //uiDomain.addConstruct(this);
  }

  public UIRefinement(UIDomain uiDomain, Refinement original) {
    //this(uiDomain);
    super();
    initing = true;
    this.uiDomain = uiDomain;
    
    try {
      baseObject = (Refinement)original.clone();
    }
    catch (CloneNotSupportedException e) {
      baseObject = (Refinement)makeBaseObject();
    }
    //if (!original.isEmpty()) { //***put back in for efficiency!
    baseReference = original.getName();
    loadFromDomain();
    //}
    uiDomain.addedObject(this, original);
    initing = false;
    //Debug.noteln("UIR new refinement/2", this.getName()+" "+this);
    //(new Throwable()).printStackTrace();
  }



  //-----------------------UIRefinement things--------------------------------

  public UIRefinement cloneThis() {
    UIRefinement copy;
    try {
      copy = (UIRefinement)this.clone();
      copy.baseObject = null;
      copy.baseReference = "";
    }
    catch (CloneNotSupportedException e) {
      copy = new UIRefinement(uiDomain);
      copy.pattern = pattern;
      copy.nodes = nodes;
      copy.orderings = orderings;
      copy.constraints = constraints;
      copy.issues = issues;
      copy.comments = comments;
      copy.annotations = annotations;
      copy.variableDeclarations = variableDeclarations;
      copy.graph = graph;
    }
    return copy;
  }


  public String getName() { return name; }
  public void setName(String newName) {
    if (!IVUtil.sameString(newName,name))
      uiDomain.changeName(this, newName);
  }
  public void setLegalName(String name) {
    //Debug.noteln("UIR: setting legal name", name);
    String old = this.name;
    this.name = name; 
    fireDataChange("name", old, name);
  }
  
  public LList getPattern() {	return pattern; }
  public void setPattern(LList pattern) { 
    LList old = this.pattern;
    this.pattern = pattern; 
    fireDataChange("pattern", old, pattern);
  }
  public void clearPattern() {
    setPattern(Lisp.NIL);
  }

  public List getNodes() { return nodes; }
  public void setNodes(List nodes) { 
    //Debug.noteln("UIR: set nodes");
    List old = new ArrayList(this.nodes);
    this.nodes = IVUtil.sortNodeCollection(nodes); 
    fireDataChange("nodes", old, this.nodes);
  }
  public void updateNodes(List nodes) { 
    //Debug.noteln("UIR: update nodes");
    List originalNodes = new ArrayList(this.nodes); //remember old nodes
    List newNodes = new ArrayList();
    if ((originalNodes == null) || (originalNodes.size() == 0)) {
      //nothing to preserve, so just set the new ones
      this.setNodes(nodes); 
    }
    else if ((nodes == null) || (originalNodes.size() == 0)) {
      //nothing wanted so nothing to preserve, so just set the new ones
      this.setNodes(nodes); 
    }
    else { // real changes, so work out which nodes to keep!
      List oldNodes = new ArrayList(this.nodes); //copy old nodes to go through
      boolean changed = false;
      List todoNodes = new ArrayList(); // new nodes not yet found
      // go through new nodes and see whether they are old
      for (Iterator i=nodes.iterator(); i.hasNext();) {
	NodeSpec node = (NodeSpec)i.next();
	LList pattern = node.getPattern();
	//Debug.noteln("UIR: looking for pattern", pattern);
	NodeSpec oldNode = findNode(oldNodes, pattern);
	if (oldNode != null) { //found it, so use old one
	  //Debug.noteln("  found it", oldNode.getId());
	  newNodes.add(oldNode);
	  // old node taken - cannot be used for a new one twice
	  int nth = oldNodes.indexOf(oldNode);
	  oldNodes.remove(nth);
	}
	else { //did not find it, so it is new or has changed
	  changed = true;
	  todoNodes.add(node); //try to find the original
	}
      }
      List freshNodes = new ArrayList();
      //todoNodes are new nodes that do not yet have an original
      //oldNodes contain those original nodes that have not yet been matched
      // go through not found nodes and see whether they are old ones edited
      for (Iterator i=todoNodes.iterator(); i.hasNext();) {
	NodeSpec node = (NodeSpec)i.next();
	LList pattern = node.getPattern();
	//Debug.noteln("UIR: looking for edited pattern", pattern);
	NodeSpec oldNode = findEditedNode(oldNodes, pattern);
	if (oldNode != null) { //found it, so use old id and new pattern
	  //Debug.noteln("  found it", oldNode.getId());
	  newNodes.add(new UINodeSpec(this.getUIDomain(), 
				      oldNode.getId(), pattern));
	  // old node taken - cannot be used for a new one twice
	  int nth = oldNodes.indexOf(oldNode);
	  oldNodes.remove(nth);
	}
	else { //did not find it, so it is new or has changed
	  changed = true;
	  //Debug.noteln("  NOT found");
	  freshNodes.add(node); //remember to add them
	}
      }
      // add nodes that have no original 
      for (Iterator i=freshNodes.iterator(); i.hasNext();) {
	newNodes.add(i.next()); 
      }
      //  originals that have been deleted will disappear
      if ((oldNodes != null) && (oldNodes.size() > 0)) changed = true;
      if (changed) {
	setNodes(newNodes);
	//next is done in setNodes
	//now make sure old ids are gone from orderings
	//fireDataChange("nodes", originalNodes, this.nodes);
      }
    }
  }
  public void clearNodes() {
    setNodes(new ArrayList());
  }
  

  public List getOrderings() { return orderings; }
  public void setOrderings(List orderings) { 
    //Debug.noteln("UIR: set orderings");
    List old = new ArrayList(this.orderings);
    this.orderings = new ArrayList(orderings); 
    fireDataChange("orderings", old, this.orderings);
  }
  public void clearOrderings() {setOrderings(new ArrayList());}
  //go through orderings and remove any that refer to old nodes
  public void updateOrderings() {
    List os = new ArrayList();    
    NodeEndRef nodeRef;
    NodeSpec node;
    boolean changed = false;
    for (Iterator i = orderings.iterator(); i.hasNext(); ) {
      Ordering o = (Ordering) i.next();
      if (IVUtil.checkNodeRefConsistency(this, o)) 
	os.add(o);
      else changed = true;
    }
    if (changed) this.setOrderings(os);
  }



  public List getConstraints() { return constraints; }
  public void setConstraints(List cs) { 
    List oldConds = new ArrayList(getConditions());
    List old = new ArrayList(this.constraints);
    this.constraints = IVUtil.sortConstraintCollection(new ArrayList(cs)); 
    fireDataChange("constraints", old, this.constraints);
    fireDataChange("conditions", oldConds, getConditions());
  }
  public void clearConstraints() {setConstraints(new ArrayList());}
  /**
   * Removes the given constraint; 
   * @return true if the constraint was removed, false if it was not a
   * constraint in the list.
   * ***********Why no fire change!?
   */
  public boolean deleteConstraint(Constraint c) {
    List constraints = getConstraints();
    if (!constraints.contains(c)) return false;
    constraints.remove(c);
    return true;
  }

  public List getConditions() {
    List constr = getConstraints();
    return filterForConditions(constr);
  }
  public void setConditions(List conditions) {
    //Debug.noteln("UIRefinement: setting conditions");
    //Debug.noteln("   new ones", UIUtil.show(conditions));
    List old = new ArrayList(getConstraints());
    List constr = getConstraints();
    List constOnly = filterOutConditions(constr);
    constOnly.addAll(conditions);
    //Debug.noteln("   all conditions", UIUtil.show(constOnly));
    setConstraints(constOnly);
    fireDataChange("conditions", old, this.constraints);
  }
  public List getTypeConstraints() {
    List constraints = getConstraints(); //keep original in touch
    return IVUtil.filterClassConstraints(constraints);
  }
  public void setTypeConstraints(List typeConstrs) {
    //Debug.noteln("UIRefinement: setting type constraints");
    //Debug.noteln("   new ones", UIUtil.show(typeConstrs));
    List oldAll = getConstraints(); //keep original in touch
    List oldClassCs = IVUtil.filterClassConstraints(oldAll);
    List newAll;
    if (oldAll == null) newAll = new ArrayList();
    else newAll = new ArrayList(oldAll); //original to update
    if (oldClassCs != null)
      newAll.removeAll(oldClassCs); //remove old class constraints
    newAll.addAll(typeConstrs); //add new class constraints
    setConstraints(newAll);
    fireDataChange("constraints", oldAll, this.constraints);
  }

  public Constraint getTypeConstraint(ItemVar var) {
    return IVUtil.findClassConstraint(var, getConstraints());
  }
  public void setTypeConstraint(ItemVar var, Constraint constraint) {
    //Debug.noteln("UIRefinement: setting type constraint for var", var);
    //Debug.noteln("   new constraint:", constraint);
    Constraint oldC = getTypeConstraint(var); 
    List oldAll = getConstraints(); //keep original in touch
    List newAll = new ArrayList(oldAll);
    if (oldC != null) newAll.remove(oldC);
    newAll.add(constraint); //add new class constraints
    setConstraints(newAll);
    fireDataChange("constraints", oldAll, this.constraints);
  }

  /**
   * Find all (world-state) constraints that have the given variable in them.
   * return@ a list of world-state constraints
   */
  public List getVariableConstraints(ItemVar var) {
    if (var == null) return null;
    Set vars = getVariablesUsed();
    if ((vars == null) || !vars.contains(var)) return null; //var not used

    //given var used, so find it in constraints
    List constraints = getConstraints(); //all constraints
    List vConstraints = filterForVariable(var, constraints); //var
    //now we have a list of all constraints that mention the var
    return vConstraints;
  }
  /**
   * Replace all Find all previous (world-state) constraints that have
   * the given variable in them with the given ones.  
   */
  public void setVariableConstraints(ItemVar var, List varConstrs) {
    if (var == null) return;

    List oldAll = getConstraints(); //all constraints
    List oldVarCs = filterForVariable(var, oldAll); //var
    List newAll;
    if (oldAll == null) newAll = new ArrayList();
    else newAll = new ArrayList(oldAll); //original to update
    if (oldVarCs != null)
      newAll.removeAll(oldVarCs); //remove old class constraints
    newAll.addAll(varConstrs); //add new class constraints
    setConstraints(newAll);
    fireDataChange("constraints", oldAll, this.constraints);
  }

  
  /**
   * Gets the list of conditions and effects that are in the constraints.
   * @param constraints a list of constraint objects that may contain
   * conditions or effects.
   * @return conditions/effects only
   */
  public static List filterForConditions(List constraints) {
    List conditions = new ArrayList();
    if (constraints != null) {    
      for (Iterator i = constraints.iterator(); i.hasNext(); ) {
	try { 
	  Constraint c = (Constraint)i.next();
	  if (c.getType().equals(Refinement.S_WORLD_STATE)
	      && (c.getRelation().equals(Refinement.S_CONDITION)
		  || c.getRelation().equals(Refinement.S_EFFECT))) 
	    conditions.add(c);
	} catch (Exception e) {} //only interested in constraints
      }
    }
    //Collections.reverse(conditions);
    //Debug.noteln("Filtering for conditions:", UIUtil.show(conditions));
    return conditions;
  }

  private static List filterOutConditions(List constraints) {
    if (constraints == null) return null;
    List conds = filterForConditions(constraints);
    List constOnly = new ArrayList(constraints);
    if (conds != null) constOnly.removeAll(conds);
    return constOnly;
  }

  /**
   * Searches the given constraints for ones that constrain the
   * variable, i.e. pattern assignments that have the variable as
   * their subject.
   * E.g. looking for ?x constraints: (name ?x = "fred") fits but not
   * (friend ?p = ?x)
   * @param var the variable to be constrained
   * @param constraints a list of constraints that may contain variable
   * constraints.
   * @return a list of variable constraints
   */
  public static List filterForVariable(ItemVar var, List constraints) {
    //Debug.noteln("UIR: filtering for variable", var);
    //Debug.noteln(" in ", UIUtil.show(constraints));
    List vConstraints = new ArrayList();
    if (constraints != null) {
      for (Iterator i = constraints.iterator(); i.hasNext(); ) {
	try { 
	  Constraint c = (Constraint)i.next();
	  PatternAssignment ass = c.getPatternAssignment();
	  if (ass != null) {
	    LList pattern = ass.getPattern();
	    //Debug.noteln("  Pattern is", UIUtil.show(pattern));
	    //Debug.noteln(" i is class", pattern.get(1).getClass());
	    if (IVUtil.isSubjectInPattern(var, pattern)) 
	      vConstraints.add(c);
	  }
	} catch (Exception e) {} //only interested in constraints
      }
    }
    return vConstraints;
  }

  public List getOtherConstraints() {
    //Debug.noteln("UIR: getting other constraints");
    List constr = getConstraints();
    //Debug.noteln(" constraints are", constr);
    //Debug.noteln(" others are", filterOutConditions(constr));
    return filterOutConditions(constr);
  }
  public void setOtherConstraints(List constraints) {
    List old = new ArrayList(getConstraints());
    List constr = getConstraints();
    List condOnly = filterForConditions(constr);
    condOnly.addAll(constraints);
    setConstraints(condOnly);
    fireDataChange("constraints", old, this.constraints);
  }

  public boolean hasOrderings() {
    return (!(orderings == null) && !(orderings.size() == 0));
  }
  public boolean hasConditions() {
    List c = getConditions();
    return (!(c == null) && !(c.size() == 0));
  }
  public boolean hasAnonConstraints() {
    List c = getOtherConstraints();
    return (!(c == null) && !(c.size() == 0));
  }

  public List getIssues() { return issues; }
  public void setIssues(List issues) { 
    List old = new ArrayList(this.issues);
    this.issues = IVUtil.sortTICollection(issues); 
    fireDataChange("issues", old, this.issues);
  }
  //public void addIssue(Issue issue) { issues.add(issue); }
  public void deleteIssue(Issue issue) { 
    List old = new ArrayList(this.issues);
    old.remove(issue);
    setIssues(old); //make sure data change is fired!
  }
  public void clearIssues() {setIssues(new ArrayList());}

  public String getComments() { return comments; }
  public void setComments(String comments) { 
    String old = this.comments;
    this.comments = comments; 
    fireDataChange("comments", old, this.comments);
  }
  public Annotations getAnnotations() { return annotations; }
  public void setAnnotations(Annotations annotations) { 
    Annotations old = this.annotations;
    this.annotations = annotations; 
    fireDataChange("annotations", old, this.annotations);
  }

  /*
  public String toString() {
    return "UIRefinement[" + name + " expands " + pattern + "]";
  }
				  */


  //-----------------------UIObject things------------------------------------
  public Domain getDomain() {
    return uiDomain.getDomain();
  }
  public UIDomain getUIDomain() {
    return uiDomain;
  }

  public void moveToUIDomain(UIDomain uiDomain) {
    if (this.uiDomain == null) this.uiDomain = uiDomain;
    else { //**this should be more difficult - what does it mean?
      this.uiDomain = uiDomain;
      uiDomain.addedObject(this, getBaseObject());
    }
  }


  public Object getBaseReference() {
    return baseReference;
  }
  public void setBaseReference(Object ref) {
    baseReference = (String) ref;
  }


  public IXObject getBaseObject() {
    return baseObject;
  }
  public IXObject makeBaseObject() {
    return new Refinement();
  }
  public void setBaseObject(IXObject object) {
    baseObject = (Refinement)object;
  }
  public Class getBaseClass() {
    return Refinement.class;
  }

  public boolean hasOriginal() {
    if ((getBaseReference() == null) || (getBaseReference().equals("")))
      return false;
    else return true;
  }


  /**
   * Saves this object into a suitable baseObject. Does not add the base object
   * to the domain.
   */
  public void saveToDomain() {
    if (baseObject == null) baseObject = (Refinement)makeBaseObject();
    //Debug.noteln("UIR-saveToDomain: ", baseObject.toString());
    saveToDomain(baseObject);
    //Debug.noteln(" nodes now:", UIUtil.listToDisplay(baseObject.getNodes()));
  }
  public void saveToDomain(Object domainObject) {
    Refinement refinement = (Refinement) domainObject;
    refinement.setName(name);
    refinement.setPattern(pattern);
    //refinement.setNodes(nodes);
    writeNodes(refinement);
    if (orderings.isEmpty()) refinement.setOrderings(null);
    else refinement.setOrderings(orderings);
    if (constraints.isEmpty()) refinement.setConstraints(null);
    else refinement.setConstraints(constraints);
    if (issues.isEmpty()) refinement.setIssues(null);
    else refinement.setIssues(issues);
    if (annotations == null) refinement.setAnnotations(null);
    else refinement.setAnnotations(new Annotations(annotations));
    refinement.setComments(comments);
    refinement.setVariableDeclarations(variableDeclarations);
    //refinement.setAnnotation("mGraph", graph);
    saveCanvasPositionGraphToDomain(refinement);
  }

  /**
   * Loads this object from its baseObject. 
   * ***Think about undoing this!!
   */
  public void loadFromDomain() {
    if (baseObject == null) {
      baseObject = (Refinement)makeBaseObject();
    }
    name = baseObject.getName();
    if (baseObject.getPattern() == null) pattern = Lisp.NIL;
    else pattern = baseObject.getPattern();
    makeNodes(baseObject.getNodes());
    if (baseObject.getOrderings() == null) orderings = new ArrayList();
    else orderings = new ArrayList(baseObject.getOrderings());
    if (baseObject.getConstraints() == null)
      constraints = new ArrayList();
    else constraints = new ArrayList(baseObject.getConstraints());
    if (baseObject.getIssues() == null) issues = new ArrayList();
    else setIssues(baseObject.getIssues());
    annotations = baseObject.getAnnotations();
    if (annotations != null) annotations = new Annotations(annotations); 
    comments = baseObject.getComments();
    variableDeclarations = baseObject.getVariableDeclarations();
    //graph = (IGraph)baseObject.getAnnotation("mGraph");
    loadCanvasPositionGraphFromDomain(baseObject);
    fireDataChange("all", null, null);
  }

  public List getVariableDeclarations() {
    return variableDeclarations;
  }
  public void setVariableDeclarations(List vars) {
    List old = this.variableDeclarations;
    variableDeclarations = vars;
    fireDataChange("variableDeclarations", old, this.variableDeclarations);
    //Debug.noteln("Refinement now: ", print());
  }

  public SortedSet getVariablesUsed() {
    Refinement refinement = new Refinement();
    saveToDomain(refinement);
    return refinement.getVariablesUsed();
  }

  

  public void loadFromDomain(Object domainObject) {
    baseObject = (Refinement)domainObject;
    loadFromDomain();
  }


  public void saveMGraphToDomain(Refinement refinement) {
    refinement.setAnnotation("mGraph", graph);
  }
  public void saveCanvasPositionGraphToDomain(Refinement refinement) {
    if (graph != null) {
      //read them from the graph
      List points = graph.getGraphCanvasPoints();
      if ((points == null) || points.isEmpty())
	//perhaps no graph but previous canvas points
	points = graph.getCanvasPoints();
      if ((points != null) && !points.isEmpty() && 
	  ((autoPoints == null) || (!Collect.equalAsSets(points, autoPoints))))
	refinement.setAnnotation("canvasPositionGraph", points);
      //no current graph info, so get rid of old graph info if there is any
      else {
	Object oldGraph = refinement.getAnnotation("canvasPositionGraph");
	if (oldGraph != null)
	  refinement.setAnnotation("canvasPositionGraph", null);
      }
    }
  }

  public void loadMGraphFromDomain(Refinement refinement) {
    graph = (IGraph)refinement.getAnnotation("mGraph");
  }
  public void loadCanvasPositionGraphFromDomain(Refinement refinement) {
    //get the posiitions
    Object cpGraph = refinement.getAnnotation("canvasPositionGraph");
    if (cpGraph == null) graph = null;
    else {
      IGraph graph = graphFromPosition(refinement, cpGraph);
      if (graph != null) {
	this.graph = graph;
	uiDomain.addGraph(graph);
      }
    }
  }

  public IGraph graphFromPosition(Refinement refinement, Object pos) {
    IGraph graph = null;
    if (pos instanceof HashMap) {
      if (!((HashMap)pos).isEmpty()) {
	graph = new IGraph(refinement.getName(), (HashMap)pos);
      }
    }
    else if (pos instanceof List) 
      if (!((List)pos).isEmpty())  {
	graph = new IGraph(refinement.getName(), (List)pos);
      }
    return graph;
  }

  public IXObject getOriginal() {
    Object ref = getBaseReference();
    //Debug.noteln("UIR: base ref is", ref);
    return getDomain().getNamedRefinement((String)ref);
  }

  /**
   * Loads this object from its original in the domain if it has one, otherwise
   * clear the object. Replaces the base-object with a fresh clone of the 
   * original or with a new base-object.
   */
  public void loadFromOriginal() {
    if (hasOriginal()) {
      Refinement original = (Refinement)getOriginal();
      try {
	setBaseObject((Refinement)original.clone());
	loadFromDomain();
      }
      catch (Exception e) { //CloneNotSupported or null pointer
	Debug.noteException(e);
	Debug.noteln("Continuing with new object as the original");
	setBaseObject(makeBaseObject());
	loadFromDomain();
      }
    }
    else {
      setBaseObject(makeBaseObject());
      loadFromDomain();
    }
  }


  private void makeNodes(List originalNodes) {
    ArrayList newNodes = new ArrayList();
    if (originalNodes != null) 
      for (Iterator i = originalNodes.iterator(); i.hasNext(); ) 
	newNodes.add(new UINodeSpec(uiDomain, (NodeSpec)i.next()));
    setNodes(newNodes); //make sure they are sorted by ID
  }
  private void writeNodes(Refinement refinement) {
    if ((nodes == null) || nodes.isEmpty()) refinement.setNodes(null);
    else {
      List adds = new LListCollector();
      for (Iterator i = nodes.iterator(); i.hasNext(); ) {
	NodeSpec ns = (NodeSpec)i.next();
	adds.add(new NodeSpec(ns.getId(), ns.getPattern()));
      }
      refinement.setNodes(adds);
    }
  }
  public boolean sameNodes(List nodes, List otherNodes) {
    if (nodes == null)
      return ((otherNodes == null) || (otherNodes.size() == 0));
    else if (otherNodes == null)
      return ((nodes == null) || (nodes.size() == 0));

    if (nodes.size() != otherNodes.size()) return false;
    else {
      //hash one list with id/pattern, then lookup nodes by ids
      Hashtable hNodes = new Hashtable();
      //Debug.noteln("UIR: nodes");
      for (Iterator i = nodes.iterator(); i.hasNext(); ) {
	NodeSpec ns = (NodeSpec)i.next();
	//Debug.note(" " + ns.getId() + UIUtil.listToDisplay(ns.getPattern()));
	hNodes.put(ns.getId(), ns.getPattern());
      }
      //Debug.noteln("UIR: other nodes");
      for (Iterator i = otherNodes.iterator(); i.hasNext(); ) {
	NodeSpec ns = (NodeSpec)i.next();
	//Debug.note(" " + ns.getId() + UIUtil.listToDisplay(ns.getPattern()));
	LList p1 = (LList)hNodes.get(ns.getId());
	LList p2 = ns.getPattern();
	if (p2 == null) {
	  if (p1 != null) return false;
	}
	else if (!p2.equal(p1)) return false;
      }
      return true;
    }
  }
  private boolean sameIssues(List issues, List otherIssues) {
    if (IVUtil.bothEmpty(issues, otherIssues)) return true;
    else if ((issues == null) || (otherIssues == null)) return false;

    //compare issues by patterns only!
    else if (issues.size() != otherIssues.size()) return false;
    else {
      //hashSet one list with patterns only, lookup other list's patterns
      HashSet hIssues = new HashSet();
      //Debug.noteln("UIR: issues");
      for (Iterator i = issues.iterator(); i.hasNext(); ) {
	Issue issue = (Issue)i.next();
	//Debug.note(" " + issue.getId());
	//Debug.note(" " + UIUtil.listToDisplay(issue.getPattern()));
	hIssues.add(issue.getPattern());
      }
      //Debug.noteln("UIR: other issues");
      for (Iterator i = otherIssues.iterator(); i.hasNext(); ) {
	Issue issue = (Issue)i.next();
	//Debug.note(" " + issue.getId());
	//Debug.note(" " + UIUtil.listToDisplay(issue.getPattern()));
	if (!hIssues.contains(issue.getPattern())) {
	  //Debug.note("UIR: Could not find issue pattern ");
	  //Debug.noteln(UIUtil.listToDisplay(issue.getPattern()));
	  return false;
	}
      }
      return true;
    }
  }

  private boolean sameOrderings(Object value, Object otherValue) {
    try { //both collections
      return IVUtil.sameSet((Collection)value, (Collection)otherValue);
    }
    catch (ClassCastException cce1) {
      try { //both ordering types
	return ((String)value).equals((String)otherValue);
      }
      catch (ClassCastException cce2) {
	try {
	  if (value instanceof String) return value.equals(getOrderingType());
	  else if (otherValue instanceof String) 
	    return otherValue.equals(getOrderingType((Collection)value));
	  else return value.equals(otherValue);
	}
	catch (Exception e) {
	  Debug.noteln("Failure to compare ordering values;" + 
		       " assuming values differ.");
	  Debug.describeException(e);
	  return false;
	}
      }


    }
    

  }

  private boolean emptyPositions(Object positions) {
    if (positions == null) return true;
    if ((positions instanceof Collection) && ((Collection)positions).isEmpty())
      return true;
    if ((positions instanceof Map) && ((Map)positions).isEmpty())
      return true;
    //Debug.noteln("UIR: positions object class", positions.getClass());
    return false;
  }

  /**
   * 
   */
  private boolean sameGraph(IGraph graph, Object positions) {
    //no graph yet, so cannot have edited the graph
    if (graph == null) return true;
    
    //empty positions and default graph means no changes!
    if (emptyPositions(positions) && graph.isSameLayout(autoPoints)) 
      return true;
    //compare graph to given positions (can handle null positions)
    else return graph.isSameLayout(positions);
   

    /*
      Debug.noteln("same node sets for", UIUtil.show((List)positions));
      Debug.noteln("  and", UIUtil.show(graphPositions));
      Debug.noteln(" is",  new Boolean(Collect.equalAsSets((List)positions, 
          graphPositions))); 
    */
    /*
    List graphPositions = graph.getCanvasPoints();
    if (IVUtil.bothEmpty((List)positions, graphPositions)) return true;
    if ((positions == null) || (graphPositions == null)) return false;
    return Collect.equalAsSets((List)positions, graphPositions);   
    */ 
  }



  //------------------------updating the original domain-------

  public void addToDomain() { addToDomain(getDomain()); }
  public void addToDomain(Domain domain) {
    //If there is no base-object, make a new one.
    if (baseObject == null) baseObject = (Refinement)makeBaseObject();

    //if there already is such an object, compare and complain if not equal

    
    //copy this into the base object (to be safe)
    saveToDomain();

    String name = baseObject.getName();
    Refinement ref = domain.getNamedRefinement(name);
    if (ref != null) {
      if (baseObject.equals(ref)) return; //nothing to be done
      else {
	Debug.noteln("UIR: Trying to add refinement that is already there");
	saveToDomain(ref);
	return;
      }
    }

    // then pass the base object to the domain.
    domain.addRefinement(baseObject);
  }
  public void updateInDomain() throws DomainReferenceException { 
    updateInDomain(getDomain()); 
  }
  public void updateInDomain(Domain domain) throws DomainReferenceException {
    //if there is no base object, complain (add instead?)
    String ref = (String)getBaseReference();
    Refinement r = getReferredObject(domain, ref, name);
    if (r == null) {
      Debug.noteln("Update refinement: cannot find base object called", ref);
      Debug.noteln(" Domain is", domain);
      return;
    }
    //copy this into base object (to be safe) and pass it with its id to the
    //domain
    //Debug.noteln("UIR: Update refinement base is ", ref);
    saveToDomain();
    domain.replaceRefinement(r, baseObject);
  }
  public void removeFromDomain() throws DomainReferenceException { 
    removeFromDomain(getDomain()); 
  }
  public void removeFromDomain(Domain domain) throws DomainReferenceException {
    //if there is no base object id, try to find the id in the domain
    //Complain if it cannot be found
    String ref = (String)getBaseReference();
    //Debug.noteln("UIR: removing refinement " + ref + " or", name);
    Refinement r = getReferredObject(domain, ref, name);
    if (r == null) {
      Debug.noteln("Remove refinement: Cannot find a base object called ",ref);
      return;
    }
    //give the object to the domain and ask for removal
    //Debug.noteln("...deleting");
    domain.deleteRefinement(r);
  }

  private Refinement getReferredObject(Domain domain, 
				       String reference, String name) 
    throws DomainReferenceException
  {
    String ref;
    if ((reference == null) || (reference.equals(""))) 
      ref = name;
    else ref = reference;
    Refinement found = domain.getNamedRefinement(ref);
    if (found == null) {
      List all = domain.getRefinements();
      throw new DomainReferenceException("Cannot find Refinement " + ref +
					 " in domain " + domain.toString() +
					 " with refinements " + 
					 IVUtil.namedListToDisplay(all));
    }
    else 
      return found;
  }

  /**
   * Get the UIDomain to note the change of this object
   */
  public void noteChange() {
    if (isEmpty() && !hasOriginal()) 
      uiDomain.removeConstruct(this); //empty with no original. Get rid of it!
    else if (hasChanged()) {
      //saving, so update the base object
      saveToDomain();

      //Debug.noteln("UIR: noting change");
      uiDomain.updateConstruct(this);
    }
  }


  /**
   * A refinement is undefined if all its data fields are empty except
   * the pattern which may have data in it.  
   */
  public boolean isUndefined() {
    if ((name == null || name.equals("") || 
	 name.equals("Undefined") || name.equals("undefined"))
	&& (nodes == null || nodes.isEmpty())
	&& (orderings == null || orderings.isEmpty())
	&& (issues == null || issues.isEmpty())
	&& ((variableDeclarations == null) || variableDeclarations.isEmpty())
	&& (graph == null)
	&& (comments == null || (comments.equals("")))
	&& (getAnnotations() == null || (getAnnotations().isEmpty()))) {
      //Debug.noteln("UIRefinement is empty");
      return true;
    }
    //Debug.noteln("refinement is not empty");
    return false;
  }
  /**
   * A refinement is empty if all its data fields are empty.
   * Note: the refinement may not have started out as empty - check whether it
   * has a base-object for full emptiness.
   */
  public boolean isEmpty() {
    if ((name == null || name.equals("") || 
	 name.equals("Undefined") || name.equals("undefined"))
	&& (pattern == null || pattern.isEmpty())
	&& (nodes == null || nodes.isEmpty())
	&& (orderings == null || orderings.isEmpty())
	&& (issues == null || issues.isEmpty())
	&& ((variableDeclarations == null) || variableDeclarations.isEmpty())
	&& (graph == null)
	&& (comments == null || (comments.equals("")))
	&& (getAnnotations() == null || (getAnnotations().isEmpty()))) {
      //Debug.noteln("UIRefinement is empty");
      return true;
    }
    //Debug.noteln("refinement is not empty");
    return false;
  }
   /**
   * A refinement is empty if all its data fields are empty.
   * Note: the refinement may not have started out as empty - check whether it
   * has a base-object for full emptiness.
   */
  public boolean isEmpty(Refinement refinement) {
    if (refinement == null) return true;
    if ((refinement.getName() == null || (refinement.getName().equals("")))
	&& (refinement.getPattern() == null || 
	    refinement.getPattern().isEmpty())
	&& (refinement.getNodes() == null || refinement.getNodes().isEmpty())
	&& (refinement.getOrderings() == null || 
	    refinement.getOrderings().isEmpty())
	&& (refinement.getVariableDeclarations() == null || 
	    refinement.getVariableDeclarations().isEmpty())
	&& (refinement.getIssues() == null || refinement.getIssues().isEmpty())
	&& (refinement.getComments() == null || 
	    refinement.getComments().equals(""))
	&& (refinement.getAnnotations() == null || 
	    refinement.getAnnotations().equals(""))) {
      //Debug.noteln("refinement is empty");
      return true;
    }
    //Debug.noteln("refinement is not empty");
    return false;
  }
  
  /** 
   * Finds out whether the refinement has changed from the base-object 
   * (i.e. since the object was last saved to draft).
   */
  public boolean hasChanged() {
    //return hasChanged((Refinement)getBaseObject());
    return hasChangedNoDebug((Refinement)getBaseObject());
  }
  /**
   * Collects the differences between this object and the given one.
   * @return a list of strings describing changes
   */
  public List collectChanges(Refinement original) {
      return collectChanges(original, "old");
  }
  public List collectChanges(Refinement original, String which) {
    List changes = new ArrayList();
    if (isEmpty(original) && isEmpty()) return changes;
    //created when "editing" a sub-node (undefined, pattern set, nothing else)
    //this does not count as a change
    if (isEmpty(original) 
	&& ((name == null) ||
	    (name.equals("Undefined") || name.equals("undefined")))
	&& (nodes == null || nodes.isEmpty())
	&& (orderings == null || orderings.isEmpty())
	&& (constraints == null || constraints.isEmpty())
	&& (issues == null || issues.isEmpty())
	&& (comments == null || (comments.equals("")))
	&& (annotations == null || (annotations.equals("")))
	&& (variableDeclarations == null)
	&& (graph == null))
      return changes;
    if (original == null) {
	if (isEmpty()) return changes;
	else {
	    changes.add("The " + which + 
			" refinement is empty, this panel is not:");
	    changes.add("  " + print());
	    return changes;
	}
    }
    else {
	if (nameHasChanged(original.getName())) 
	    changes.add("Name has changed from " + original.getName() +
		" to " + getName());
	if (! IVUtil.sameList(pattern,original.getPattern())) 
	    changes.add("Pattern has changed from " + original.getPattern() +
		" to " + pattern);
	if (!sameNodes(nodes,original.getNodes())) 
	    changes.add("Nodes have changed from " + 
		IVUtil.printNodes(original.getNodes()) +
		" to " + IVUtil.printNodes(nodes));
	if (!IVUtil.sameSet(orderings, original.getOrderings()))
	    changes.add("Orderings have changed from " + 
		IVUtil.printOrderings(original.getOrderings()) +
		" to " + IVUtil.printOrderings(orderings));
	if (!IVUtil.sameList(constraints, original.getConstraints()))  
	    changes.add("Constraints have changed from " + 
		IVUtil.printConstraints(original.getConstraints()) +
		" to " + IVUtil.printConstraints(constraints));
	if (!sameIssues(issues, original.getIssues()))  
	    changes.add("Issues have changed from " + 
		IVUtil.textRenderPatterns(original.getIssues()) +
		" to " + IVUtil.textRenderPatterns(getIssues()));
	if (!IVUtil.sameString(comments,original.getComments()))  
	    changes.add("Comments have changed from " + 
		original.getComments() + " to " + comments);
	if (!IVUtil.sameMap(annotations,original.getAnnotations()))  
	    changes.add("Annotations have changed from " + 
		original.getAnnotations() + " to " + annotations);
	if (!IVUtil.sameList(variableDeclarations, 
			     original.getVariableDeclarations())) 
	    changes.add("Variable declarations have changed from " + 
		IVUtil.printVarDecs(original.getVariableDeclarations()) +
		" to " + IVUtil.printVarDecs(getVariableDeclarations()));
	if (!sameGraph(graph, original.getAnnotation("canvasPositionGraph")))
 	    changes.add("Graph has changed");
	Debug.noteln("Changes are:", changes);
    }
   return changes;				   
  }
  /**
   * Collects the differences between this object and the base one.
   * @return a list of strings describing changes
   */
  public List collectChanges() {
    return collectChanges((Refinement)getBaseObject(), "draft");
  }
  /**
   * Collects the differences between this object and the base one.
   * @return a list of strings describing changes
   */
  public List collectChangesFromOriginal() {
    Refinement original = getDomain().getNamedRefinement(baseReference);
    return collectChanges(original, "published");
  }
  /** 
   * Finds out whether the refinement has changed from the given object.
   * The refinement has not changed if the fields have not changed.
   * The refinement's fields are:
   * name, pattern, nodes, orderings, constraints, issues, comments
   * The refinement has also not changed if the original is null and it is
   * empty.
   */
  public boolean hasChangedNoDebug(Refinement original) {
    //created when "editing" a sub-node (undefined, pattern set, nothing else)
    if (isEmpty(original) 
	&& (name != null) && (name.equals("Undefined") 
			      || name.equals("undefined"))
	&& (nodes == null || nodes.isEmpty())
	&& (orderings == null || orderings.isEmpty())
	&& (constraints == null || constraints.isEmpty())
	&& (issues == null || issues.isEmpty())
	&& (comments == null || (comments.equals("")))
	&& (annotations == null || annotations.isEmpty())
	&& (variableDeclarations == null || variableDeclarations.isEmpty())
	&& (graph == null))
      return false;
    if (original == null) return !isEmpty();
    if (!nameHasChanged(original.getName()) && 
	IVUtil.sameList(pattern,original.getPattern()) && 
	sameNodes(nodes,original.getNodes()) && 
	IVUtil.sameSet(orderings, original.getOrderings()) && 
	IVUtil.sameList(constraints, original.getConstraints()) && 
	sameIssues(issues, original.getIssues()) && 
	IVUtil.sameString(comments,original.getComments()) && 
	IVUtil.sameMap(annotations,original.getAnnotations()) && 
	IVUtil.sameList(variableDeclarations, original.getVariableDeclarations()) && 
	sameGraph(graph, original.getAnnotation("canvasPositionGraph")))
      return false;
    else return true;				   
  }
  /** 
   * Finds out whether the refinement has changed from the given object.
   * The refinement has not changed if the fields have not changed.
   * The refinement's fields are:
   * name, pattern, nodes, orderings, constraints, issues, comments
   * The refinement has also not changed if the original is null and it is
   * empty.
   */
  public boolean hasChanged(Refinement original) {
    Debug.noteln("UIR: Checking hasChanged for ", print());
    if (isEmpty(original))
      Debug.noteln(" against an empty one");
    else Debug.noteln(" against", original.toString());
    //created when "editing" a sub-node (undefined, pattern set, nothing else)
    if (isEmpty(original) && isEmpty()) {
      Debug.noteln(" -no change");
      return false;
    }
    if (isEmpty(original)
	&& (name != null) && (name.equals("Undefined") 
			      || name.equals("undefined"))
	&& (nodes == null || nodes.isEmpty())
	&& (orderings == null || orderings.isEmpty())
	&& (constraints == null || constraints.isEmpty())
	&& (issues == null || issues.isEmpty())
	&& (comments == null || (comments.equals("")))
	&& (annotations == null || annotations.isEmpty())
	&& (variableDeclarations == null)
	&& (graph == null)) {
      Debug.noteln("Empty sub-node editing - no change");
      return false;
    }
    if (original == null) {
      Debug.noteln("UIR hasChanged from null original");
      return !isEmpty();
    }
    if (!nameHasChanged(original.getName())) {
      if (IVUtil.sameList(pattern, original.getPattern())) {
	if (sameNodes(nodes,original.getNodes())) {
	  if (IVUtil.sameSet(orderings, original.getOrderings())) {
	    if (IVUtil.sameList(constraints, original.getConstraints())) {
	      if (sameIssues(issues, original.getIssues())) {	
		if (IVUtil.sameString(comments,original.getComments())) { 
		  if (IVUtil.sameList(variableDeclarations, 
			       original.getVariableDeclarations())) {
		    if (IVUtil.sameMap(annotations,original.getAnnotations())){
		      if (sameGraph(graph, original.
				    getAnnotation("canvasPositionGraph"))) {
			Debug.noteln("***No Change");
			return false;
		      } else Debug.noteln("UIR hasChanged graph");
		    } else Debug.noteln("UIR hasChanged annotations");
		  } else Debug.noteln("UIR hasChanged variable declarations");
		} else Debug.noteln("UIR hasChanged comments");
	      } else Debug.noteln("UIR hasChanged issues");
	    } else Debug.noteln("UIR hasChanged constraints");
	  } else Debug.noteln("UIR hasChanged orderings");
	} else Debug.noteln("UIR hasChanged nodes");
      } else Debug.noteln("UIR hasChanged pattern");
    } else Debug.noteln("UIR hasChanged name from", original.getName());
    //Debug.noteln(" - no change");
    return true;				   
  }

  /** 
   * Finds out whether the refinement has changed from the original.
   * The refinement has not changed if there is an original (baseReference)
   * and the refinement's fields have not changed.
   * The refinement's fields are:
   * name, pattern, nodes, orderings, constraints, issues, annotations
   */
  public boolean hasChangedFromOriginal() {
    //has no original so it is new (changed!)
    if (!this.hasOriginal()) return true;

    //has original, so compare the two
    Refinement original = getDomain().getNamedRefinement(baseReference);
    return hasChangedNoDebug(original);
  }

  public boolean nameHasChanged(Object nameThing) {
    String original = (String) nameThing;
    if ((((name == null) || name.equals("")) &&
	 ((original == null) || (original.equals("")))) ||
	((name != null) && name.equals(original)))
      return false;
    else return true;
  }

  public boolean sameValue(String field, Object value, Object otherValue) {
    if (field == null) return false;
    if (value == null) return UIUtil.isEmptyThing(otherValue);
    if (otherValue == null) return false;
    if (field.equals("name") || field.equals("comments"))
      return value.equals(otherValue);
    if (field.equals("nodes")) 
      return sameNodes((List)value, (List)otherValue);
    if (field.equals("constraints") || 
	field.equals("conditions") || field.equals("variableDeclarations")) 
      return IVUtil.sameSet((Collection)value, (Collection)otherValue);
    if (field.equals("orderings")) return sameOrderings(value, otherValue);
    if (field.equals("issues")) 
      return sameIssues((List)value, (List)otherValue);
    //if (field.equals("graph")) 
    //  return sameGraph(value, otherValue);
    if (field.equals("annotations")) 
      return IVUtil.sameMap((Map)value, (Map)otherValue);
    if (field.equals("pattern")) 
      return IVUtil.sameList((List)value, (List)otherValue);
    return value.equals(otherValue); //safe exit
  }

  public boolean setValue(String field, Object value) {
    //Debug.noteln("UIR: setting value for field", field);
    if (field == null) return false;
    try {
      if (field.equals("name")) setName((String) value);
      else if (field.equals("comments")) setComments((String) value);
      else if (field.equals("annotations")) setAnnotations((Annotations)value);
      else if (field.equals("nodes")) updateNodes((List)value);
      else if (field.equals("orderings")) setOrderings((List)value);
      else if (field.equals("issues")) setIssues((List)value);
      //if (field.equals("graph")) 
      //  return sameGraph(value, otherValue);
      else if (field.equals("pattern")) setPattern((LList)value);
      else if (field.equals("constraints")) setConstraints((List)value);
      else if (field.equals("miniConstraints")) setOrderingType((String)value);
      else if (field.equals("conditions")) setConditions((List)value);
      else if (field.equals("variableDeclarations")) 
	setVariableDeclarations((List)value);
      else {
	Debug.noteln("UIRefinement cannot set value for field", field);
	if (value != null) Debug.noteln(" value class is", value.getClass());
	return false;
      }
      return true;
    }
    catch (Exception e) {
      Debug.describeException(e);
      return false;
    }	
  }

  public void checkConsistency() {
    Refinement checkRef = new Refinement();
    saveToDomain(checkRef);
    checkRef.checkConsistency();
  }

  //-----------------------UI service things-----------------------------------


  /**
   * Builds a LList of nodes from a string that contains node descriptions.
   * 
   * @param expansion a string containing node descriptions (patterns)
   * seperated by new lines.
   * @return a LList of Nodes.
   */
  public static List parseNodes (UIDomain uiDomain, String expansion) {
    List nodes = new ArrayList();
    long i = 1;
    // For some reason, it doesn't work on Windows to
    // use lineSpearator here.
    while (!expansion.equals("")) {
      // String[] parts =
      //    Util.breakStringAtFirst(expansion, lineSeparator);
      String[] parts = UIUtil.breakStringAtFirst(expansion, "\n");
      Long index = new Long(i);
      LList childPattern = Lisp.elementsFromString(parts[0]);
      if (!childPattern.isNull()) {
	nodes.add(new UINodeSpec(uiDomain, index, childPattern));
	//Debug.noteln("got child", index + childPattern.toString());
	//nodes.addElement(new DefaultNode(index, childPattern));
	i++;
      }
      expansion = parts[1];
    }
    return nodes;
  }

  /**
   * Builds a LList of nodes from a string that contains node descriptions.
   * 
   * @param expansion a string containing node descriptions (patterns)
   * seperated by new lines.
   * @return a LList of Nodes.
   */
  public static List parseIssues (UIDomain uiDomain, String sIssues) {
    List issues = new ArrayList();
    long i = 1;
    // For some reason, it doesn't work on Windows to
    // use lineSpearator here.
    while (!sIssues.equals("")) {
      // String[] parts =
      //    Util.breakStringAtFirst(sIssues, lineSeparator);
      String[] parts = UIUtil.breakStringAtFirst(sIssues, "\n");
      LList pattern = Lisp.elementsFromString(parts[0]);
      if (!pattern.isNull()) {
	Issue newIssue = new Issue(pattern);
	issues.add(newIssue);
      }
      sIssues = parts[1];
    }
    return issues;
  }

  public void deleteNode(NodeSpec node) {
    deleteNodeConstraints(node);
    List oldNodes = new ArrayList(this.nodes);
    oldNodes.remove(node);
    setNodes(oldNodes); //make sure data change is fired!
  }

  protected void deleteNodeConstraints(NodeSpec node) {
    Name nodeId = node.getId();
    //Debug.noteln("Removing orderings for node", nodeId);
    List os = new ArrayList();    
    for (Iterator i = orderings.iterator(); i.hasNext(); ) {
      Ordering constraint = (Ordering) i.next();
      if (!constraintMentionsNode(constraint, nodeId))
	os.add(constraint);
    }
    this.setOrderings(os);
  }

  public boolean matchesPattern(LList pattern) {
    //Debug.noteln("UIR: checking "+pattern.toString()+" in", getPattern());
    if (((pattern == null) && (getPattern() == null)) ||
	(pattern.equal(getPattern())) || 
	(pattern.equals(getPattern())))
      return true;
    else return (null != SimpleMatcher.match(pattern, getPattern()));
  }

  public NodeSpec findNode(Name id) {
    for (Iterator i = nodes.iterator(); i.hasNext(); ) {
      NodeSpec node = (NodeSpec)i.next();
      if (id.equals(node.getId())) return node;
    }
    return null;
  }
  /**
   * Tries to find a node within this refinement whose pattern matches
   * the given pattern.
   * @return the first node spec whose pattern matches the given
   * pattern, null if none match.
   */
  public NodeSpec findNode(LList pattern) {
    for (Iterator i = nodes.iterator(); i.hasNext(); ) {
      NodeSpec node = (NodeSpec)i.next();
      if (pattern.equals(node.getPattern())) return node;
    }
    return null;
  }
  /**
   * Tries to find a node in the given list whose pattern matches the
   * given pattern.
   * @return the first node spec whose pattern matches the given
   * pattern, null if none match.
   */
  public NodeSpec findNode(List nodeList, LList pattern) {
    for (Iterator i = nodeList.iterator(); i.hasNext(); ) {
      NodeSpec node = (NodeSpec)i.next();
      if (pattern.equals(node.getPattern())) return node;
    }
    return null;
  }
  /**
   * Tries to find a node in the given list with a similar pattern.
   * Similar patterns are those that ***.
   * @return the first node spec whose pattern is similar to the given
   * pattern, null if none are similar.
   */
  public static NodeSpec findEditedNode(List oldNodes, LList pattern) {
    if (oldNodes == null) return null;
    for (Iterator i = oldNodes.iterator(); i.hasNext(); ) {
      NodeSpec node = (NodeSpec)i.next();
      if (similarPattern(pattern, node.getPattern())) return node;
    }
    return null;
  }

  public static boolean similarPattern(LList patternOne, LList patternTwo) {
    String sPatOne = patternOne.toString();
    String sPatTwo = patternTwo.toString();
    if (sPatOne == sPatTwo) return true;
    else if (sPatOne == null) return false;
    else if (sPatTwo == null) return false;
    else if (sPatOne.matches(sPatTwo)) return true;
    else if (sPatOne.equalsIgnoreCase(sPatTwo)) return true;
    else {
      /*
      Object carOne = patternOne.car();
      Object carTwo = patternTwo.car();
      //same first element and others same except one
      if ((carOne == carTwo) ||
	  ((carOne != null) && carOne.equals(carTwo)))
	return sameButOne(patternOne.cdr(), patternTwo.cdr());
      else
      */
	return false;
    }
  }

  public static boolean constraintMentionsNode(Constraint c, Name nodeId) {
    Debug.noteln("*****UIRef: cannot see whether constraint mentions node");
    return false;
  }
  public static boolean constraintMentionsNode(Ordering o, Name nodeId) {
    NodeEndRef ref;
    ref = o.getFrom();
    if ((ref != null) && (nodeId.equals(ref.getNode()))) return true;
    else {
      ref = o.getTo();
      if ((ref != null) && (nodeId.equals(ref.getNode()))) return true;
      else return false;
    }
  }

  public Name getNewNodeId() {
    Name largest = getHighestNodeId();
    Integer l = new Integer(largest.toString());
    int i = l.intValue();
    i++;
    return Name.valueOf("" + i); 
  }
  public Name getHighestNodeId() {
    return getHighestNodeId(getNodes());
  }

  public static Name getHighestNodeId(List nodes) {
    Integer largest = new Integer(0);
    Name largestId = Name.valueOf(largest.toString());
    int c;
    Iterator inodes = nodes.iterator();
    while (inodes.hasNext()) {
      Name id = ((NodeSpec)inodes.next()).getId();
      try {
	Integer idInt = new Integer(id.toString());
	c = largest.compareTo(idInt);
	if (c <= 0) { largestId = id; largest = idInt; }
      }
      catch (NumberFormatException nfe) { //id not an int, so keep old largest
      }
    }
    return largestId;
  }

  public void addNode(NodeSpec node) {
    //Debug.noteln("UIR: addNode - nodes list is class", nodes.getClass());
    List newNodes = new ArrayList(nodes);
    newNodes.add(node);
    setNodes(newNodes); //setNodes sorts the nodes and fires
  }
  public void addIssue(Issue issue) {
    List newIssues = new ArrayList(issues);
    newIssues.add(issue);
    setIssues(newIssues); //setIssues sorts and fires
  }

  public void addConstraint(Object newValue) throws ConstraintTypeException {
    Constraint constraint = null;
    if (newValue instanceof Constraint) {
      List newCs = new ArrayList(getConstraints());
      //Debug.noteln("UIR: addConstraint old:", UIUtil.show(constraints));
      newCs.add((Constraint)newValue);
      //Debug.noteln(" new:", UIUtil.show(newCs);
      setConstraints(newCs); //setConstraints sorts and fires
      //Debug.noteln(" new sorted:", UIUtil.show(constraints));
    }
    else if (newValue instanceof Ordering) 
      addOrdering((Constraint)newValue);
    else {
       String message = "Cannot recognise this as a constraint: " + '\n' +
	newValue.toString();
      throw new ConstraintTypeException(message);
    }
    /*
    Name cType = constraint.getType();
    if (cType.equals(Refinement.S_ORDERING)) addOrdering(constraint);
    else if (cType == Refinement.S_EFFECT) addEffect(constraint);
    else if (cType == Refinement.S_CONDITION) addCondition(constraint);
    else if (cType == ) add(constraint);
    else {
      String m = "addConstraint found constraint with non-valid type" + cType;
      throw new ConstraintTypeException(m);
    }
    */
  }

  public void addOrdering(Constraint newValue) {
    List old = new ArrayList(orderings);
    orderings.add(newValue);
    fireDataChange("orderings", old, orderings);
  }
  public void addOrdering(Ordering newValue) {
    List old = new ArrayList(orderings);
    orderings.add(newValue);
    fireDataChange("orderings", old, orderings);
  }


  /**
   * Gets the refinement's ordering type (none, parallel, sequence, other).
   * @return <ul>
   * <li> none (undecided) if there are less than two nodes
   * <li> parallel if there are no orderings an 2 or more nodes
   * <li> sequence if there are 2 or more nodes and they are in sequence
   * <li> other otherwise.
   * </ul>
   */
  public String getOrderingType() {
    if ((nodes == null) || (nodes.size() < 2)) return O_NONE;
    else if ((orderings == null) || (orderings.isEmpty())) return O_PARALLEL;
    else if (orderingsAreSequential()) return O_SEQUENCE;
    else return O_OTHER;
  }
  public String getOrderingType(Collection orderings) {
    if ((nodes == null) || (nodes.size() < 2)) return O_NONE;
    else if ((orderings == null) || (orderings.isEmpty())) return O_PARALLEL;
    else if (orderingsAreSequential(orderings)) return O_SEQUENCE;
    else return O_OTHER;
  }
  /**
   * Sets the refinement's ordering type (none, parallel, sequence, other)
   * if it is allowed to.
   */
  public void setOrderingType(String type) {
    //Debug.noteln("UIR: setting orderings to", type);
    String current = getOrderingType();
    if (O_OTHER.equals(current)) {
      //cannot change "other" like this
      if (!current.equals(type)) 
	Debug.warn("I-DE: Cannot change complex orderings" +
		    " in the minimal view.");
      return; 
    }
    else if (O_NONE.equals(type) || (O_PARALLEL.equals(type))) {
	clearOrderings();
    }
    else if (O_SEQUENCE.equals(type)) {
      setOrderings(makeSequence());
    }
    else Debug.noteln("UIR: cannot recognise simple ordering:", type);
  }

  /** 
   * Checks whether the orderings are a sequence of the nodes.
   */
  public boolean orderingsAreSequential() {
    HashSet sequence = new HashSet(makeSequence());
    return new HashSet(orderings).equals(sequence);
  }
  public boolean orderingsAreSequential(Collection orderings) {
    HashSet sequence = new HashSet(makeSequence());
    return new HashSet(orderings).equals(sequence);
  }

  public List makeSequence() {
    //Debug.noteln("UIR - makeSequenceNodes");
    // make all current sub-nodes sequential

    List nodes = this.getNodes();
    Iterator i = nodes.iterator();
    NodeSpec previous = null;
    NodeSpec current = null;
    List orderings = new ArrayList();
    while (i.hasNext()) {
      current = (NodeSpec)i.next();
      if (previous != null) {
	Ordering refRel = 
	  new Ordering(new NodeEndRef(End.END ,previous.getId()), 
		       new NodeEndRef(End.BEGIN, current.getId()));
	orderings.add(refRel);
	//Debug.noteln(" Orderings are", orderings);
      }
      previous = current;
    }
    return orderings;
  }

  public class NodeTypeException extends Exception {
    public NodeTypeException() { super();}
    public NodeTypeException(String message) { super(message);}
  }
  public class ConstraintTypeException extends Exception {
    public ConstraintTypeException() { super();}
    public ConstraintTypeException(String message) { super(message);}
  }

  /**
   * Check whether the given object is a reference to the given node.
   * checks equals on node, node's toString, node's pattern
   * for NodeSpec refs checks node/ref's id, node/ref's pattern
   * for NodeEndRef checks isRef on ref's node
   */
  public static boolean isRef(NodeSpec node, Object o) {
    //Debug.noteln("Comparing " + node.toString() + " to " + o.toString());
    //Debug.noteln(" i.e. " + node.getId().toString() + " to " + o.toString());
    if ((node.equals(o)) 
	|| (node.getId().toString().equals(o.toString()))
	|| (node.getPattern().toString().equals(o.toString()))) return true;
    if ((o instanceof NodeSpec)
	&& (node.getId().equals(((NodeSpec)o).getId()))
	&& (node.getPattern().equals(((NodeSpec)o).getPattern()))) return true;
    //Debug.noteln("false");
    if ((o instanceof NodeEndRef)
	&& isRef(node, ((NodeEndRef)o).getNode())) return true;
    return false;
  }

  public String print() {
    String names = name + " " + pattern.toString();
    String ns = IVUtil.printNodes(nodes);
    String os = IVUtil.printOrderings(orderings);
    List varDecs = getVariableDeclarations();
    int vdn = 0;
    if (varDecs != null) vdn = varDecs.size();
    String vd = IVUtil.printVarDecs(varDecs);
    return "Refinement: " + names + "\n" +
      "  Nodes: " + ns +  "\n" +
      "  Orderings: " + os +  "\n" +
      "  " + vdn + " Variable Declarations: " + vd 
      //+  "\n" + " DatChngListeners: " + UIUtil.show(dataChangeListeners)
      ;
  }


  //-----------------------Event things----------------------------------------

  HashSet dataChangeListeners = new HashSet();

  public void addDataChangeListener(DataChangeListener l) {
    dataChangeListeners.add(l);
  }
  public void removeDataChangeListener(DataChangeListener l) {
    dataChangeListeners.remove(l);
  }

  private void fireDataChange(String field, Object oldValue, Object newValue) {
    //Debug.noteln("UIR: changing value for " + field + " from", oldValue);
    //Debug.noteln(" to", newValue);
    if (sameValue(field, oldValue, newValue)) return; //no change
    if (field.equals("nodes")) updateOrderings();
    if (!initing) //unless i am initialising myself
      uiDomain.updateConstruct(this);
      //uiDomain.ensureConstruct(this); //make sure domain knows about this!
    for (Iterator i = dataChangeListeners.iterator(); i.hasNext(); ) {
      DataChangeListener l = (DataChangeListener)i.next();
      l.dataChanged(this, field, oldValue, newValue);
    }
  }



  //-----------------------Graph things----------------------------------------

  public void setGraph(IGraph g) { 
    IGraph old = graph;
    graph = g;
    fireDataChange("graph", old, graph);
  }
  public IGraph getGraph() { return graph; }
  public void removeGraph(IGraph g) { setGraph(null); }

  public void setAutoPoints(List points) {
    autoPoints = points;
  }


  //-----------------------TreeNode things-------------------------------------

  private static final int NOFIND = -1;
  public Enumeration children() { 
    //nodes = this.getNodes();
    return new Enumeration() {
      /*
      int i = 0;
      public boolean hasMoreElements() { return i < nodes.size(); }
      public Object nextElement() { return nodes.get(i++); }
      */
      int i = nodes.size()-1;
      public boolean hasMoreElements() { return i >= 0; }
      public Object nextElement() { return nodes.get(i--); }
    };
  }
  public boolean getAllowsChildren() { return true; }
  public TreeNode getChildAt(int index) { 
    return null;//dummy
  }
  public int getChildCount() { 
      if (getNodes() == null) return 0;
      else return this.getNodes().size(); 
  }
  public int getIndex(TreeNode node) {
    return NOFIND;//dummy
  }
  public TreeNode getParent() { return null; }//one level deep only
  public boolean isLeaf() { return (getChildCount() == 0);}

}

// Issues:
// * Ordering needs to sub-class Constraint
// * Think about undoing/redoing loadFromDomain!

