/****************************************************************************
 * An editor for specifying variable declarations
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Jun 22 13:32:08 2006
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.ip2.Ip2ModelManager;

/****************************************************************************
 * An editor for specifying variable declarations
 *
 * Declarations are attached to self (the refinement). <p>
 *
 * The editor is implemented as a frame using a JTextArea and JRadioButtons.<p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose variable declarations
 * are to be specified. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class SimpleDeclarationEditor extends ConstraintEditor
{
  //relation items
  

  //protected JLabel varLabel; 
  protected JTextArea varList; 
  protected JRadioButton used;
  protected JRadioButton any;
  protected JRadioButton given;
  protected JRadioButton none;
  protected JButton add;
  protected JButton delete;
  private ButtonGroup group;
  
  /**
   * Creates an editor for specifying variable declarations.
   * The editor has components for listing declarations and a button group
   * for setting declarations to
   * - any variables (no constraint),
   * - currently used variables (i.e. freeze vars as used),
   * - none (no variables allowed)
   * - as listed
   */
  public SimpleDeclarationEditor(Frame owner){
    super(owner, "Simple Variable Declaration Editor", 
	  "Please specify the variable declarations");
    }
    
  /**
   * Creates an editor for specifying variable declarations within the given
   * refinement.
   * As SimpleDeclarationEditor() but with a given refinement.
   * @param refinement the refinement whose variable declarations are to be
   * specified
   */
  public SimpleDeclarationEditor(Frame owner, UIRefinement refinement) {
    this(owner);
    setObject(refinement);
  }
  
  
  /**
   * Makes all items on the relations panel.
   * These are a type field, a relations (sub-type) field, and a statement
   * field.
   */
  protected Component makeRels() {
    //make a panel that can take all the bits
    JPanel panel = new JPanel(new GridBagLayout());
    
    //make the type and relations field
    JLabel varLabel = new JLabel("Variable declarations");
    varList = new JTextArea();
    JScrollPane jsp = new JScrollPane(varList);
    used = new JRadioButton("Currently Used");
    any = new JRadioButton("Any");
    none = new JRadioButton("None");
    given = new JRadioButton("As Given Here");
    group = new ButtonGroup();
    group.add(used);
    group.add(any);
    group.add(none);
    group.add(given);
    any.setActionCommand("Any");
    any.setToolTipText("Allow any variables");
    none.setActionCommand("None");
    none.setToolTipText("Allow no variables");
    used.setActionCommand("Currently Used");
    used.setToolTipText("Allow only currently used variables");
    given.setActionCommand("As Given Here");
    given.setToolTipText("Allow only given variables");

    any.addMouseListener(this);
    used.addMouseListener(this);
    none.addMouseListener(this);
    given.addMouseListener(this);
    any.setSelected(true);

    add = new JButton("Add");
    delete = new JButton("Delete");

    add.addMouseListener(this);
    delete.addMouseListener(this);

    //add it all to the panel
    GridBagConstraints c;
    c = new GridBagConstraints(0, 0, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,12),0,0); //insets and padding
    panel.add(varLabel, c);
    c = new GridBagConstraints(0, 1 , //x,y
			       1, GridBagConstraints.REMAINDER, //width height
			       1.0,1.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.BOTH,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(jsp, c);
    c = new GridBagConstraints(1, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(any, c);
    c = new GridBagConstraints(1, 2, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(used, c);
    c = new GridBagConstraints(1, 3, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(given, c);
    c = new GridBagConstraints(1, 4, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(none, c);

    /*
    c = new GridBagConstraints(2, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(add, c);
    c = new GridBagConstraints(2, 2, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(delete, c);
    */
    return panel;
  }


  /**
   * Sets the refinement whose variables are to be edited
   * Fills the current declarations (if any) into the list and selects
   * the appropriate radio button.
   * Perhaps this should really be done in setFromObject(varDecs), but we
   * only ever edit the refinement's variable declarations rather than adding
   * a declaration one at a time, so we do it here.
   */
  protected void initFromObject() throws NullPointerException {
    UIRefinement refinement = (UIRefinement) currentObject;
    if (refinement == null) return;    
    Vector vars;
    List decs = refinement.getVariableDeclarations();
    if (decs != null) vars = new Vector(decs);
    else vars = new Vector();
    setListData(vars);
    if (decs == null) {
      Debug.noteln("DecE: got no declarations");
      any.setSelected(true);
    }
    else if (vars.size() == 0) {
      //Debug.noteln("DecE: no vars allowed");
      none.setSelected(true);
    }
    else {
      //we don't care about the order as long as all elements are there
      boolean match = true;
      SortedSet usedVars = refinement.getVariablesUsed();
      Iterator i = decs.iterator();
      while (match && i.hasNext()) {
	Object o = i.next();
	if (o instanceof VariableDeclaration) {
	  if (!usedVars.contains(((VariableDeclaration)o).getName()))
	    match = false;
	}
	else if (!usedVars.contains(o)) match = false;
      }
      if (match) {
	//Debug.noteln("DecE: currently used vars allowed");
	used.setSelected(true);
      }
      else {
	//Debug.noteln("DecE: vars allowed are", UIUtil.listToDisplay(vars));
	given.setSelected(true);
      }
    }
  }

  /**
   * Handles lists of ItemVar or VariableDeclaration (one for variablesUsed,
   * the other for initialisation from previously declared variables)
   */
  public void setListData(Collection data) {
    //Debug.noteln("DecE: setting list data", UIUtil.listToDisplay(data));
    varList.setText("");
    String text = "";
    for (Iterator i = data.iterator(); i.hasNext(); ) {
      Object next = i.next();
      if (next instanceof VariableDeclaration) 
	next = ((VariableDeclaration)next).getName();
      //VariableDeclaration dec = (VariableDeclaration)i.next();
      ItemVar var = (ItemVar)next;
      if (text == "") 
	text = text + var.toString();
      else text = text + UIUtil.lineSeparator + var.toString();
    }
    varList.setText(text);    
  }

  /**
   * We do editing of all variable declarations from initFromObject so this is
   * obsolete.
   */
  protected void setFromObject(Object original) {
  }

  private void setUsed() {
    UIRefinement refinement = (UIRefinement) currentObject;
    Set vars = refinement.getVariablesUsed();
    if ((vars == null) || (vars.size() == 0))
      none.setSelected(true);
    else {
      setListData(vars);
      used.setSelected(true);
    }
  }

  /**
   * Reads the given declaration and lets interested parties know.
   */
  protected Object collectConstraint() {
    //Debug.noteln("DecEd: collecting constraint");
    String text = varList.getText();
    List vars = new ArrayList();
    if (text != "") { //vars given, so use these
      while (!text.equals("")) {
	// String[] parts =
	//    Util.breakStringAtFirst(expansion, lineSeparator);
	String[] parts = UIUtil.breakStringAtFirst(text, "\n");
	String varName = IVUtil.ensureVarName(parts[0].trim());
	if (varName != "") {
	  Symbol var = Symbol.intern(varName);
	  if (var != null) vars.add(new VariableDeclaration((ItemVar)var));
	  //Debug.noteln("got var", var);
	}
	text = parts[1];
      }
    }
    if (vars.size() > 0) {
      varDecs = vars;
      return new ArrayList(varDecs); //constraint different obj from varDecs
    }
    else {//none in list, i.e. allow any or none, depending on selection
      ButtonModel bm = group.getSelection();
      //Debug.noteln("DecE: selection is", bm.getActionCommand());
      if (bm.getActionCommand() != null) {
	if ((bm.getActionCommand().equals("None")) ||
	    (bm.getActionCommand().equals("Given"))||
	    (bm.getActionCommand().equals("As Given Here"))) {
	  varDecs = new ArrayList();
	  return new ArrayList(); //constraint different object from vardecs
	}
      }
    }
    //this cannot return null because the ObjectEditor takes that as cancel
    varDecs = null;
    return "any"; //default: allow any definition
  }

  /**
   * Note any variable declarations set in this editor.
   */
  protected void handleVariables() {
    //if (varDecs == null) Debug.noteln("DecEd: got null varDecs");
    //else Debug.noteln("CE: got varDecs", varDecs);     

    try {
      UIRefinement uiRef = (UIRefinement)currentObject;
      if (varDecs != null) {
	uiRef.setVariableDeclarations(new ArrayList(varDecs));
	varDecs.clear();
      }
      else uiRef.setVariableDeclarations(null);
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }

  //Value already noted under handleVariables()
  protected Object noteNewValue(Object constraint) {
    //Debug.noteln("DecEd: constraint is ", constraint);
    try {
      //lookup what was noted (varDecs have been cleared!)
      UIRefinement uiRef = (UIRefinement)currentObject;
      List decs = uiRef.getVariableDeclarations();
      //Debug.noteln("DecEd: noteValue returning", decs);
      return decs;
    }
    catch (Exception e) {
      Debug.noteException(e);
      return constraint;
    }
  }
  

  public void mouseClicked(MouseEvent e){
    if ((e.getSource() == this.any) || (e.getSource() == this.none)) {
      varList.setText("");
    }
    else if (e.getSource() == this.used) {
      setUsed();
    }
    else if (e.getSource() == this.given) {
    }
    else super.mouseClicked(e);
  }

  public void start() {
    moreBut.setVisible(false);
    moreBut.setEnabled(false);
    super.start();
  }

}


/** Changes
 * 25/10/04: updated variable handling problem (overwriting ConstraintEditor
 *           behaviour)
 **/
