/****************************************************************************
 * A panel containing form-style sub-panels for viewing/editing an object
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Mon Dec 11 11:38:05 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import ix.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.iface.ui.util.*;
import ix.iface.ui.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iview.util.*;
import ix.iview.domain.*;

import java.util.*;
import java.lang.reflect.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;       

/**
 * A panel containing form-style sub-panels for viewing/editing an object
 *
 */
public class ActionFormPanel extends IFormPanel 
  implements ConstructEditing, UndoEditing {

  AConstructFramePanel parent;
  private UIObject uiConstruct;

  ActionFormPanel(AConstructFramePanel parent) {
    super();
    this.parent = parent;
  }

  //-----------------------IFormPanel things-------------------------------

  public void displayFieldData(String field) {
    if (field == null) return;
    //in simple editor, orderings are shown as miniConstraints
    if (field.equals("orderings") && 
	getModel().isFieldShowing("miniConstraints")) {
      displayFieldData("miniConstraints");
    }
    else {
      super.displayFieldData(field);
      if (field.equals("constraints")) {
	//Debug.noteln("ActFP: got constraint to display");
	displayFieldData("orderings");
	displayFieldData("conditions");
      }
    }
  }      

  protected void setRenderers() {
    IXRenderer r = new IXLabelRenderer(false) { //IXTextRenderer
      public void setInfo(Object object){
	if (object == null) setText("");
	else setText(IVUtil.printOrdering((Ordering)object));
      }
    };
    fieldRenderers.put("orderings", r);

    r = new IXLabelRenderer(false) {//IXTextRenderer
      public void setInfo(Object object){
	if (object == null) setText("");
	else setText(IVUtil.printCondition((Constraint)object));
      }
    };
    fieldRenderers.put("conditions", r);

    r = new IXLabelRenderer(false) {//IXTextRenderer
      public void setInfo(Object object){
	if (object == null) setText("");
	else setText(IVUtil.printConstraint((Constraint)object));
      }
    };
    fieldRenderers.put("constraints", r);
 }
 

  /**
   * Render values for: 
   * pattern (list to display),
   * text nodes (write a line for each node)
   */
  protected Object renderValue(String field, Object value) {
    //Debug.noteln("Doing field with name " + field);
    if (value == null) return null;
    else if (field.equals("pattern")) {
      //Debug.noteln("making pattern value");
      return UIUtil.listToDisplay((LList) value);
    }
    else if (field.equals("nodes") && isSimpleField(field)) {
      //Debug.noteln("making simple nodes value");
      //Collection sorted = IVUtil.sortNodeCollection((List) value);
      NodeSpec spec;
      LList childPattern;
      String text = "";
      for (Iterator i = ((List)value).iterator(); i.hasNext(); ) {
	spec = (NodeSpec)i.next();
	childPattern = spec.getPattern();
	text = 
	  text + Lisp.elementsToString(childPattern) + UIUtil.lineSeparator;
      }  
      return text;
    }
    else if (field.equals("issues") && isSimpleField(field)) {
      //Debug.noteln("making simple issues value");
      //Collection sorted = IVUtil.sortNodeCollection((List) value);
      Issue issue;
      LList pattern;
      String text = "";
      for (Iterator i = ((List)value).iterator(); i.hasNext(); ) {
	issue = (Issue)i.next();
	pattern = issue.getPattern();
	text = text + Lisp.elementsToString(pattern) + UIUtil.lineSeparator;
      }  
      return text;
    }
    else return super.renderValue(field, value);
  }
  

  /**
   * De-render values for: 
   * pattern (lisp.elementsFromString),
   * nodes (parse if text, otherwise ignore because updates are immediate)
   * orderings (return als LList)
   */
  protected Object deRenderValue(String field, Object data) {
    //Debug.noteln("AFP: de-rendering field with name " + field);
    //Debug.noteln("AFP: Data is of class " + data.getClass().getName());
    if (field.equals("pattern")) {
      if (data == null) return Lisp.NIL;
      //Debug.noteln("making pattern value from data");
      return Lisp.elementsFromString((String)data);
    }
    else if (field.equals("nodes")) {
      if (data == null) return Lisp.NIL;
      if (data.getClass() == String.class) {
	//Debug.noteln("AFP: Parsing nodes from data: " + data);
	return UIRefinement.parseNodes(getUIDomain(), (String)data);
      }
      else {
	//nodes are noted as we go along, so look them up and forget data!
	//Debug.noteln(" AFP: getting nodes from refinement");
	UIRefinement uir = (UIRefinement)model.getObject();
	if (uir != null) 
	  return uir.getNodes();
	else return Lisp.NIL;
      }
    }
    else if (field.equals("issues")) {
      if (data == null) return Lisp.NIL;
      if (data.getClass() == String.class) {
	//Debug.noteln("AFP: Parsing issues from data: " + data);
	return UIRefinement.parseIssues(getUIDomain(), (String)data);
      }
      else {
	//issues are noted as we go along, so look them up and forget data!
	//Debug.noteln(" AFP: getting issues from refinement");
	UIRefinement uir = (UIRefinement)model.getObject();
	if (uir != null) 
	  return uir.getIssues();
	else return Lisp.NIL;
      }
    }
    else if (field.equals("orderings") || field.equals("conditions")
	     || field.equals("constraints")) {
      if (data == null) return Lisp.NIL;
      java.util.List values = Arrays.asList((Object[])data);
      LListCollector constraints = new LListCollector(values);
      return constraints;
    }
    else return super.deRenderValue(field, data);
  }

  /**
   * make a special display bit for miniConstraints
   * proper patterns
   * enable variable checking in pattern and text versions of nodes and issues
   */
  protected IXEditorPanel makeDisplayBit(final String field) {
    //Debug.noteln("AFP: Making display bit for " + field);
    if (field.equals("miniConstraints")) {
      //Debug.noteln("AFP: miniConstraint");
      //String[] buttons = {"Comprehensive"};
      String[] buttons = {};
      return new MiniConstraintPanel(this, "Orderings", buttons);
      //return new MiniConstraintPanel(this, "Orderings");
    }
    //patterns are lists, edited as strings
    else if (field.equals("pattern")) {
      //Debug.noteln("AFP: pattern");
      JTextField jtf = new JTextField();
      jtf.addFocusListener(new FocusAdapter() {
	  public void focusLost(FocusEvent fEvent) {
	    saveFieldData(field);
	  };
	});
      IVUtil.enableVars(parent, jtf);
      return new ThingEditorPanel(this, model.getFieldName(field), jtf);
    }
    else {
      IXEditorPanel bit = super.makeDisplayBit(field);
      JComponent ic = bit.getItemComponent();
      if ((field.equals("nodes") || field.equals("issues")) &&
	  JTextComponent.class.isInstance(ic)) {
	IVUtil.enableVars(parent, (JTextComponent)ic);
      }
      return bit;
    }
  }

  //-----------------------Action panel things-------------------------------

  /**
   * @return the current setting of the sequence flag
   */
  public boolean isSequenceSet() { 
    Object data = getFieldData("miniConstraints");
    //Debug.noteln("SEP: Mini constraings orderings are " + data);
    return (UIRefinement.O_SEQUENCE.equals(data));
  }

  private String getNameString() {
    Object data = getFieldData("name");
    if (data == null) return "";
    else return data.toString().trim();
  }
  private LList getPatternLList() {
    Object data = getFieldData("pattern");
    return Lisp.elementsFromString((String)data);
  }


  //-----------------------UndoEditing things-----------------------
  // reading/updating the model will have the desired effect

  /** Sets the given field to the given value in the editor. */
  public void undoSetValue(String field, Object value) {
    Debug.noteln("AFP: got undo set for field "+field+" val", value);
    getModel().setValue(field, value);
  }

  /** Gets the given field to the given value in the editor. */
  public Object undoGetValue(String field) {
    Debug.noteln("AFP: got undo get for field", field);
    return getModel().getValue(field);
  }


  //-----------------------ConstraintEditing things-----------------------


   /** Gets the current UIDomain. */
  public UIDomain getUIDomain() { 
    return parent.getUIDomain();
  }
  /** Gets the current construct. */
  public IXObject getConstruct() { 
    return uiConstruct.getBaseObject(); 
  }
  /** Sets the current construct. */
  public void setConstruct(IXObject construct) {
    setUIConstruct(getUIDomain().getUIObject(construct));
  }
  /** Gets the current UI construct. */
  public UIObject getUIConstruct() { 
    return uiConstruct; 
  }

  public void setUIConstruct(UIObject uiObject) {
    //if (!(uiObject == null) && (uiObject.equals(uiConstruct))) return;
    UIObject old = uiConstruct;
    uiConstruct = uiObject;
    getModel().setObject(uiObject);
    parent.fireConstructChanged(old, uiObject, this);
  }

  public void setUIConstructOnly(UIObject uiObject) {
    //if (!(uiObject == null) && (uiObject.equals(uiConstruct))) return;
    UIObject old = uiConstruct;
    uiConstruct = uiObject;
    getModel().setObject(uiObject);
    ((ActionEditorPanel)parent).updateTreeSelection();
    parent.fireConstructChanged(old, uiObject, this);
  }

  public void loadFromObject() {
    displayModelData();
  }

  public void saveToObject(EditableObject object) {
    if (isSequenceSet()) {
      UIRefinement newUIR = (UIRefinement)object;
      newUIR.setOrderings(newUIR.makeSequence());
    }
    super.saveToObject(object);
  }
    
  public void saveToObject() {
    saveToObject(model.getObject());
  }
    
}
