/****************************************************************************
 * Support for editing sets of named objects (e.g. refinements or
 * object classes)
 *
 * File: AbstractUISetEditor.java
 * @author Jussi Stader
 * @version 4.3
 * Updated: Mon Jan 29 10:54:15 2007
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import java.util.*;
import java.lang.reflect.Field;
import java.awt.Component;
import java.awt.Cursor;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import java.io.*;

import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.icore.domain.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.util.*;
import ix.util.match.*;
import ix.util.lisp.*;

/**
 * Support for editing sets of named objects (e.g. refinements or
 * object classes).  Looks after adding/removing/editing the objects
 * and keeping the set consistent. Handles overwrites, finding objects
 * in the set, working out whether the set has changed.
 */

public abstract class AbstractUISetEditor extends AbstractSetEditor 
{ 
  public AbstractUISetEditor(DomainObjectManager manager, LinkedList objects) {
    super((ObjectManager)manager, objects);
  }

  public void setManager(DomainObjectManager manager, LinkedList objects) {
    super.setManager(manager, objects);
  }

  public void setObjects(LinkedList objects) {
    //get all relevant objects in their order and note them in nodes
    //Debug.noteln("UISetEd: setting Objects", this);
    clear();
    members.clear();
    if (objects == null) return;
    for (Iterator i = objects.iterator(); i.hasNext(); ) {
      Named r = (Named)i.next();
      members.put(r.getName(), r); 
    }
    //Debug.noteln(" Objects now", members);
  }

  public Domain getDomain() {
    return ((DomainObjectManager)manager).getDomain();
  }
  public UIDomain getUIDomain() {
    return ((DomainObjectManager)manager).getUIDomain();
  }

  public Object getOriginal(Object uiOb) {
    if (uiOb instanceof UIObject) 
      return ((UIObject)uiOb).getOriginal();
    else return super.getOriginal(uiOb);
  }
  public void setOriginal(Object own, Object original) {
    super.setOriginal(own, original); //note in originals
    try {
      UIObject uiOb = (UIObject)own;
      uiOb.setBaseReference(getName(original));
      uiOb.setBaseObject((IXObject)cloneConstruct(original));
    }
    catch (Exception e) {Debug.noteException(e);}
  }


  public void rememberClear() {
    clear();
    members.clear();
    //Debug.noteln("UISetEd: remembering clear");
    removed.addAll(getOriginals());
  }


  //-------------------------abstracts---------------------------------------


  public String getName(Object object) {
    if (object instanceof Named) return ((Named)object).getName();
    else return "";
  }
  public void setName(Object object, String name) {
    if (object instanceof EditableObject) 
      ((EditableObject)object).setValue("name", name);
  }


  /**
   * Finds an object whose name matches the given string.
   * @return a UIObject, an IXObject, or null - depending on which exists. 
   * Checks whether the named object is current, but will return stale objects.
   */
  public Named getAnyNamedObject(String name) {
    if (name == "") return null;
    Named object = (Named)findOwnObject(name);
    if (object != null) return object;
    else {//find an original that has not changed
      //Debug.noteln("AUISetEd: looking in originals for",name);
      object = getStringOriginal(name);
      if (!isCurrentNamedObject(object)) //!name.equals(object.getName())
	object = null;
      //Debug.noteln(" found", object);
      return object;
    }
  }
  /**
   * Finds an object whose name matches the given string.
   * @return a UIObject, an IXObject, or null - depending on which exists. 
   * Checks whether the named object is current.
   */
  public Named getNamedObject(String name) {
    if (name == "") return null;
    Named object = (Named)findOwnObject(name);
    if (object != null) return object;
    else {//find an original that has not changed
      //Debug.noteln("AUISetEd: looking in originals for",name);
      //(new Throwable()).printStackTrace();
      object = getStringOriginal(name);
      if (!isCurrentNamedObject(object) || !name.equals(object.getName()))
	object = null;
      //Debug.noteln(" found", object);
      return object;
    }
  }

  /* gets the named object from the IXDomain. 
   * BEWARE: this will return old objects whose names have been edited! Use 
   * isCurrentNamedObject to check.
   */
  public abstract Named getStringOriginal(String name);

  public void loadOriginal(Object own) {
    super.loadOriginal(own);
    ((UIObject)own).setBaseReference(getName(own));
  }

  /** Checks whether the given object has an original base object */
  public boolean hasOriginal(Named object) {
    if (object instanceof UIObject) return ((UIObject)object).hasOriginal();
    else return false;
  }
  /** Checks whether the given object is empty */
  public boolean isEmpty(Object object) {
    if (object instanceof UIObject) return ((UIObject)object).isEmpty();
    else return false;
  }

  public boolean isUndefined(Object object) {
    if (object instanceof UIObject) return ((UIObject)object).isUndefined();
    else return true;
  }

  public IXObject getBaseObject(Named object) {
    if (object instanceof UIObject) return ((UIObject)object).getBaseObject();
    else return null;
  }

  public Object cloneConstruct(Object o) throws CloneNotSupportedException {
    if (o instanceof Refinement) return ((Refinement)o).clone();
    else if (o instanceof ObjectClass) return ((ObjectClass)o).clone();
    else return null;
  }

  //--------------------------Handling changes--------------------------------
  

  public void saveToDomain(Domain domain) {
      String exception = "";
    try {
      for (Iterator i=removed.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: removing object", o.toString());
	try { //if removing does not work, just skip it!
	  if ((o != null) && (o instanceof UIObject))
	    ((UIObject)o).removeFromDomain(domain);
	  else removeFromDomain(domain, o);
	}
	catch (Exception e) {
	  Debug.noteException(e);
	  //exception = Debug.foldException(e);
	  //JOptionPane.showMessageDialog(null, Debug.foldException(e));
	}
      }
      for (Iterator i=added.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: adding object", o.toString());
	if ((o instanceof UIObject) && (!((UIObject)o).isEmpty()))
	  ((UIObject)o).addToDomain(domain);
      }
      for (Iterator i=edited.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: updating object", o.toString());
	if ((o != null) && (o instanceof UIObject))
	  ((UIObject)o).updateInDomain(domain);
      }
      //Debug.noteln("UID - finished saveToDomain(domain)");
    }
    catch (DomainReferenceException re) {
      Debug.noteln(Debug.foldException(re));
      exception = Debug.foldException(re);
      //JOptionPane.showMessageDialog(null, Debug.foldException(re));
      //Debug.describeException(re);
    }
    if (exception != "") 
      JOptionPane.showMessageDialog(null, "Last exception saving to domain" 
				    +" (there may be more): "+exception);
  }


  public void removeFromDomain(Domain domain, Object object) {
    if (object == null) return;
    if (object instanceof Refinement) 
      domain.deleteNamedRefinement(((Refinement)object).getName());
    else if (object instanceof ObjectClass) {
      domain.deleteNamedObjectClass(((ObjectClass)object).getName());
    }
  }


  /** 
   * Collects descriptions of any changes in constructs in the set.
   */
  public List collectConstructChanges() {
    List changes = new ArrayList();
    if (hasChangedConstructs()) {
      changes.add("Changes in " + label);
      if (!added.isEmpty()) 
	changes.add(" Added: " + IVUtil.namedListToDisplay(added));
      if (!removed.isEmpty()) 
	changes.add(" Removed: " + IVUtil.namedListToDisplay(removed));
      if (!edited.isEmpty()) 
	changes.add(" Edited: " + IVUtil.namedListToDisplay(edited));
    }
    return changes;
  }

  //--------------------------Notifying listeners-----------------------------


  public void fireConstructRemoved(Object object) {
    //Debug.noteln("UISetEd firing construct removed");
    if (object instanceof UIObject)
      manager.fireConstructRemoved(this, (UIObject)object);
    else
      Debug.noteln("UISE: Deleted object without UI version. Not firing.",
		   object);
  }

}


// Issues:
// * tidy up load/insert domain stuff 
// * 
// * 
