/****************************************************************************
 * A class for useful static methods for interfaces
 *
 * @author: Jussi Stader <J.Stader@ed.ac.uk>
 * @version: 4.0+
 * Updated: Mon Mar 26 13:15:16 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui.util;

import ix.iface.ui.*;
import ix.util.*;
import ix.iface.util.*;
import ix.util.lisp.*;
import java.lang.reflect.*;
import java.util.*;
import java.io.*;
import java.net.URL;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.Container;
import java.awt.Component;
import java.awt.event.*;


/** Class for useful static methods. */

public class UIUtil {

  public static final String lineSeparator = 
       System.getProperty("line.separator");

  /**
   * Brings up a message that the given string item is not yet supported.
   * e.g. "Editing AROs is not yet supported"
   */
  public static void notImplemented(Component parent, String item){
    JOptionPane.showMessageDialog(parent, 
				  item + " is not yet supported.");
  }
  /**
   * Brings up a message that the given string item is not yet supported.
   * e.g. "Editing AROs is not yet supported"
   */
  public static void warning(Component parent, String item){
    JOptionPane.showMessageDialog(parent, "Warning: " + item);
  }



  public static void addListener(JComponent component, EventListener el) {
      boolean added = false;
      if (el instanceof ActionListener) {
	  added = addActionListener(component, (ActionListener)el);
      }
      if (!added && (el instanceof MouseListener)) {
	  component.addMouseListener((MouseListener)el);
      }
  }
  public static boolean addActionListener(JComponent c, ActionListener al) {
      boolean added = false;
      if (c instanceof AbstractButton) {
	  ((AbstractButton)c).addActionListener(al);
	  return true;
      }
      if (c instanceof JTextField) {
	  ((JTextField)c).addActionListener(al);
	  return true;
      }
      if (c instanceof JComboBox) {
	  ((JComboBox)c).addActionListener(al);
	  return true;
      }
      return false;
  }

  public static IXEditorPanel getEventPanel(ActionEvent ae) {
    Object component = ae.getSource();
    //Debug.noteln(" event source is", component);
    if ((component != null) && (component instanceof Component)) {
      Container butPan = ((Component)component).getParent();
      if (butPan != null) {
	Container parent = butPan.getParent();
	//Debug.noteln(" parent container is", parent);
	if ((parent != null) && (parent instanceof IXEditorPanel))
	  return (IXEditorPanel)parent;
      }
    }
    return null;
  }
  public static AbstractButton getEventButton(ActionEvent ae) {
    Object component = ae.getSource();
    if ((component != null) && (component instanceof AbstractButton))
      return (AbstractButton)component;
    else return null;
  }


  //---------------------------Menu things

  /**
   * Fills the given menu with the names of the given objects, attaching the
   * given command.
   * Always clears the menu first and disables it. Only enables it if there are
   * items in it.
   * Sets the label to the given command if the menu has no label.
   * @param al the ActionListener that will process menu selections
   * @param menu the menu to be filled
   * @param command attached to each new menu item
   * @param objects an ArrayList of named objects to be put in the menu
   */
  public static void populateMenu(ActionListener al, JMenu menu, 
				  String command, List objects) {
    String label = menu.getText();
    if ((label == null) || (label.equals(""))) 
      populateMenu(al, menu, command, command, objects);
    else populateMenu(al, menu, label, command, objects);
  }

  /**
   * Fills the given menu with the names of the given objects, attaching the
   * given command.
   * Always clears the menu first and disables it. Only enables it if there are
   * items in it.
   * 
   * @param menu the menu to be filled
   * @param command attached to each new menu item
   * @param label sets the label of the menu unless it is null
   *   (leaves label alone if it is null)
   * @param objects an ArrayList of named objects to be put in the menu
   */
  public static void populateMenu(ActionListener al, JMenu menu, String label,
				  String command, List objects) {
    boolean enabled = menu.isEnabled();
    menu.setEnabled(false);
    menu.removeAll();
    if ((objects != null) && (!objects.isEmpty())) {
      int max = 20;
      int remaining = objects.size();
      int count = 0;
      JMenu shortMenu = menu;
      for (Iterator i = objects.iterator(); i.hasNext();) {
	if (count == max && remaining > 5) {
	  JMenu submenu = new JMenu("More");
	  shortMenu.add(submenu);
	  shortMenu = submenu;
	  count = 0;
	}
	ix.icore.domain.Named o = (ix.icore.domain.Named)i.next();
	makeMenuItem(al, shortMenu, command, o.getName());
	remaining--;
	count++;
      }
      if ((label != null) && (label != "")) menu.setText(label);
      //menu.setEnabled(enabled);
      menu.setEnabled((count > 0));
    }
  }

  /**
   * Fills the given menu with the names of the given objects, attaching the
   * given command.
   * Always clears the menu first and disables it. Only enables it if there are
   * items in it.
   * 
   * @param menu the menu to be filled
   * @param command attached to each new menu item
   * @param objects an ArrayList of named objects to be put in the menu
   */
  public static void populateMenu(ActionListener al, JObjectMenu menu, 
				  Object obj, String command, List objects) {
    menu.setObject(obj);
    populateMenu(al, menu, command, objects);
  }

  /**
   * Makes a menu item with the given label and command and adds it to the
   * given (menu) parent. 
   *
   * @param al the action listener that is interested in the item's selection
   * @param menu the menu that the item should be added to
   * @param command attached to the item for recognising actions
   * @param label displayed in the menu
   */
  public static JMenuItem makeMenuItem(ActionListener al, JComponent menu, 
				       String command, String label){
    JMenuItem item = new javax.swing.JMenuItem(label);	 
    item.setActionCommand(command);
    item.addActionListener(new CatchingActionListener(al));
    menu.add(item);
    return item;
  }

  /**
  public static void clearMouseListeners(Component c) {
    MouseListener[] mls = c.getMouseListeners();
    for (int i = 0; i < mls.length; i++) c.removeMouseListener(mls[i]);
  }
  **/

  //---------------------------String things

  public static String toCapitalised(String inString) {
    char upFirst = Character.toUpperCase(inString.charAt(0));
    return upFirst + inString.substring(1);
  }
  public static String toUncapitalised(String inString) {
    char upFirst = Character.toLowerCase(inString.charAt(0));
    return upFirst + inString.substring(1);
  }

  public static class ToStringComparator implements Comparator {
    public int compare(Object o1, Object o2) {
      try {
	String s1 = o1.toString();
	String s2 = o2.toString();
	return s1.compareTo(s2);
      }
      catch (Exception cce) {
	Debug.noteException(cce);
	return 0;
      }
    }

    //public boolean equals(Object o) {
    //  return super.equals(o);
    //}
  }

    /**
     * Makes sure that the given string is enclosed in parentheses.
     * @return the given string if it is already in parentheses, 
     * the given string with added ( and ) if not.
     */
  public static String ensureParenthesized(String text) {
    if (text.startsWith("(") && text.endsWith(")"))
      return text;
    else
      return "(" + text + ")";
  }


    /**
     * breakStringAtFirst takes a string containing fields separated by
     * a (string) delimiter and returns a two-element string array containing
     * the substring before the first occurrence of the char, and the
     * substring after.  Neither substring contains the delimiter.  If
     * the delimiter does not appear in the string at all, the values
     * are the string and "".
     */
    public static String[] breakStringAtFirst(String s, String separator) {
	int i = s.indexOf(separator);
	if (i == -1)
	    return new String[]{s, ""};
	else
	    return new String[]{
		s.substring(0, i),
	        s.substring(i + separator.length())
	    };
    }


  //---------------------------Property things

  public static Symbol propertyToSymbol(String sValue) {
    return Symbol.intern(sValue);
  }
  public static int propertyToInt(String sValue) {
    return Integer.parseInt(sValue);
  }
  public static boolean propertyToBoolean(String sValue) {
    return new Boolean(sValue).booleanValue();
  }
  public static String booleanToProperty(boolean value) {
    return new Boolean(value).toString();
  }
  public static String symbolToProperty(Symbol value) {
    if (value == null) return "";
    return value.toString();
  }
  public static String intToProperty(int value) {
    return Integer.toString(value);
  }

  //---------------------------Comparing things

  /** Returns true if the given thing is null, empty string, 
      or empty collection, or map with no values **/
  public static boolean isEmptyThing(Object thing) {
    if (thing == null) return true;
    if (thing instanceof String) return (thing.equals(""));
    if (thing instanceof Collection) return (0 == ((Collection)thing).size());
    if (thing instanceof Map) {
      //Debug.noteln("UIU: map values", ((Map)thing).values());
      return (((Map)thing).isEmpty() || isEmptyThing(((Map)thing).values()));
    }
    if (thing.getClass().isArray()) return (0 == ((Object[])thing).length);
    return false;
  }


  //---------------------------Field things
  public static Field stringToField(Class oClass, String field) throws 
                              NoSuchFieldException, IllegalAccessException {
    if (oClass == null) return null;
    else return oClass.getField(field);
  }



  /**
   * Looks for a field that sounds like a good name field and returns its
   * index in the given list.
   */
  public static int findNameField(ArrayList fieldNames) {
    int index = fieldNames.indexOf("name");
    if (index != -1) return index;
    else {
      index = fieldNames.indexOf("Name");
      if (index != -1) return index;
      else {
	String fName;
	for (int i = 0; i<fieldNames.size(); i++) {
	  fName = toCapitalised((String)fieldNames.get(i));
	  if (fName.indexOf("NAME") != -1)
	    return i;
	}
      }
    }
    return -1;  //no useful field found, so use the first one.
  }


  public static Object getObjectFieldValue(Object object, String field) {
    if (object == null) {
      Debug.noteln("Util: Cannot get value for a null object.");
      return null;
    }
    else {
      try {
	Field oField = stringToField(object.getClass(), field);
	return oField.get(object);
      }
      catch (Exception e) {
	Debug.noteln("Cannot get field " + field + " Trying get<Field>");
	String mName = "get" + toCapitalised(field);
	try {
	  Class classes[] = {};
	  Object args[] = {};
	  Method method = object.getClass().getMethod(mName, classes);
	  //return method.invoke(object, args);
	  return method.invoke(object, args);
	}
	catch (Exception me) {
	  Debug.noteException(me);
	  return null;
	}  
      }
    }
  }

  /**
   * Sets the object's field to the given value 
   */
  public static void setObjectFieldValue(Object object, String field,
					 Class fieldClass, Object value) {
    if (object == null) {
      Debug.noteln("Util: Cannot set value for a null object.");
      return;
    }
    if (field == null) {
      Debug.noteln("Util: Cannot set value for a null field.");
      return;
    }
    else {
      try {
	Field oField = stringToField(object.getClass(), field);
	oField.set(object, value);
      }
      catch (Exception e) {
	Debug.noteln("Cannot set field " + field + " Trying setField method");
	String mName = "set" + toCapitalised(field);
	try {
	  Class valClass;
	  //if (value == null) valClass = Object.class;
	  //else valClass = value.getClass();
	  Class[] classes = {fieldClass};
	  Method method = object.getClass().getMethod(mName, classes);
	  if (value == null) value = fieldClass.newInstance();
	  else Debug.noteln("Value is  ", value.toString());
	  Object[] args = {value};
	  method.invoke(object, args);
	}
	catch (Exception me) {
	  Debug.noteln("setObjectFieldValue: problem with field " + field);
	  Debug.note(" method " + mName);
	  Debug.noteln(", value class" + value.getClass().getName());
	  Debug.noteException(me);
	  return;
	}  
      }
    }
  }


  /**
   * Checks whether the given thing is empty. Checks for null, empty string, 
   * and empty list. If in doubt, say it is not empty.
   */
  public static boolean isEmptyValue(Object value) {
    if (value == null) return true;
    if (value instanceof String) {
      if ((String)value == "") return true;
      else return false;
    }
    if (value instanceof Collection) return ((Collection)value).isEmpty();
    else return false;
  }



  //---------------------------List things

  public static String show(Collection l) {
    String text = "";
    if (l == null) return text;
    Iterator i = l.iterator();
    while (i.hasNext()) {
      String s;
      Object next = i.next();
      if (next == null) s = "null";
      else s = next.toString();
      text = text + s + " ";
    }
    return text.trim();
  }
  public static String showClass(Collection l) {
    String text = "";
    if (l == null) return text;
    Iterator i = l.iterator();
    while (i.hasNext()) {
      String s;
      Object next = i.next();
      if (next == null) s = "null";
      else s = next.getClass().toString();
      if (s == null) s = "null";
      text = text + s + " ";
    }
    return text.trim();
  }
  public static String show(Enumeration e) {
    String text = "";
    if (e == null) return text;
    while (e.hasMoreElements()) {
      Object next = e.nextElement();
      text = text + next.toString() + " ";
    }
    return text.trim();
  }
  public static String listToDisplay(Collection l) {
    return show(l);
  }
  public static String listToText(Collection l) {
    String text = "";
    if (l == null) return text;
    Iterator i = l.iterator();
    while (i.hasNext()) {
      String nextText = "";
      Object next = i.next();
      if (Collection.class.isInstance(next)) 
	nextText = show((Collection)next);
      else nextText = next.toString();
      text = text + nextText + lineSeparator;
    }
    return text.trim();

  }
  public static String[] listToTextArray(Collection l) {
    if (l == null) return new String[0];
    String[] text = new String[l.size()];
    Iterator i = l.iterator();
    int index = 0;
    while (i.hasNext()) {
      String nextText = "";
      Object next = i.next();
      if (Collection.class.isInstance(next)) 
	nextText = show((Collection)next);
      else nextText = next.toString();
      text[index++] = nextText.trim();
    }
    return text;

  }
  public static String fieldsToNames(Field[] fields) {
    String text = "";
    for (int i=0; i<fields.length; i++) 
      text = text + fields[i].getName() + " ";
    return text;
  }


  public static LList removeDuplicates(LList list) {
    if ((list == null) || (list.isEmpty()) || (list.isNull())) return list;
    else {
      Object o = list.car();
      LList withoutCdr = remove(o, list.cdr());
      LList newCdr = removeDuplicates(withoutCdr);
      return new Cons(o, newCdr);
    }
  }

  //****This should not be needed, usually class problem! use showClass
  public static Collection without(Collection things, Collection notThings) {
    if ((things == null) || things.isEmpty() || 
	(notThings == null) || notThings.isEmpty())
      return things;
    //else
    Collection newThings;
    try {
      newThings = (Collection)things.getClass().newInstance();
    }
    catch (Exception e) {
      Debug.noteln("UIUtil without: cannot create class", things.getClass());
      Debug.noteln(" using ArrayList or HashSet instead");
      if (things instanceof Set) newThings = new HashSet();
      else newThings = new ArrayList();
    }

    for (Iterator i = things.iterator(); i.hasNext(); ) {
      Object nextThing = i.next();
      //if (!notThings.contains(nextThing)) newThings.add(nextThing);
      if (nextThing != null) {
	boolean found = false;
	for (Iterator in = notThings.iterator(); 
	     in.hasNext() && !found; ) {
	  Object nextNot = in.next();
	  //Debug.noteln("Checking", nextThing + " against " + nextNot);
	  //Debug.noteln(nextThing.getClass() + " - ", nextNot.getClass());
	  if (nextThing.equals(nextNot)) found = true;
	}
	if (!found) newThings.add(nextThing);
	//else Debug.noteln(" UIUtil: found", nextThing);
      }
    }
    return newThings;
  }

  public static LList remove(Object o, LList list) {
    if ((list == null) || (list.isEmpty()) || (list.isNull())) return list;
    else {
      Object car = list.car();
      if (o.equals(car)) return remove(o, list.cdr());
      else return new Cons(car, remove(o, list.cdr()));
    }
  }

  public static List getFlatValues(Map map) {
    List valueList = new ArrayList();
    if (map != null) {
      LinkedList values = new LinkedList(map.values());
      if (values != null) { //values is a list of lists; flatten it
        for (Iterator i = values.iterator(); i.hasNext(); ) {
	  Object next = i.next();
	  if (next instanceof Collection)
	    valueList.addAll((Collection)next);
	  else valueList.add(next);
	}
      }
    }
    return valueList;
  }


  public static List enumerationToList(Enumeration e) {
    List list = new LinkedList();
    if (e == null) return list;
    while (e.hasMoreElements()) {
      Object next = e.nextElement();
      list.add(next);
    }
    //Debug.noteln("UIUtil: got list from enumeration");
    return list;
  }
  //Make a list of the given class, then add all given elements
  public static TypedList newTypedList(Class listClass, Collection data) {
    try {
      Class impClass = 
	ListOf.findImplementationClass(listClass, LinkedList.class);
      TypedList tList = (TypedList)impClass.newInstance();
      tList.addAll(data);
      return tList;
    }
    catch (Exception e) { //java.lang.InstantiationException
      Debug.noteException(e);
      return null;
    }
  }
  public static TypedList newTypedList(Class listClass, Object[] data) {
    return newTypedList(listClass, Arrays.asList(data));
  }
  public static TypedList newTypedList(Class listClass, Object data) {
    if (data.getClass().isArray()) 
      return newTypedList(listClass, Arrays.asList((Object[])data));
    else {
	List l = new LinkedList();
	l.add(data);
      return newTypedList(listClass, l);
    }
  }




  //--------------------------- URL things

    public static URL resourceURL(String filename) {
        String fullname = "resources/html/" + filename;
        URL result = UIUtil.class.getClassLoader().getResource(fullname);
        //Debug.noteln("Making resource URL", result);
        return result;
	/*
	String base = System.getProperty("ix.base", ".");
	File resources = new File(base, "resources");
	File html = new File(resources, "html");
	File urlLocation = new File(html, filename);
	Debug.noteln("Making URL from", urlLocation);
	try {
	    return urlLocation.toURL();
	}
	catch (java.net.MalformedURLException e) {
	    Debug.noteException(e);
	    throw new RuntimeException("Can't handle URL filename "
				       + filename);
	}
	*/
    }

  //--------------------------- JComboBox hack things

  public static void listenToCB(JComboBox jcb, PopupMenuListener listener) {
    // This nasty-lloking code uses reflection to call a method, which
    // only exists in JDK 1.4. This gets around a compilation error, when
    // compiling under earlier JDKs
    try {
      jcb.getClass().getMethod("addPopupMenuListener", 
			       new Class[] {PopupMenuListener.class})
        .invoke(jcb, new Object[] {listener});
    } catch (Exception e) {}  
  }


  //--------------------------- things



}
