/****************************************************************************
 * An IXTree that can be edited.
 *
 * @author Jussi Stader
 * @version 2.0
 * Updated: Tue Mar 13 12:00:35 2007
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui.tree;

import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.awt.Point;
import java.awt.datatransfer.*;
import java.awt.dnd.*;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.domain.*;

//***Implement drag and drop!
/****************************************************************************
 * An IXTree that can be edited. Mostly relays to model, except
 * showing and selecting nodes.
 *
 *****************************************************************************
 */
public class EditableTree extends IXTree implements ClipboardOwner {
	
  Clipboard clips = new Clipboard("Tree Clipboard");
  TreePath path = null;
  boolean isModified = false;
  HashMap nodeMap;
    
  /**
   * Creates a new tree which is editable.
   */
  public EditableTree() {
    super();
  }
  /**
   * Creates a new instance with the given model
   */
  public EditableTree(EditableTreeModel model) {
    super(model);
  }

  public void setSelectedObject(Object userObject) {
    clearSelection();
    IXTreeNode tn = ((EditableTreeModel)getModel()).findTreeNode(userObject);
      if (tn != null)
	setSelectionPath(new TreePath(tn.getPath()));
  }


  //-------------showing and selecting nodes---------------------
  //*** should this lot be in IXTree?

  /**
   * Scroll and open the tree to make the given node visible.
   */
  public void showUserNode(Object userNode) {
    TreePath oldPath = getSelectionPath();
    if (oldPath != null) { //old selection any use?
      Object node = oldPath.getLastPathComponent();
      if ((node != null) && (node instanceof IXTreeNode))
	showUserNode(userNode, ((IXTreeNode)node).getUserObject());
      else showUserNode(userNode, null);
    }
  }
  /**
   * Scroll and open the tree to make the given node visible.
   * The old node may have been selected before, so if it is an
   * ancestor, show the version of userNode that descends from it.
   */
  public void showUserNode(Object userNode, Object oldUserNode) {
    try {
      EditableTreeModel model = (EditableTreeModel) getModel();
      if (model.isUserRoot(userNode)) {
	//Debug.noteln("EdTM: got user root");
	showRoot();
	return;
      }
      Collection treeNodes = model.findTreeNodes(userNode);
      if (treeNodes != null) 
	while (treeNodes.contains(null)) treeNodes.remove(null);
      if ((treeNodes == null) || (treeNodes.size() == 0)) {
	showRoot();
	return;
      }
      Collection oldNodes = model.findTreeNodes(oldUserNode);
      if (oldNodes != null) 
	while (oldNodes.contains(null)) oldNodes.remove(null);

      //Debug.noteln("EdT: got tree nodes", treeNodes);
      IXTreeNode showNode = null;

      if ((treeNodes.size() != 1) && //only one choice, so treat simply
	  (oldNodes != null) && (oldNodes.size() > 0)) {//no others; simple
	for (Iterator i = treeNodes.iterator(); 
	     i.hasNext() && (showNode == null); ) {
	  try {
	    DefaultMutableTreeNode treeNode = (DefaultMutableTreeNode)i.next();
	    if (treeNode != null) {
	      for (Iterator j = oldNodes.iterator();  
		   j.hasNext() && (showNode == null); ) {
		if (treeNode.isNodeAncestor((TreeNode)j.next())) 
		  showNode = (IXTreeNode)treeNode;
	      }
	    }
	  }
	  catch (Exception eLoop) {}
	}
      }
      if (showNode == null) {//simple or nothing found
	try {showNode = (IXTreeNode)treeNodes.iterator().next();}
	catch (Exception eLoop2) {}
	//Debug.noteln("EdT: got simple node", showNode);
      }
      if (showNode == null)  //this may happen after delete
	showRoot();
      else showPath(showNode.getPath());
    }

    catch (Exception e) {
      Debug.noteException(e);
    }
  }

  protected void showRoot() {
    try {
      EditableTreeModel model = (EditableTreeModel) getModel();
      if (model.getRoot() != null) {
	showPath(((IXTreeNode)model.getRoot()).getPath());
      }
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }
  private void showPath(TreeNode[] nodes) {
    showPath(new TreePath(nodes));
  }
  private void showPath(TreePath path) {
    //Debug.noteln("EdTM: showing path", path);
    scrollPathToVisible(path);
    setSelectionPath(path);

  }


  //------------------ Relays to model ---------------------------

  /** Makes a node for this tree from the given user object*/
  public IXTreeNode newTreeNode(Object userObject) {
    EditableTreeModel model = (EditableTreeModel)getModel();
    return model.newIXTreeNode(userObject);
  }

  /** Makes a user object with the given string as its name*/
  public Object makeUserObject(String name){
    EditableTreeModel model = (EditableTreeModel)getModel();
    return model.makeUserObject(name);
  }  
  /** Makes a user object as above with the given userObjects as children*/
  public Object makeUserObject(String name, List userChildren){
    EditableTreeModel model = (EditableTreeModel)getModel();
    return model.makeUserObject(name, userChildren);
  }




  
  //------------------ Clipboard Owner stuff---------------------------

  public void lostOwnership(Clipboard clipboard, Transferable contents) {
    Debug.noteln("Tree lost ownership of clipboard", this);
  }
  

}
