/****************************************************************************
 * A panel with a type-in box, open-button, and a scrollable JEditorPane for
 * HTML.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Thu Sep 21 16:23:37 2006 by Jeff Dalton
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.iface.ui.util.*;

// Imports added by JD
import ix.ip2.Ip2;
import ix.icore.IXAgent;
import ix.icore.domain.Domain;
import ix.icore.plan.Plan;
import ix.iface.util.IXHtmlEditorKit;
import ix.iface.util.IFUtil;
import ix.util.xml.*;

import java.util.*;
import java.awt.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.io.IOException;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A panel with a type-in box, open-button, and a scrollable JEditorPane for
 * HTML.<p>
 *
 * Example code for using the HTMLPanel:
 *<PRE><code>
 *    ...
 *    frame.getContentPane().setLayout(new BorderLayout(0,0));
 *    HTMLPanel htmlPanel = new HTMLPanel(fileName);
 *    getContentPane().add(BorderLayout.CENTER, htmlPanel);
 *    ...
 *</code></PRE>
 *
 * For a real example @see ix.iface.ui.HelpFrame
 */
public class HTMLPanel extends JPanel implements HyperlinkListener 
{
  
  private final JEditorPane htmlPane = new JEditorPane();
  private final JTextField urlField = new JTextField("          ");
  private URL currentURL;

  private ArrayList history = new ArrayList();
  private int currentHistory = 0;


  /**
   * Puts up a panel with a text-field and a button at the top and a 
   * scrollable text area underneath. The text area can display HTML 
   * (including images), follow links when clicked, and select text,
   * but NOT edit. The text-field is for typing in a URL reference, the button
   * (or return) will open the page into the text area.
   */
  public HTMLPanel(){
    super(new BorderLayout());
    setSize(550,650);

    final Action open = new AbstractAction("Open URL") {
      public void actionPerformed(ActionEvent e) {
	String text = urlField.getText();
	try {
	  displayURL(new URL(text));
	}
	catch (MalformedURLException ue) {
	  String message = text + " is not a URL this panel can display.";
	  JOptionPane.showMessageDialog(HTMLPanel.this, message);
	}
      }
    };


    InputMap iMap = urlField.getInputMap(); //default: WHEN_FOCUSED
    ActionMap aMap = urlField.getActionMap();
    aMap.put("open", open);
    iMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0), "open");

    //text field for URL and load button
    final JButton urlLoadBut = new JButton(open);

    
    final JPanel refPanel = new JPanel(new GridBagLayout());

    //final JPanel refPanel = new ThingEditorPanel(ml, "URL", urlField, buts);
	
    GridBagConstraints gbc =
      new GridBagConstraints(0,0,  1,1, //x,y, w,h
			     3.0,0.0,  //weight x,y
			     GridBagConstraints.WEST, //anchor
			     GridBagConstraints.HORIZONTAL,  //fill
			     new Insets(0,4,0,0),0,0);
    refPanel.add(urlField, gbc);
    gbc = new GridBagConstraints(1,0,  1,1, //x,y, w,h
				 0.0,0.0,  //weight x,y
				 GridBagConstraints.WEST, //anchor
				 GridBagConstraints.NONE,  //fill
				 new Insets(0,2,0,0),0,0);
    refPanel.add(urlLoadBut, gbc);

    this.add(BorderLayout.NORTH, refPanel);

    //html page in a scroller
    htmlPane.setEditable(false);
    htmlPane.setEditorKitForContentType("text/html", new IXHtmlEditorKit());
    htmlPane.addHyperlinkListener(this);
    htmlPane.addMouseListener(new LinkMouseListener());
    final JScrollPane htmlView = new JScrollPane(htmlPane);
    this.add(BorderLayout.CENTER, htmlView);

  }

  /**
   * As above but the given URL is displayed and remembered as the original
   * page.
   */
  public HTMLPanel(URL url){
    this();	
    displayURL(url);
  }
  /**
   * As above but the given file name is converted to a URL using the html
   * area of the resources. 
   * An example call is <code> HTMLPanel("ide-help.html") <\code>
   */
  public HTMLPanel(String fileName){
    this();	
    try {
      URL url = UIUtil.resourceURL(fileName);
      displayURL(url);
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }

  public URL currentURL() {
    return currentURL;
  }

  private boolean setURL(URL url) {
    urlField.setText(url.toString());
    try {
      htmlPane.setPage(url);
      currentURL = url;
      return true;
    } 
    catch (IOException e) {
      UIUtil.warning(this, Debug.foldException(e));
      /*
      if ((url != null) && (!url.getProtocol().equals("http"))) 
	UIUtil.warning(this, "Cannot deal with non-http links in URL: " + 
		       url.toString());
      else
	UIUtil.warning(this, "Attempted to read a bad URL: " + url.toString());
      */
      return false;
    }      
  }
  	

  /**
   * Displays the given URL in the text area. 
   * Keeps track of previous url for "back" buttons.
   * Will remove forward stack.
   * Also shows a wait cursor while it is displaying.
   */
  public boolean displayURL(URL url) {
    boolean result = false;
    Debug.noteln("Displaying url", url.toString());
    Cursor cursor = getCursor();
    //Debug.noteln("HTMLP: setting wait cursor");
    setCursor(new Cursor(Cursor.WAIT_CURSOR));
    if (setURL(url)) {
      result = true;
      while (currentHistory > 0) { //remove forward stack
	currentHistory = currentHistory-1;
	history.remove(currentHistory);
      }
      history.add(0, url);
    }
    //Debug.noteln(" setting cursor back");
    setCursor(cursor);  
    return result;
  }

  public boolean back() {
    try {
      currentHistory = currentHistory + 1;
      if (setURL((URL)history.get(currentHistory)))
	return true;
      else {
	currentHistory = currentHistory - 1;
	return false;
      }
    }
    catch (Exception e) {
      try { //something went wrong, try setting it to previous page
	setURL((URL)history.get(currentHistory));
	return false;
      }
      catch (Exception e2) { //still wrong, setting to previous page
	setURL(currentURL);
	int i = history.indexOf(currentURL);
	if (0 <= i) currentHistory = i;
	return false;	
      }
    }
  }
  public boolean forward() {
    try {
      currentHistory = currentHistory - 1;
      if (setURL((URL)history.get(currentHistory)))
	return true;
      else {
	currentHistory = currentHistory + 1;
	return false;
      }
    }
    catch (Exception e) {
      try { //something went wrong, try setting it to previous page
	setURL((URL)history.get(currentHistory));
	return false;
      }
      catch (Exception e2) { //still wrong, setting to previous page
	setURL(currentURL);
	int i = history.indexOf(currentURL);
	if (0 <= i) currentHistory = i;
	return false;	
      }
    }
  }

  /**
   * Gets called when a link is clicked in the current page. Use this to 
   * implement following links.
   * The events seem to be of type "ACTIVATED". Other types are ENTERED and 
   * EXITED but I don't know what they are!
   */
  public void hyperlinkUpdate(HyperlinkEvent e) {
    //Debug.noteln("HTMLPanel got hyperlink update", e.getDescription());
    //Debug.noteln(" type", e.getEventType().toString());
    //only do things when clicked (not interested in enter/exit events)
    if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
      URL current = htmlPane.getPage();
      if (loadFromURL(e.getURL()))
	  return;
      if (!displayURL(e.getURL())) {
	Debug.noteln("The bad link's description is", e.getDescription());
	displayURL(current);
      }
    }
  }

    protected boolean loadFromURL(URL url) {
	IXAgent agent = IXAgent.getAgent();
	if (agent == null || !(agent instanceof Ip2))
	    return false;
	Ip2 ip2 = (Ip2)agent;
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	String type = fsm.getType(url.getFile());
	boolean maybePlan = fsm.getInputTypesForClass(Plan.class)
	                          .contains(type);
	boolean maybeDomain = fsm.getInputTypesForClass(Domain.class)
	                            .contains(type);
	if (!(maybePlan || maybeDomain))
	    return false;
	JPopupMenu popup = new LinkPopupMenu(ip2, url, maybePlan, maybeDomain);
	if (mostRecentMouseEvent == null)
	    return false;
	MouseEvent e = mostRecentMouseEvent;
	popup.show(e.getComponent(), e.getX(), e.getY());
	return true;
    }

    MouseEvent mostRecentMouseEvent;

    class LinkMouseListener extends MouseAdapter {
	public void mouseClicked(MouseEvent e) {
	    mostRecentMouseEvent = e;
	}
    }

    class LinkPopupMenu extends JPopupMenu implements ActionListener {

	Ip2 ip2;
	URL url;

	LinkPopupMenu(Ip2 ip2, URL url,
		      boolean maybePlan, boolean maybeDomain) {
	    this.ip2 = ip2;
	    this.url = url;
	    if (maybePlan)
		add(IFUtil.makeMenuItem("Load Plan", this));
	    if (maybeDomain)
		add(IFUtil.makeMenuItem("Load Domain", this));
	    add(IFUtil.makeMenuItem("View Page", this));
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Link popup command", command);
	    if (command.equals("Load Plan"))
		loadPlan();
	    else if (command.equals("Load Domain"))
		loadDomain();
	    else if (command.equals("View Page")) {
                URL current = htmlPane.getPage();
                if (!displayURL(url))
                    displayURL(current);
            }                    
            else
		throw new ConsistencyException
		    ("Noting to do for " + command);
	}

	void loadPlan() {
	    ip2.loadPlan(url.toString());
	}

	void loadDomain() {
	    ip2.readDomain(ip2.getDomain(), url.toString());
	}

    }

}
/*
 * Document all public methods
 */
