/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jan 20 19:07:40 2010 by Jeff Dalton
 * Copyright: (c) 2009, AIAI, University of Edinburgh
 */

package ix.test;

import org.jdom.*;
import org.jdom.filter.*;

import java.net.*;
import java.io.*;
import java.text.*;
import java.util.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.http.*;

/**
 * Tests HTTP "GET" requests that obtain XML.
 */
public class TwitterAtomSearchTest {

    String SEARCH_URL = "http://search.twitter.com/search.atom";

    String DEFAULT_QUERY = "#openvce";

    int MAX_RESULTS = 10;

    int CONNECT_TIMEOUT = 90 * 1000 ; // milliseconds

    int READ_TIMEOUT = 90 * 1000 ; // milliseconds

    Namespace ATOM_NAMESPACE =
        Namespace.getNamespace("http://www.w3.org/2005/Atom");

    Namespace TWITTER_NAMESPACE =
        Namespace.getNamespace("twitter", "http://api.twitter.com/");

    String dateTimePattern = "yyyy'-'MM'-'dd'_'HH'-'mm'-'ss";

    SimpleDateFormat format = new SimpleDateFormat(dateTimePattern);
    {
	format.setTimeZone(TimeZone.getTimeZone("GMT"));
	format.setLenient(false);
    }

    public TwitterAtomSearchTest() {
    }

    public static void main(String[] argv) throws IOException {

        new TwitterAtomSearchTest().do_main(argv);

    }

    void do_main(String[] argv) {

        Debug.off();
        String query = getEncodedQueryArgs();
        Map<String,List<String>> args = decodeQueryArgs(query);
        Document doc;
        try {
            String q = args.get("q").get(0);
            if (q.trim().equals("")) {
                System.out.println("No search query was given.");
                return;
            }
            doc = readResultXML(q);
        }
        catch (Exception e) {
            writeErrorReport(e);
            return;
        }

        Element root = doc.getRootElement();

        // Now we have the document and want to process the 'entry' elements.

        Filter entryFilter = new ElementFilter("entry");

        int count = 0;
        for (Iterator i = root.getContent(entryFilter).iterator()
                 ; i.hasNext() && ++count <= MAX_RESULTS; ) {
            Element entry = (Element)i.next();
            System.out.println(getAuthorName(entry) + ": " +
                               getContentText(entry));
        }

    }

    String getAuthorName(Element entry) {
        Element author = entry.getChild("author", ATOM_NAMESPACE);
        Element name = author.getChild("name", ATOM_NAMESPACE);
        return name.getText();
    }


    String getContentText(Element entry) {
        Element content = entry.getChild("content", ATOM_NAMESPACE);
        return stripMarkup(content.getTextNormalize());
    }

    String stripMarkup(String text) {
        StringBuilder result = new StringBuilder();
        int len = text.length();
        boolean inTag = false;
        for (int i = 0; i < len; i++) {
            char c = text.charAt(i);
            if (inTag) {
                if (c == '>')
                    inTag = false;
            }
            else {
                if (c == '<')
                    inTag = true;
                else
                    result.append(c);
            }
        }
        return result.toString();
    }

    String getEncodedQueryArgs() {
        String query = System.getenv("QUERY_STRING");
        if (query != null && !query.equals(""))
            return query;
        try {
            return "q=" + URLEncoder.encode(DEFAULT_QUERY, "utf-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RethrownIOException(e);
        }
    }

    Map<String,List<String>> decodeQueryArgs(String args) {
        // /\/ Doesn't get actually decode.
        Map<String,List<String>> result =
            new LinkedHashMap<String,List<String>>();
        for (String arg: Strings.breakAt("&", args)) {
            String[] parts = Strings.breakAtFirst("=", arg);
            String var = parts[0];
            String val = parts[1];
            List<String> already = result.get(var);
            if (already == null) {
                already = new LinkedList<String>();
                result.put(var,already);
            }
            already.add(val);
        }
        return result;
    }

    Document readResultXML(String q) throws Exception {

        // See whether there's a sufficiently recent cached result.
        File cached = new File("cache/query_" + q);
        if (cached.exists()) {
            long lastMod = cached.lastModified();
            Debug.noteln("Cached date", new Date(lastMod));
            if (new Date().getTime() - lastMod < 2 * 60 * 1000) {
                Debug.noteln("Reading cached result");
                return XML.parseXML(cached);
            }
            Debug.noteln("Cached result is too old");
        }

        // Make a new search request.
        SearchClient http = new SearchClient();

        String url = SEARCH_URL + "?q=" + q;

        String result = http.sendGetRequest(new URL(url));

        // Cache the result.
        File tmp = File.createTempFile("new_" + q, ".tmp",
                                       new File("cache/"));

        Writer w = new FileWriter(tmp);
        w.write(result);
        w.flush();
        w.close();
        
        if (cached.exists())
            cached.delete();

        tmp.renameTo(cached);

        // Return the result as a Document.
        return XML.parseXML(result);
    }

    private class SearchClient extends HttpStringClient {
        private SearchClient() {
            setConnectTimeout(CONNECT_TIMEOUT);
            setReadTimeout(READ_TIMEOUT);
        }
        protected boolean handleResponseCode(HttpURLConnection conn)
	      throws HttpRequestException, 
		     IOException {
            int status = conn.getResponseCode();
            String message = conn.getResponseMessage();
            if (status == 503) {
                throw new HttpRequestException
                    (status, message + " Headers: " + conn.getHeaderFields());
            }
            else
                return handleResponseCode(status, message);
        }
    }

    void writeErrorReport(Exception e) {
        Date now = new Date();
        String fileName = "tmp/http-get-error-" + format.format(now) + ".txt";
        try {
            PrintWriter out = new PrintWriter(new FileWriter(fileName));
            e.printStackTrace(out);
            out.close();
        }
        catch (Exception grrr) {
            Debug.noteln("Cannot write error report");
            Debug.noteException(grrr);
        }
    }

}
