/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr 23 14:35:45 2009 by Jeff Dalton
 * Copyright: (c) 2007, 2008, 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc.sl;

import java.net.URL;

import java.util.*;

import org.jdom.*;
import org.jdom.filter.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.http.*;

/**
 * Second Life XML-RPC utilities for sending XML-RPC requests
 * and decoding the replies.
 */
public class SLRpc {

    // sl-xmlrpc-server specifies the URI of the Linden Labs XML-RPC server
    // for Second Life:
    protected String defaultServerURL =
        "http://xmlrpc.secondlife.com/cgi-bin/xmlrpc.cgi";

    protected HttpStringClient http = makeHttpStringClient();

    public SLRpc() {
        defaultServerURL = 
            Parameters.getParameter("sl-xmlrpc-server", defaultServerURL);
    }

    /**
     * Creates a suitable client for use by this instance of SLRpc.
     */
    protected HttpStringClient makeHttpStringClient() {
        HttpStringClient client = new HttpStringClient();
        client.setRequestContentType("application/xml");
        client.setConnectTimeout((int) 1 * 60 * 1000);
        client.setReadTimeout((int) (0.5 * 60 * 1000));
        return client;
    }

    /**
     * Sends an XML-PRC request to the default server URL and
     * returns whatever reply is received.
     */
    public SLRpcMessage sendRequest(String channelId, int i, String s) {
        return sendRequest(XML.toURL(defaultServerURL), channelId, i, s);
    }

    /**
     * Sends an XML-PRC request to the specified server URL and
     * returns whatever reply is received.
     */
    public SLRpcMessage sendRequest(URL serverURL, String channelId,
                                    int i, String s) {
        Debug.noteln("Sending XML-RCP to", serverURL);
        String call = makeCallString(channelId, i, s);
        String reply = http.sendRequest(serverURL, call);
        return decodeResponse(reply);
    }

    /**
     * Constructs the XML string that would be sent to SL.
     */
    public String makeCallString(String channelId, int i, String s) {
	SLRpcMessage m = new SLRpcMessage(channelId, i, s);
	Element call = makeCall(m);
	Document doc = new Document(call);
	return XML.documentToXMLString(doc);
    }

    private Element makeCall(SLRpcMessage m) {
	Element call = new Element("methodCall");
	call.addContent(new Element("methodName").setText("llRemoteData"));
	call.addContent
	    (new Element("params")
	     .addContent(new Element("param")
			 .addContent(new Element("value")
				     .addContent(makeStruct(m)))));
	return call;
    }

    private Element makeStruct(SLRpcMessage m) {
	Element struct = new Element("struct");
	struct.addContent(makeField("Channel", m.getChannel()));
	struct.addContent(makeField("IntValue", m.getIntValue()));
	struct.addContent(makeField("StringValue", m.getStringValue()));
	return struct;
    }

    private Element makeField(String fieldName, Object value) {
	Element field = new Element("member");
	field.addContent(new Element("name").setText(fieldName));
	field.addContent
	    (new Element("value")
	     .addContent(new Element(typeTag(value))
			 .setText(value.toString())));
	return field;
    }

    private String typeTag(Object value) {
	if (value instanceof Integer || value instanceof Long)
	    return "int";
	else if (value instanceof String)
	    return "string";
	else
	    throw new IllegalArgumentException
		("Cannot determine a type tag for " + value);
    }

    /**
     * Decodes an XML response from SL into an SLRpcMessage.
     */
    public SLRpcMessage decodeResponse(String r) {
	Document doc = XML.parseXML(r);
	Element response = doc.getRootElement();
	Element params = response.getChild("params");
	Element param = params.getChild("param");
	Element value = param.getChild("value");
        // When testing, we usually have the struct as a child of the
        // "value" element, but in the real SL response, it's down inside
        // an array.  So we try to fish it out wherever it is.
        ElementFilter filter = new ElementFilter("struct");
        for (Iterator i = value.getDescendants(filter); i.hasNext();) {
            Element struct = (Element)i.next();
            Debug.expectEquals("struct", struct.getName());
            return decodeMessageStruct(struct);
        }
        throw new IllegalArgumentException("Missing struct in " + r);
    }

    private SLRpcMessage decodeMessageStruct(Element struct) {
	String channel = (String)getFieldValue(struct, "Channel", "");
	Integer i = (Integer)getFieldValue(struct, "IntValue", 0);
	String s = (String)getFieldValue(struct, "StringValue", "");
	return new SLRpcMessage(channel, i, s);
    }

    private Object getFieldValue(Element struct, String fieldName,
				 Object defaultValue) {
	for (Iterator i = struct.getChildren().iterator(); i.hasNext();) {
	    Element field = (Element)i.next();
	    Element name = field.getChild("name");
	    if (name.getText().equals(fieldName)) {
		Element value = field.getChild("value");
		Element typed = (Element)value.getChildren().get(0);
		String typeTag = typed.getName();
		String v = typed.getText();
		return unstringValue(typeTag, v);
	    }
	}
	return defaultValue;
    }

    private Object unstringValue(String typeTag, String valueText) {
	if (typeTag.equals("int") || typeTag.equals("i4")) {
	    try {
		return Integer.parseInt(valueText);
	    }
	    catch (NumberFormatException e) {
		throw new RethrownException(e);
	    }
	}
	else if (typeTag.equals("string"))
	    return valueText;
	else
	    throw new IllegalArgumentException
		("Unknown type tag " + typeTag);
    }

    public static void main(String[] argv) {
        Parameters.processCommandLineArguments(argv);

        String to = Parameters.requireParameter("to");
        String channel = Parameters.requireParameter("channel");
        String string = Parameters.requireParameter("string");
        int i = Parameters.getInt("int", 0);

        URL toUrl = XML.toURL(to);

	SLRpc rpc = new SLRpc();

	String call = rpc.makeCallString(channel, i, string);

//      System.out.println(call);
// 	System.out.println(rpc.decodeResponse(call));

        HttpStringClient http = rpc.makeHttpStringClient();
        http.setConnectTimeout((int) 1 * 60 * 1000);
        http.setReadTimeout((int) (0.5 * 60 * 1000));

        String reply = http.sendRequest(toUrl, call);

        System.out.println(reply);
        System.out.println(rpc.decodeResponse(reply));

    }

}
