/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Apr 27 19:23:55 2009 by Jeff Dalton
 * Copyright: (c) 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc.sl;

import java.util.*;

import ix.iserve.ipc.*;

import ix.icore.*;
import ix.icore.domain.*;

import ix.util.PatternParser.LispPatternParser;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/* Syntax, from the SL script:

string PROGRESS_REPORT = "activity progressing";
string COMPLETION_REPORT = "activity completed";
string INFORMATION_REPORT = "activity information";
string STARTED_ACTIVITY = "activity started";
string STOPPED_ACTIVITY = "activity stopped";
string STATE_CONSTRAINT = "state-constraint";

send_constraint(string attribute, string object, string value)
{
    llMessageLinked(LINK_THIS, 1, 
        STATE_CONSTRAINT+" "+attribute+" "+object+" = "+value, NULL_KEY);
}


send_completion_report()
{
    llMessageLinked(LINK_THIS, 1, COMPLETION_REPORT, NULL_KEY);
}


send_progress_report()
{
   llMessageLinked(LINK_THIS, 1, PROGRESS_REPORT, NULL_KEY);
}


send_information_report(string contents)
{
   llMessageLinked(LINK_THIS, 1, INFORMATION_REPORT+" "+contents, NULL_KEY);
}

*/

public class SLReplyDecoder {

    // /\/: Not clear how STARTED_ACTIVITY and STOPPED_ACTIVITY are used.

    String PROGRESS_REPORT = "activity progressing";
    String COMPLETION_REPORT = "activity completed";
    String INFORMATION_REPORT = "activity information";
    String STARTED_ACTIVITY = "activity started";
    String STOPPED_ACTIVITY = "activity stopped";
    String STATE_CONSTRAINT = "state-constraint";

    Symbol Q_ATTRIBUTE = Symbol.intern("?attribute");
    Symbol Q_OBJECT    = Symbol.intern("?object");
    Symbol Q_VALUE     = Symbol.intern("?value");

    LispPatternParser lpp = new LispPatternParser();

    LList CONSTRAINT_PATTERN =
        lpp.parsePattern("state-constraint ?attribute ?object = ?value");

    public SLReplyDecoder() {
    }

    public void fillInReply(MessageWrapper reply,
                            String contents,
                            MessageWrapper message) {
        Sendable ix_contents = decodeContents(contents);
        ix_contents.setSenderId(Name.valueOf(reply.getFrom()));
        reply.setContents(ix_contents);
        if (message == null || message.getFrom() == null) {
            reply.setTo("unknown recipient");
        }
        else {
            reply.setTo(message.getFrom());
            if (ix_contents instanceof Report)
                fillInReport((Report)ix_contents, message);
        }
    }

    Sendable decodeContents(String contents) {
        if (contents.startsWith(STATE_CONSTRAINT))
            return decodeConstraint(contents);
        else
            return decodeReport(contents);
    }

    Constraint decodeConstraint(String contents) {
        MatchEnv env = SimpleMatcher.mustMatch(CONSTRAINT_PATTERN,
                                               lpp.parsePattern(contents));
        Object attribute = env.get(Q_ATTRIBUTE);
        Object object = env.get(Q_OBJECT);
        Object value = env.get(Q_VALUE);
        return new Constraint
            ("world-state",
             "effect",
             Lisp.list(new PatternAssignment
                               (Lisp.list(attribute, object),
                                value)));
    }

    Report decodeReport(String contents) {
        if (contents.equals(PROGRESS_REPORT))
            return new Report(ReportType.PROGRESS, contents);
        else if (contents.equals(COMPLETION_REPORT) ||
                 contents.equals(STARTED_ACTIVITY) ||
                 contents.equals(STOPPED_ACTIVITY))
            return new Report(ReportType.SUCCESS, contents);
        else if (contents.startsWith(INFORMATION_REPORT + " "))
            return new Report(ReportType.INFORMATION,
                              Strings.afterFirst(INFORMATION_REPORT + " ",
                                                 contents));
        else
            return new Report(ReportType.PROGRESS, contents);
    }

    void fillInReport(Report report, MessageWrapper message) {
        Object sentContents = message.getContents();
        if (sentContents != null && sentContents instanceof Activity) {
            Activity act = (Activity)sentContents;
            if (act.getRef() != null)
                report.setRef(act.getRef());
        }
    }

}
