/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jun 10 15:38:44 2009 by Jeff Dalton
 * Copyright: (c) 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc;

import java.util.*;

import ix.iserve.ipc.*;

import ix.icore.*;
import ix.icore.domain.*;

import ix.ichat.ChatMessage;

import ix.util.PatternParser.LispPatternParser;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.XML;

/**
 * A simple string representation of I-X items for use by communication
 * strategies.
 *
 * <p>Syntax out of I-X Java representation:
 * <pre>
 *   Activity:   Aid^from^pattern
 *   Chat:       M^from^text
 * </pre>
 * </p>
 *
 * <p>Syntax into I-X Java representation:
 * <pre>
 *   Activity:   Aid^to^pattern
 *   Constraint: C^to^type subtype pattern=value
 *   Report:     Rrefid^to^type text
 *   Chat:       M^to^text
 * </pre>
 * </p>
 *
 * <p>For messages coming into I-X, 'from' will be attached or determined
 * somehow by the communication strategy, either on the sending or receiving
 * side, and will be installed as the senderId on the receiving side
 * after decoding from linear syntax.</p>
 *
 * <p>The 'refid' will be the id of an activity and will include the
 * initial 'A' for 'activity'.</p>
 *
 * <p>In the "into I-X" syntax for a constraint, the type "world-state"
 * can be abbreviated to "state".</p>
 */
public class LinearIXSyntax {

    private String from = "";

    public LinearIXSyntax(String fromName) {
        this.from = fromName;
    }

    /*** Encoding ***/

    public String encode(Sendable item) {
        if (item instanceof Activity)
            return encode((Activity)item);
        else if (item instanceof ChatMessage)
            return encode((ChatMessage)item);
        else
            throw new UnsupportedOperationException
                ("Can't yet encode " + item);
    }

    public String encode(Activity act) {
        // Do we need to consider report-back? /\/
        LList pattern = (LList)Variable.removeVars(act.getPattern());
        String pat = PatternParser.unparse(pattern);
        act.ensureId();
        return "A" + act.getId() + "^" + senderId(act) + "^" + pat;
    }

    public String encode(ChatMessage m) {
        return "M^" + string(senderId(m)) + "^" + string(m.getText());
    }

    private String string(Object o) {
        return o == null ? "" : o.toString();
    }

    private String senderId(Sendable s) {
        // Not clear it makes sense to call this for an activity
        // in normal use.  /\/
        return s.getSenderId() != null ? s.getSenderId().toString() : from;
    }

    /*** Decoding ***/

    public Sendable decode(String s) {
        if (!s.equals("")) {
            switch (s.charAt(0)) {
            case 'A':
                return decodeActivity(s);
            case 'C':
                return decodeConstraint(s);
            case 'R':
                return decodeReport(s);
            case 'M':
                return decodeChatMessage(s);
            default:
            }
        }
        throw new IllegalArgumentException
            ("Cannot decode " + Strings.quote(s));
    }

    public Activity decodeActivity(String s) {
        String[] fields = breakup(s);
        String id = fields[0];
        String to = fields[1];
        String pat = fields[2];
        LList pattern = PatternParser.parse(pat);
        Activity act = new Activity(pattern);
        setTo(act, to);
        act.setRef(Name.valueOf(id));
        return act;
    }

    public Constraint decodeConstraint(String s) {
        String[] fields = breakup(s);
        String to = fields[1];
        String spec = fields[2];
        String[] c_fields = breakup(spec, " ", 3);
        String type = c_fields[0];
        String subtype = c_fields[1];
        String pv_spec = c_fields[2];
        if (type.equals("state"))
            type = "world-state";
        String[] pv_fields = Strings.breakAtFirst("=", pv_spec);
        String pat = pv_fields[0].trim();
        String val = pv_fields[1].trim();
        if (val.equals(""))
            val = "true";
        LList pattern = PatternParser.parse(pat);
        Object value = Lisp.readFromString(val);
        PatternAssignment pv = new PatternAssignment(pattern, value);
        Constraint c = new Constraint(type, subtype, Lisp.list(pv));
        setTo(c, to);
        return c;
    }

    public Report decodeReport(String s) {
        String[] fields = breakup(s);
        String id = fields[0];
        String to = fields[1];
        String content = fields[2];
        String[] content_fields = breakup(content, " ", 2);
        String type = content_fields[0];
        String text = content_fields[1];
        Report r = new Report();
        setTo(r, to);
        r.setText(text);
        r.setReportType(ReportType.valueOf(type));
        r.setRef(Name.valueOf(id));
        return r;
    }

    public ChatMessage decodeChatMessage(String s) {
        String[] fields = breakup(s);
        String to = fields[1];
        String text = fields[2];
        ChatMessage m = new ChatMessage();
        setTo(m, to);
        m.setText(text);
        return m;
    }

    private String[] breakup(String s) {
        return breakup(s, "^", 3);
    }

    private String[] breakup(String s, String separator, int n_fields) {
        List<String> parts = Strings.breakAt(separator, s);
        if (parts.size() < n_fields)
            throw new IllegalArgumentException
                ("Missing fields in " + Strings.quote(s));
        String[] result = new String[n_fields];
        ListIterator<String> pi = parts.listIterator();
        for (int i = 0; i < n_fields -1; i++) {
            result[i] = pi.next();
            pi.remove();
        }
        result[n_fields -1] = Strings.joinWith(separator, parts);
        return result;
    }

    private String getTo(Sendable s) {
        return (String)((Annotated)s).getAnnotation("to");
    }

    private void setTo(Sendable s, String toName) {
        ((Annotated)s).setAnnotation("to", toName);
    }

    public static void main(String[] argv) {
        LinearIXSyntax syntax = new LinearIXSyntax("AnAgent");
        while (true) {
            String line = Util.askLine("-- ");
            if (line.equals("bye"))
                break;
            try {
                Sendable s = syntax.decode(line);
                System.out.println(XML.objectToXMLString(s));
                System.out.println(syntax.encode(s));
            }
            catch (Exception e) {
                System.out.println(Debug.describeException(e));
            }
        }
    }

}
