/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 13 15:55:10 2008 by Jeff Dalton
 * Copyright: (c) 2005 - 2008, AIAI, University of Edinburgh
 */

package ix.http;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.ipc.ServiceAddress;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A name-server for an {@link HttpCommunicationStrategy}.
 */
public class HttpNameServer {

    public static String DEFAULT_ADDRESS = "localhost:5555";

    private Map<String,ServiceAddress> nameToAddrMap =
	new Hashtable<String,ServiceAddress>();         // synchronized

    private Syntax syntax = new Syntax();

    private LispPrinter lispPrinter = new LispPrinter();
    { lispPrinter.setPrintUnprintablesUsingToString(true); }

    private TextAreaFrame transcriptFrame;

    public HttpNameServer() {
    }

    public static void main(String[] argv) {
	Parameters.processCommandLineArguments(argv);
	new HttpNameServer().start();
    }

    private void start() {

	// Get the host and port we're meant to be.
	ServiceAddress addr =
	    new ServiceAddress
	        (Parameters.getParameter("name-server",
					 HttpNameServer.DEFAULT_ADDRESS));

	// Create a comm strategy that listens at that port.
	NSCommStrategy strategy = new NSCommStrategy(addr.getPort());
	strategy.setupServer("name-server", null);

	transcriptFrame = new TranscriptFrame(addr);

    }

    private class NSCommStrategy extends HttpCommunicationStrategy {

	private NSCommStrategy(int port) {
	    super(port);
	}

	@Override
	protected Object handleInput(Object contents) {
	    return handleRequest(contents);
	}

	@Override
	protected void registerWithNameServer() {
	    // Don't register -- we *are* the name-server.
	}

    }

    private Object handleRequest(Object request) {
	Debug.noteln("NS request:", request);
	if (syntax.isLookupMessage(request)) {
	    String agentName = syntax.getLookupAgentName(request);
	    ServiceAddress addr = nameToAddrMap.get(agentName);
	    if (addr == null)
		return "unknown";
	    else
		return addr;
	}
	else if (syntax.isRegisterMessage(request)) {
	    String agentName = syntax.getRegisterAgentName(request);
	    ServiceAddress addr = syntax.getRegisterAddress(request);
	    nameToAddrMap.put(agentName, addr);
	    transcript(request);
	    return "OK";
	}
	else
	    throw new IllegalArgumentException
		("Invalid name-server request: " + request);
    }

    private class TranscriptFrame extends TextAreaFrame {
	private TranscriptFrame(ServiceAddress addr) {
	    super("I-X HTTP Name-Server at " + addr);
	}
	public void whenClosed() {
	    if (Util.dialogConfirms(this, "Exit name-server?"))
		System.exit(0);
	}
    }

    private void transcript(Object a) {
	transcript(lispPrinter.elementsToString(a));
    }

    private void transcript(final String line) {
	javax.swing.SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		transcriptFrame.appendLine(line);
		if (!transcriptFrame.getFrame().isShowing())
		    transcriptFrame.setVisible(true);
	    }
	});
    }

    /**
     * Name-server message syntax.
     */
    public static class Syntax {

        // Registration

	public static Object makeRegisterMessage(String agentName,
						 ServiceAddress addr) {
	    return Lisp.list("register", agentName, addr);
	}

	public boolean isRegisterMessage(Object m) {
	    return hasForm
		(m, Lisp.list("register", String.class, ServiceAddress.class));
	}

        public String getRegisterAgentName(Object m) {
            return (String)((List)m).get(1);
        }

        public ServiceAddress getRegisterAddress(Object m) {
            return (ServiceAddress)((List)m).get(2);
        }

        // Address lookup

	public static Object makeLookupMessage(String agentName) {
	    return Lisp.list("lookup", agentName);
	}

	public boolean isLookupMessage(Object m) {
	    return hasForm(m, Lisp.list("lookup", String.class));
	}

        public String getLookupAgentName(Object m) {
            return (String)((List)m).get(1);
        }

        // Syntax-checking utility

	private boolean hasForm(Object m, List syntax) {
	    if (!(m instanceof List))
		return false;
	    Iterator mi = ((List)m).iterator(), si = syntax.iterator();
	    while (mi.hasNext() && si.hasNext()) {
		Object s = si.next();
		if (s instanceof String) {
		    if (!s.equals(mi.next()))
			return false;
		}
		else if (s instanceof Class) {
		    if (!((Class)s).isInstance(mi.next()))
			return false;
		}
		else
		    throw new ConsistencyException
			("Bad hasForm syntax: " + syntax);
	    }
	    return !mi.hasNext() && !si.hasNext();
	}

    }

}
