;;;; util-macros
;;; Author: Jeff Dalton <J.Dalton@ed.ac.uk>
;;; Created: July 2003
;;; Updated: Fri Aug  8 19:43:06 2003 by Jeff Dalton

(cl:in-package :util)

;;; Inline functions

;;; Using a special macro is neater than writing a proclamation and a
;;; defun, and it makes it easier for us to exploit other ways (such as
;;; defsubst) to get a function call expanded inline when (as in some
;;; Common Lisps) inline proclamations don't suffice.

(defmacro defun-inline (name parameters &body body)
  `(progn
     (proclaim '(inline ,name))
     (defun ,name ,parameters
       .,body)))


;;;; Output macro
;;;
;;; (OUTPUT directive...) is similar to the Franz Lisp "msg" macro.
;;; It is in some ways an alternative to FORMAT.  A directive can be:
;;;   a literal string       --  the string is printed by PRINC
;;;   the symbol //          --  a newline is printer by TERPRI
;;;   some other symbol      --  a variable: the value is printed by PRINC
;;;   a cons which can be:
;;;     a form               --  the value is printed by PRINC
;;;     a special directive  --  see below
;;;   something else         --  treated as a literal and printed by PRINC
;;;
;;; Special directives are defined by DEFINE-OUTPUT-EXPANDER.  This
;;; defines a function that is called with the cdr of the directive
;;; as it's arguments.  The function returns a form that is used
;;; directly in the expansion of OUTPUT.  Predefined special directives
;;; are:
;;;   (:STREAM form)
;;;      redirects output to the value of the form
;;;   (:INCLUDE form)
;;;      places the form in-line in the expansion of OUTPUT.
;;;   (:FORMAT format-string format-arg...)
;;;      becomes a call to FORMAT.
;;;

(defvar *output* (make-synonym-stream '*standard-output*))

(defmacro output (&rest directives)
  `(let ((*output* *output*))
     ,@(mapcar #'expand-output-directive directives)))

(eval-when (eval compile load)
  
(defun expand-output-directive (d)
  (cond ((stringp d)
         `(princ ,d *output*))
        ((eq d '//)
         `(terpri *output*))
        ((symbolp d)
         `(princ ,d *output*))
        ((consp d)
         (if (symbolp (car d))
             (let ((expander (get (car d) :output-expander)))
               (if expander
                   ;; Special-case expansion
                   (apply expander (cdr d))
                 ;; A random form
                 `(princ ,d *output*)))
           (error "Illegal output directive: ~S." d)))
        (t
         ;; A random object
         `(princ ',d *output*))))

);end eval-when


(defmacro define-output-expander (name parameters &body forms)
  (let ((fn-name (intern (concatenate 'string 
                           "%" (string name) "-OUTPUT-EXPANDER"))))
    `(eval-when (eval compile load)
       (defun ,fn-name ,parameters
         ,@forms)
       (setf (get ',name :output-expander) ',fn-name))))

;;; :STREAM sets *output* and hence the output destination (dynamically)
;;; within an OUTPUT form.

(define-output-expander :stream (form)
  `(setq *output* ,form))

;;; :INCLUDE gives a form complete control over what output occurs.

(define-output-expander :include (form)
  form)

;;; :FORMAT

(define-output-expander :format (format-string &rest format-args)
  `(format *output* ,format-string ,@format-args))

;;; End
