;;;; listof-gen (listof generator)
;;; Author: Jeff Dalton <J.Dalton@ed.ac.uk>
;;; Created: August 2003
;;; New method: November 2004
;;; Updated: Thu Sep 28 16:49:57 2006 by Jeff Dalton

;;; New method:
;;;   gen/gen-all-listofs java

;;; Full syntax:
;;;   gen/gen-all-listofs source_dir [output_dir]
;;;     where output_dir defaults to java-listofs.

;;; Testing:
;;;   gen/gen-listofs
;;;   etc/path-javac `find java-listofs -name \*.java`

;;; Jikes compile:
;;;   etc/path-jikes `find java-listofs/ -name \*.java`

;;; To specify the base directory (default "java-listofs"):
;;;   gen/gen-listofs dir

;;; To generate the classes for one specified list type:
;;;   gen/gen-listofs [dir] package element-class

;;; Finding classes:
;;;   grep Type doc/xml-syntax/ix.xsd | grep list-of | grep -v element

(cl:in-package :listof)

;;; The old method needs:

(defparameter *specs*
  ;; Package              Element class
  '(("ix.icore"           "Issue")
    ("ix.icore.domain"    "Constrainer")
    ("ix.icore.domain"    "Constraint")
    ("ix.icore.domain"    "Ordering")
    ("ix.icore.domain"    "PatternAssignment")
    ("ix.icore.domain"    "Refinement")
    ("ix.icore.domain"    "VariableDeclaration")
    ("ix.icore.domain"    "NodeSpec")
    ("ix.icore.plan"      "PlanVariableDeclaration")
    ("ix.icore.plan"      "PlanIssue")
    ("ix.icore.plan"      "PlanIssueRefinement")
    ("ix.icore.plan"      "PlanNode")
    ("ix.icore.plan"      "PlanRefinement")
    ("ix.test.xml"        "MapEntry")));

(defparameter *implementation-classes*
  '("ArrayList" "LinkedList"))

;;; The script for the new method needs:

(defparameter *listof-classes* nil)
(defparameter *element-classes* nil)

;;; Other parameters:

(defparameter *base-directory* "java-listofs")

(defparameter *default-listof-package* "ix.util")

(defparameter *always-implement*
  '("LinkedList"))

(defvar *java-package* "Not a java package")
(defvar *elt-class* "Not an element class")

(defvar *class-name*)			;bound by with-output-to-class-file
(defvar *class-filename*)		;bound by with-output-to-class-file

;;; Old method:

(defun generate (&rest args
		 &key (specs *specs* specs-p)
		      ((:base-directory *base-directory*) *base-directory*)
		      (package nil)
		      (element-class nil))
  ;; Can specify specs or else both package and element class.
  ;; The default is *specs* as the specs.
  (format t "Generating~{ ~S~}~%" args)
  (cond ((or package element-class)
	 (if (and package element-class (not specs-p))
	     (gen-defs package element-class)
	   (error "Must give :package and :element-class, without :specs")))
	(t
	 (loop for (package element-class) in specs
	       do (gen-defs package element-class)))))

;;; New method:

(defun generate-listofs
           (&key ((:base-directory *base-directory*) *base-directory*)
		 (listof-classes *listof-classes*)
		 (element-classes *element-classes*))
  "The :base-directory is where the output file tree will be built.
  :listof-classes contains all the names of the form ListOfE or ImplListOfE.
  :element-classes contains package-qualified E class names."
  (format t "~&Base = ~S~%Listofs = ~S~%Elements = ~S~%"
	  *base-directory* listof-classes element-classes)
  ;; For each element class, E, element-classes should contain
  ;; a package-qualified name for the class, and listof-classes
  ;; should contain ListOfE and implementation class names
  ;; such as LinkedListOfE and ArrayListOfE.
  (let* ((listofs (mapcar #'string listof-classes))
	 (elements-with-packages (mapcar #'string element-classes))
	 (elements (mapcar #'(lambda (x) (sequence-after-last "." x))
			   elements-with-packages))
	 (packages (mapcar #'(lambda (x) (sequence-before-last "." x))
			   elements-with-packages))
	 (impl-table (make-implementation-table listofs)))
    (let ((packageless-elements '()))
      (maphash #'(lambda (k v)
		   (unless (member k elements :test #'equal)
		     (push k packageless-elements)))
	       impl-table)
      (when packageless-elements
	(warn "Element classes with unknown packages: ~S.~%~
              The ~S package will be used for the generated definitions."
	      packageless-elements
	      *default-listof-package*)
	(setq elements (append elements packageless-elements))
	(setq packages
	      (append packages
		      (loop for e in packageless-elements
			    collect *default-listof-package*)))))
    (maphash #'(lambda (k v)
		 (format t "~&Implementations for ~S = ~S~%" k v))
	     impl-table)
    (mapc #'(lambda (pack elt-class)
	      (let ((impls (gethash elt-class impl-table)))
		(when (null impls)
		  (warn "No implementations for ~S" elt-class))
		(let ((*implementation-classes* impls))
		  (format t "~&Generating ~S ~S ~S"
			  pack elt-class impls)
		  (gen-defs pack elt-class))))
	  packages
	  elements)))

(defun make-implementation-table (listofs )
  (let ((table (make-hash-table :test #'equal))
	(given-elt-classes '())		;appear as ListOfE
	(derived-elt-classes '()))	;appear in ImplListOfE for some Impl
    (dolist (name listofs)
      (cond ((not (search "ListOf" name))
	     (error "Invalid listof name ~S" name))
	    ((sequence-begins "ListOf" name)
	     (let ((elt-class (sequence-after "ListOf" name)))
	       (assert (not (member elt-class given-elt-classes)))
	       (push elt-class given-elt-classes)
	       ;; Ensure that there's a table entry even if no impls.
	       (setf (gethash elt-class table) (gethash elt-class table))))
	    (t
	     (let* ((elt-class (sequence-after-first "ListOf" name))
		    (impl (concat-string (sequence-before-first "ListOf" name)
					 "List"))
		    (already (gethash elt-class table)))
	       (assert (not (member impl already)))
	       (pushnew elt-class derived-elt-classes :test #'equal)
	       (setf (gethash elt-class table)
		     (cons impl already))))))
    (unless (equal-as-sets given-elt-classes derived-elt-classes)
      ;; Sometimes there are ListOf classes without implementations.
      ;; This warning also happens if a ListOfE class is mentioned
      ;; but E isn't defined.
      (warn "Given and derived elt classes don't match~%~
             Given = ~S~%~
             Derived = ~S~%~
             Not in common = ~S"
	    given-elt-classes derived-elt-classes
	    (set-difference
	      (union given-elt-classes derived-elt-classes :test #'equal)
	      (intersection given-elt-classes derived-elt-classes
			    :test #'equal)
	      :test #'equal)))
    (maphash #'(lambda (elt-class impl-classes)
		 (let ((missing (set-difference *always-implement*
						impl-classes :test #'equal)))
		   (when missing
		     (warn "ListOf~S is missing implementations ~S."
			   missing)
		     (setf (gethash elt-class table)
			   (union (gethash elt-class table) *always-implement*
				  :test #'equal)))))
	     table)
    table))
	       

;;;; gen-defs
;;;
;;; Given a package and an element class name, EltClass, generate:
;;;  - an interface, ListOfEltClass
;;;  - implementations ArrayListOfEltClass and LinkedListOfEltClass
;;;  - an interface, EltClassIterator
;;;  - an implementation of that interface for each of the
;;;    implementation classes above.
;;;
;;; /\/: For now we won't have EltClassListIterator.

(defun gen-defs (package elt-class)
  (let ((*java-package* package)
	(*elt-class* elt-class))
    (gen-list-interface)
    (gen-list-implementations)
    (gen-iterator-interface)))

(defun class-filename (class-name)
  (let ((pkg-path (replace-subseq "/" "." *java-package*)))
    (concat-string *base-directory* "/" pkg-path "/" class-name ".java")))

(defun gen-list-interface ()
  (with-output-to-class-file ("ListOf" *elt-class*)
    (output-file-header)
    (output
      "import ix.util.TypedList;"
      //
      "/**" //
      " * A List whose elements are instances of {@link " *elt-class* "}." //
      " */" //
      "public interface " *class-name* 
                 " extends TypedList<" *elt-class* "> {" //
      "    public static final Class _elementClass = " *elt-class*".class;" //
      "    public static final Class[] _implementations = {" //
               (:include 
                 (output-comma-separated "        " (impl-class-refs))) //
      "    };" //
      "    public " *elt-class* " elt(int index);" //
      "    public " *elt-class*"Iterator " "elements();" //
      "}" //)))

(defun impl-class-refs ()
  (mapcar #'(lambda (impl-class)
	      (concat-string impl-class "Of" *elt-class* ".class"))
	  *implementation-classes*))

(defun gen-list-implementations ()
  (dolist (impl-class *implementation-classes*)
    (gen-list-impl impl-class)))

(defun gen-list-impl (impl-class)
  (with-output-to-class-file (impl-class "Of" *elt-class*)
    (output-file-header)
    (when (equal impl-class "ContextList")
      ;; /\/: Irritating special case.
      (output "import ix.util.context.ContextList;" //));
    (output
      "import ix.util.ListOf;" //
      //
      "public class " *class-name*
                    " extends " impl-class "<" *elt-class* ">" //
      "       implements " "ListOf"*elt-class* " {" //
      //
      "    public " *class-name* "() { super(); }" //
      "    public " *class-name* "(Collection c) { super(c); check(c); }" //
      //
      "    public Class elementClass() { return _elementClass; }" //
      //
      "    public boolean add(" *elt-class* " o) {" //
      "        return super.add(o);" //
      "    }" //
      "    public void add(int i, " *elt-class* " o) {" //
      "        super.add(i, o);" //
      "    }" //
      "    public boolean addAll(Collection<? extends " *elt-class* "> c) {" //
      "        check(c); return super.addAll(c);" //
      "    }" //
      "    private void check(Collection c) {" //
      "        ListOf.checkElements(c, " *elt-class*".class);" //
      "    }" //
      //
      "    public " *elt-class* " elt(int index) {" //
      "        return ("*elt-class*")super.get(index);" //
      "    }" //
      //
      "    public " *elt-class*"Iterator " "elements() {" //
      "        return new TypedIterator();" //
      "    }" //
      //
      "    private class TypedIterator implements " *elt-class*"Iterator { " //
      "        private Iterator iter = iterator();" //
      "        public boolean hasNext() {" //
      "            return iter.hasNext();" //
      "        }" //
      "        public " *elt-class* " next() {" //
      "            return ("*elt-class*")iter.next();" //
      "        }" //
      "        public void remove() {" //
      "            iter.remove();" //
      "        }" //
      "    }" //
      //
      "}" //)))

(defun gen-iterator-interface ()
  ;; Does not extend Iterator, because the programmer can always
  ;; ask the collection for an ordinary Iterator if that's what's
  ;; wanted -- and this lets us use the next() method rather
  ;; than have to define a different name.
  (with-output-to-class-file (*elt-class* "Iterator")
    (output-file-header)
    (output
      "public interface " *class-name* " {" //
      "    boolean hasNext();" //
      "    " *elt-class* " next();" //
      "    void remove();" //
      "}" //)))

(defun output-file-header ()
  (output "/* " *class-name* ", " (time-and-date-string) //
	  " * Generated file - do not edit." //
	  " */" //
	  //
	  "package " *java-package* ";" //
	  //
	  "import java.util.*;" //
	  //))

;;; End

