;;; Generates field declarations and corresponding get and set methods.

(use-package :util)

(eval-when (eval compile load)
  (setf (readtable-case *readtable*) :invert))

(defparameter *output-directory* "/tmp")

(defparameter *java-package* nil)

(defvar *all-defined-classes* '())

(defmacro define-fields (class-name &rest options-and-fields)
  ;; options are:
  ;;   (:abstract t)
  ;;   (:extends class-name)
  ;;   (:imports (package-name...))
  (let ((options (remove-if-not #'keywordp options-and-fields :key #'car))
	(fields (remove-if #'keywordp options-and-fields :key #'car)))
    `(progn
       (push ',class-name *all-defined-classes*)
       (setf (get ',class-name 'options) ',options)
       (setf (get ',class-name 'fields) ',fields))))

(defmacro in-java-package (package-name)
  `(setq *java-package* ',package-name))

(defun generate-classes (&key ((:output-directory *output-directory*)
			       *output-directory*))
  (dolist (class (reverse *all-defined-classes*))
    (let ((filename (concatenate 'string *output-directory* "/"
				 (string class) ".java")))
      (ensure-directories-exist filename :verbose t)
      (with-open-file (out filename :direction :output :if-exists :supersede)
	(let ((*standard-output* out))
	  (output-file-header class)
	  (display-fields class))))))

(defun output-file-header (class)
  (format t "/* ~S, ~A */~%" class (time-and-date-string))
  (format t "/* Generated file -- do not edit. */~%~%")
  (when *java-package*
    (format t "package ~A;~%~%" *java-package*))
  (let ((imports (get-option class :imports)))
    (when imports
      (format t "~{import ~S;~%~}~%" imports))))

(defun display-fields (class-name)

  ;; Class dcl
  (format t "public ~Aclass ~S~A {~%~%"
	  (if (get-option class-name :abstract) "abstract " "")
	  class-name
	  (let ((super (get-option class-name :extends)))
	    (if super
		(concat-string " extends " super)
	      "")))

  ;; Fields
  (gen-fields (get class-name 'fields))

  ;; A 0-arg constructor
  (format t "    public ~S() {~%" class-name)
  (format t "        super();~%")
  (format t "    }~%~%")

  ;; Get and Set methods
  (gen-methods (get class-name 'fields))
  
  (format t "}~%"))

(defun gen-fields (field-specs)
  ;; A field-spec is (class-name field-name).
  ;; Output field declarations
  (dolist (spec field-specs)
    (let ((class-name (first spec))
	  (field-name (second spec)))
      (format t "    protected ~S ~S;~%" class-name field-name)))
  (format t "~%"))

(defun gen-methods (field-specs)
  ;; A field-spec is (class-name field-name).
  ;; Output method declarations
  (dolist (spec field-specs)
    (let ((class-name (first spec))
	  (field-name (second spec)))
      (format t "    public ~S get~S() {~%"
	      class-name (capitalize-1st field-name))
      (format t "        return ~S;~%" field-name)
      (format t "    }~%~%")
      (format t "    public void set~S(~S ~S) {~%"
	      (capitalize-1st field-name) class-name field-name)
      (format t "        this.~S = ~S;~%"
	      field-name field-name)
      (format t "    }~%~%"))))

(defun get-option (class-name option-name)
  (cadr (assoc option-name (get class-name 'options))))

(defun capitalize-1st (string)
  (setq string (princ-to-string string)) ;allow symbol
  (read-from-string			 ;N.B. intern makes single cap into lc
   (concatenate 'string
		(string-capitalize (subseq string 0 1))
		(subseq string 1))))

;;; End

