;;; LispInterpreter tests
;;; Author: Jeff Dalton
;;; Updated: Wed Apr 13 06:37:11 2005 by Jeff Dalton

;;; Remember that each test group gets a new LispInterpreter.

(define-test-group predicate-tests

  ((symbolp 'a) ==> TRUE)
  ((symbolp 12) ==> FALSE)

  ((numberp 12) ==> TRUE)
  ((numberp 'a) ==> FALSE)
  ((numberp "12") ==> FALSE)
  ((numberp '|12|) ==> FALSE)

  ((stringp "apple") ==> TRUE)
  ((stringp 'pie) ==> FALSE)

  ((listp '(apple pie)) ==> TRUE)
  ((listp '()) ==> TRUE)
  ((listp 'a) ==> FALSE)

  ((consp '(apple pie)) ==> TRUE)
  ((consp '()) ==> FALSE)
  ((consp 'a) ==> FALSE)

  ((null '(apple pie)) ==> FALSE)
  ((null '()) ==> TRUE)
  ((null 'a) ==> FALSE)

  ((not (numberp 12)) ==> FALSE)
  ((not (not (numberp 12))) ==> TRUE))

(define-test-group nil-tests
  ((symbolp 'nil) ==> FALSE)		;unlike Common Lisp
  ((listp 'nil) ==> TRUE)
  ((symbolp nil) ==> FALSE)
  ((listp nil) ==> TRUE))

(define-test-group list-tests
  ((cons 1 nil) ==> '(1))
  ((cons 1 (cons 2 nil)) ==> '(1 2))
  ((list) ==> '())
  ((list 1) ==> '(1))
  ((list 1 2) ==> '(1 2))
  ((list 1 2 3) ==> '(1 2 3))
  ((list 'some (list 'apple 'pie) 'is 'nice) ==> '(some (apple pie) is nice))
  ((car '(1 2 3)) ==> '1)
  ((cdr '(1 2 3)) ==> '(2 3))
  ((car (cdr (cdr '(1 2 3)))) ==> 3)
  ((car nil) ==> nil)
  ((cdr nil) ==> nil))

(define-test-group eq-tests
  ;; We're avoiding any tests that depend on whether separate
  ;; occurrences of equal quoted or self-evaluating objects are
  ;; eq, except, of course, for symbols.  We also don't assume
  ;; that integer arithmetic always produces a new object for
  ;; the result.
  ((eq 'a 'a) ==> TRUE)
  ((eq 'a 'b) ==> FALSE)
  ((eq nil '()) ==> TRUE)
  ((eq (list 'a) (list 'a)) ==> FALSE))

(define-test-group equal-tests
  ((equal 'a 'a) ==> TRUE)
  ((equal 'a 'b) ==> FALSE)
  ((equal (list 'a) (list 'a)) ==> TRUE)
  ((equal 1 1) ==> TRUE)
  ((equal 1.0 1.0) ==> TRUE)
  ((equal 1 1.0) ==> FALSE)
  ((equal 1.0 1) ==> FALSE)
  ((equal '(some (apple pie) is nice) '(some (apple pie) is nice)) ==> TRUE)
  ((equal '(apple (pie) is nice) '(apple (crumble) is nice)) ==> FALSE)
  ((equal "apple" "apple") ==> TRUE)
  ((equal "apple" "apqle") ==> FALSE)
  ((equal "Apple" "apple") ==> FALSE))

(define-test-group arithmetic-tests

  ((+) ==> 0)
  ((+ 3) ==> 3)
  ((+ 1 2 3) ==> 6)
  ((+ 1 2 3 4 5 6 7) ==> 28)

  ((*) ==> 1)
  ((* 3) ==> 3)
  ((* 1 2 3) ==> 6)
  ((* 1 2 3 4 5 6 7) ==> 5040)

  ((- 10 9) ==> 1)
  ((- 10 9.0) ==> 1.0)
  ((- 10.0 9) ==> 1.0)
  ((- 10.1 9.1) ==> 1.0)

  ((/ 7 2) ==> 3)
  ((/ 7 2.0) ==> 3.5)
  ((/ 7.0 2) ==> 3.5)

  ((* (/ 10 2) 2) ==> 10)

  )

(define-test-group numeric-comparison-tests ; /\/
  ((= 1 1) ==> TRUE)
  ((= 1.0 1.0) ==> TRUE)
  ((= 1 1.0) ==> TRUE))

(define-test-group and-tests
  ((and) ==> TRUE)
  ((and 1) ==> 1)
  ((and 1 2) ==> 2)
  ((and 1 2 3) ==> 3)
  ((and 1 FALSE 3) ==> FALSE))

(define-test-group or-tests
  ((or) ==> FALSE)
  ((or 1) ==> 1)
  ((or 1 2) ==> 1)
  ((or 1 2 3) ==> 1)
  ((or FALSE 2 3) ==> 2)
  ((or (= 0 0) unbound) ==> TRUE))

(define-test-group if-tests
  ((if 1 2 3) ==> 2)
  ((if (not TRUE) 2 3) ==> 3)
  ((if 1 (if 2 3 4) (if 5 6 7)) ==> 3)
  ((if 1 2 unbound) ==> 2)
  ((if (= 7 8) unbound 9) ==> 9))

(define-test-group cond-tests
  ((cond) ==> nil)
  ((cond (1)) ==> 1)
  ((cond (1 2)) ==> 2)
  ((cond (1 2 3)) ==> 3)
  ((cond (1 2) (err or)) ==> 2)
  ((cond ((= 1 2) "no!") ((> 1 2) "not!") ((< 1 2) "yes!")) ==> "yes!"))

(define-test-group numeric-same-or-different-tests
  ((defun same (a b)
     (and (= a b) (>= a b) (<= a b)
	  (not (/= a b))
	  (not (> a b))
	  (not (< a b)))) ==> 'same)
  ((defun different (a b)
     (and (not (= a b))
	  (not (and (>= a b) (<= a b)))
	  (/= a b)
	  (or (> a b) (< a b)))) ==> 'different)
  ((same 3 3) ==> TRUE)
  ((same 3.0 3) ==> TRUE)
  ((same 3 3.0) ==> TRUE)
  ((same 3.0 3.0) ==> TRUE)
  ((same 3 4) ==> FALSE)
  ((same 4 3) ==> FALSE)
  ((same 3.0 4.0) ==> FALSE)
  ((same 4.0 3.0) ==> FALSE)
  ((different 3 4) ==> TRUE)
  ((different 4 3) ==> TRUE)
  ((different 3.0 4.0) ==> TRUE)
  ((different 4.0 3.0) ==> TRUE)
  ((different 3 3) ==> FALSE)
  ((different 3.0 3) ==> FALSE)
  ((different 3 3.0) ==> FALSE)
  ((different 3.0 3.0) ==> FALSE))

(define-test-group apply-tests
  ((apply + '()) ==> 0)
  ((apply + '(1)) ==> 1)
  ((apply + '(1 2)) ==> 3)
  ((apply + '(1 2 3)) ==> 6)
  ((apply + 1 '(2 3)) ==> 6)
  ((apply + 1 2 '(3)) ==> 6)
  ((apply + 1 2 3 '()) ==> 6))

(define-test-group identity-tests
  ((identity 1) ==> 1)
  ((identity (+ 2 3)) ==> 5))

(define-test-group simple-setq-tests
  ((setq apple 'pie) ==> 'pie)
  (apple ==> 'pie))

(define-test-group simple-progn-tests
  ((progn) ==> nil)
  ((progn 1 2) ==> 2)
  ((progn 1 2 3) ==> 3)
  ((progn 1 FALSE 3) ==> 3)
  ((progn (setq a 10) (setq a 2) a) ==> 2))

(define-test-group simple-let-tests

  ;; This also tests that variable references work correctly.

  ((let () 10) ==> 10)

  ((let ((a 1) (b 2))
     (let ((a 3))
       (list a b)))
   ==> '(3 2))

  ((let ((a 1) (b 2))
     (let ((c 3))
       (let ((d 4))
         (list a b c d))))
   ==> '(1 2 3 4))

  ((let ((a 1) (b 2))
     (let ((c 3))
       (let ((d 4)) d)
       (list a b c)))
   ==> '(1 2 3))

  ((let ((a 1) (b 2))
     (let ((c 3))
       (let ((d 4))
	 ;; This time, evaluate the vars more than once
         (let ((e1 (list a b c d))
	       (e2 (list a b c d)))
	   (list e1 e2 (equal e1 e2))))))
   ==> (list '(1 2 3 4) '(1 2 3 4) TRUE))

  ((let ((a 1))
     (setq a 11)  ; set local a
     (setq b 2)   ; set top-level b
     (setq c 3)   ; set top-level c
     (let ((c 4)) ; shadow top-level c
       (list a b c)))
   ==> '(11 2 4))

  (a ==> :error)  ; check that top-levell a still has no value
  (b ==> 2)       ; but top-level b and c should have the assigned values
  (c ==> 3))      ;

(define-test-group let-eq-tests
  ;; Lists
  ((let ((a (list 1 2 3)))
     (let ((b a))
       (eq a b))) ==> TRUE)
  ((let ((a (list 1 2 3)))
     (let ((b a))
       (eq (cdr a) (cdr b)))) ==> TRUE)
  ;; Numbers
  ((let ((a 12.3))
     (eq a a)) ==> TRUE)
  ((let ((a 12.3))
     (let ((b a))
       (eq a b))) ==> TRUE))

(define-test-group simple-let*-tests
  ((let* () 10) ==> 10)
  ((let* ((a 1) (b (+ a a)) (c (+ a b)))
     c) ==> 3))

(define-test-group simple-closure-with-assignment-tests

  ((let ((f nil))
     (let ((a 1))
       (setq f (lambda (k) (setq a (+ a k)))))
     (f 10)
     (f 22)) ==> (+ 1 10 22))

  (a ==> :error) ; check that top-level a still has no value
  (f ==> :error)

  ((let ((up nil) (down nil))
     (let ((a 1))
       (setq up (lambda (k) (setq a (+ a k))))
       (setq down (lambda (k) (setq a (- a k)))))
     (up 10)
     (down 22)
     (up 17)) ==> (+ 1 10 -22 17))

  (a ==> :error))

(define-test-group collection-tests
  ((setq listCol (make-collection 'list '(a a c c c b b))) ==> listCol)
  ((setq setCol (make-collection 'set listCol)) ==> setCol)
  ((setq sortedCol (make-collection 'sorted-set listCol)) ==> sortedCol)
  ((make-list listCol) ==> '(a a c c c b b))
  ((make-list sortedCol) ==> '(a b c))
  ((contains setCol 'a) ==> TRUE)
  ((contains setCol 'b) ==> TRUE)
  ((contains setCol 'c) ==> TRUE)
  ((progn
     (defun is-abc (col)
       (let ((elts (make-list col)))
	 (and (null (cdr (cdr (cdr elts)))) ; instead of length
	      (contains elts 'a)
	      (contains elts 'b)
	      (contains elts 'c))))
     (is-abc setCol))
   ==> TRUE)
  ((remove setCol 'q) ==> FALSE)
  ((add setCol 'q) ==> TRUE)
  ((add setCol 'a) ==> FALSE)
  ((remove setCol 'q) ==> TRUE)
  ((is-abc setCol) ==> TRUE))

(define-test-group empty-tests
  ((empty '()) ==> TRUE)
  ((empty "") ==> TRUE)
  ((empty (make-collection 'set '())) ==> TRUE)
  ((empty '(a)) ==> FALSE)
  ((empty "a") ==> FALSE)
  ((empty (make-collection 'set '(a))) ==> FALSE))

(define-test-group length-tests
  ((length '()) ==> 0)
  ((length "") ==> 0)
  ((length (make-collection 'set '())) ==> 0)
  ((length '(a)) ==> 1)
  ((length "a") ==> 1)
  ((length (make-collection 'set '(a))) ==> 1)
  ((length (make-collection 'set '(a b b c a))) ==> 3)
  ((length '(a b c)) ==> 3)
  ((length "apple pie") ==> 9)
  ((let ((c (make-collection 'list '(a))))
     (add c 'b)
     (add c 'd)
     (length c)) ==> 3))

(define-test-group elt-tests
  ((elt '(a b c) 0) ==> 'a)
  ((elt "abc" 0) ==> "a")
  ((elt (make-collection 'list '(a b c)) 0) ==> 'a))

(define-test-group map-tests
  ((map (lambda (x) (+ x 1)) '(1 2 3 4))
   ==> '(2 3 4 5)))

(define-test-group for-each-tests
  ((let ((rev '()))
     (for-each (lambda (e) (setq rev (cons e rev)))
	       '(a b c d e f g))
     rev)
   ==> '(g f e d c b a)))

(define-test-group mapcar-tests
  ;; Remember that we define mapcar in Lisp, not Java.
  ((mapcar (lambda (x) (+ x 1)) '(1 2 3 4))
   ==> '(2 3 4 5)))

(define-test-group hash-table-tests
  ((setq colours (make-hash-map)) ==> colours)
  ((put colours 'grass 'green) ==> 'green)
  ((put colours 'apple 'green) ==> 'green)
  ((put colours 'sky 'blue) ==> 'blue)
  ((get colours 'apple 'unknown) ==> 'green)
  ((get colours 'sky 'unknown) ==> 'blue)
  ((get colours 'grass 'unknown) ==> 'green)
  ((get colours 'banana 'unknown) ==> 'unknown)
  ((put colours 'apple 'red) ==> 'red)	;change colour
  ((get colours 'apple 'unknown) ==> 'red)
  ((let ((mappings (make-collection 'sorted-set '())))
     (for-each-entry (lambda (k v) (add mappings (list k v)))
		     colours)
     (make-list mappings))
   ==> '((apple red) (grass green) (sky blue))))

;;; From this point, the tests use more interesting algorithms.

(define-test-group factorial-tests
  ((defun f (n) (if (= n 0) 1 (* n (f (- n 1))))) ==> 'f)
  ((f 7) ==> 5040)
  ((defun f2 (n)
     (let ((result 1))
       (while (> n 0)
         (setq result (* n result))
	 (setq n (- n 1)))
       result)) ==> 'f2)
  ((f2 7) ==> 5040))

(define-test-group Y-factorial-tests
  ((defun Y (f)
     (let ((r (lambda (r) (lambda (arg) ((f (r r)) arg)))))
       (r r)))
   ==> 'Y)
  ((let ((factorial
          (lambda (f)
            (lambda (n)
              (if (= n 0)
                  1
                  (* n (f (- n 1))))))))
     ((Y factorial) 7))
   ==> 5040))

(define-test-group ackermann-tests
  ;; See http://en.wikipedia.org/wiki/Ackermann_function
  ((defun ack (m n)
     (if (= m 0)
	 (+ n 1)
       (if (= n 0)
	   (ack (- m 1) 1)
	 (ack (- m 1) (ack m (- n 1))))))
   ==> 'ack)
  ((ack 1 3) ==> 5)
  ((ack 1 4) ==> 6)
  ((ack 2 3) ==> 9)
  ((ack 3 3) ==> 61)
  ((ack 3 4) ==> 125)
  ((ack 3 5) ==> 253)
  ((- (* 8 2 2 2 2 2) 3) ==> 253)
  ((ack 4 0) ==> 13))

(define-test-group truth-table-tests
  ;; Here we apply a tautologous formula to every possible assignment
  ;; of truth values to the variables in the formula and expect
  ;; that the result will always be true.  The formula is represented
  ;; as a function.
  ((defun truth-table (number-of-vars fn)
     (truth1 number-of-vars '() fn))
   ==> 'truth-table)
  ((defun truth1 (n vals fn)
     (if (= n 0)
	 (apply fn vals)
       (and
	 (truth1 (- n 1) (cons TRUE vals) fn)
	 (truth1 (- n 1) (cons FALSE vals) fn))))
   ==> 'truth1)
  ((defun implies (p q) (or (not p) q))
   ==> 'implies)
  ;; Test modus ponens
  ((truth-table 2
     (lambda (p q)
       (implies (and p (implies p q))
		q)))
   ==> TRUE)
  ;; Test constructive dilemma
  ((truth-table 4
     (lambda (a b c d)
       (implies (and (implies a c)
		     (implies b d)
		     (or a b))
		(or c d))))
   ==> TRUE)
  ;; Test the fallacy of affirming the consequent.
  ;; This time we expect the result to be false.
  ((truth-table 2
     (lambda (p q) 
       (implies (and (implies p q)
		     q)
		p)))
   ==> FALSE)
  ;; De Morgan
  ((defun equiv (p q) (and (implies p q) (implies q p)))
   ==> 'equiv)
  ((truth-table 2
     (lambda (p q)
       (and (equiv (not (and p q)) (or (not p) (not q)))
	    (equiv (not (or p q)) (and (not p) (not q))))))
   ==> TRUE))

;;; End
