/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun May 14 18:58:55 2006 by Jeff Dalton
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 */

package ix.util.lisp;

import java.util.*;

import junit.framework.*;

import ix.util.AbstractListTest;
import ix.util.Seq;

/**
 * LList test cases.
 */
public class LListTest extends AbstractListTest
       implements ConvenientSymbols {

    protected LList abc;

    public LListTest(String name) {
	super(name);
    }

    protected Collection makeEmptyCollection() {
	return Lisp.list();	// N.B. not modifiable
    }

    protected Collection makeCollection(Collection source) {
	// Because the empty LList isn't modifiable, we can't use
	// the inherited method which makes an empty collection
	// and then calls its addAll method.
	return LList.newLList(source);
    }

    protected void setUp() {
	abc = (LList)Lisp.readFromString("(a b c)");
    }

    public void testLength() {
	assertEquals(3, abc.length());
    }

    public void testAppend() {
	assertEquals(Lisp.readFromString("(a b c a b c)"),
		     abc.append(abc));
    }

    public void testWithoutFirst() {
	// Try removing something that's there.
	LList ac = abc.withoutFirst(SYM_B);
	assertEquals(Lisp.readFromString("(a c)"), ac);
	// Now something that's not there.
	LList same = abc.withoutFirst("not there");
	assertTrue(same == abc);
    }

    public void testWithoutAll() {
	// Try removing something that's there.
	LList a = abc.withoutAll(Lisp.list(SYM_C, SYM_B));
	assertEquals(Lisp.readFromString("(a)"), a);
	// Now something that's not there.
	LList same = abc.withoutAll(Lisp.list("apple", "pie"));
	assertTrue(same == abc);
    }

    public void testAlistLookup() {
	LList alist = (LList)Lisp.readFromString("((a 1) (b 2) (c 3))");
	assertEquals(new Long(1), alist.alistValue(SYM_A));
	assertEquals(new Long(2), alist.alistValue(SYM_B));
	assertEquals(new Long(3), alist.alistValue(SYM_C));
	assertNull(alist.alistValue("Apple pie"));
    }

    public void testAlistToMapAndBack() {
	LList alist = (LList)Lisp.readFromString("((a 1) (b 2) (c 3))");
	assertEquals(alist,
		     // Must use a SortedMap to preserve the order.
		     LList.mapToAlist(alist.alistToMap(new TreeMap())));
    }

    public void testListCompareTo() {
	assertTrue(((LList)Lisp.readFromString("(1 2 2)"))
	    .compareTo(Lisp.readFromString("(1 2 3)"))
	    == -1);
	assertTrue(((LList)Lisp.readFromString("(1 2 2)"))
	    .compareTo(Lisp.readFromString("(1 2 2)"))
	    == 0);
	assertTrue(((LList)Lisp.readFromString("(1 2 3)"))
	    .compareTo(Lisp.readFromString("(1 2 2)"))
	    == +1);
    }

    // /\/: Convert the rest of the old tests.

}
