;;; A simple house-building domain with resources,
;;; based on the O-Plan domain three-pigs-2.

;;; Author: Jeff Dalton

;;; O-Plan domain created: 02 March 1994
;;; I-X domain created: 19 May 2007

;;; Updated: Sun May 20 03:05:29 2007 by Jeff Dalton

;;; Wolf-proof houses are more expensive.

;;; Tasks:
;;;
;;;   build-house
;;;     Cost is no object, but there's no requirement that the
;;;     house be secure.
;;;   build-secure-house
;;;     The house must be wolf-proof.
;;;     (Only one solution.)
;;;   better-build-secure-house
;;;     Like build-secure-house, but uses a more (search-) efficient builder.
;;;   build-cheap-house
;;;     The house must be inexpensive.
;;;     (More solutions than you expect.)
;;;   build-cheaper-house
;;;     New in I-X version.  The house must be even less expensive.
;;;     (Still more solutions than you expect but there isn't quite
;;;     enough money to use sticks this time.  So only straw plans remain.)
;;;   build-cheap-secure-house
;;;     The house must be inexpensive and wolf-proof.
;;;     (No solutions.)

;;; /\/: A number of schemas can be combined by using variables
;;; more widely and picking up prices from the world-state, etc.
;;; (That hadn't been done in the O-Plan version, and we're following that.)

;;; /\/: In constraints, we call the resources (resource money)
;;; rather than plain (money), following O-Plan's convention.

;;; /\/: Variables types have been replaces by "type" effects
;;; combined with "type" conditions that pick up type-members.

;;; /\/: We have had to omit supervised conditions.

;;; /\/: In the O-Plan domain, only_use_for_effects was used for
;;; all of the effects in the primitives, but there were no achieve
;;; conditions, so it didn't matter.  Hence this annotation:

(annotations
  (achievable-world-state-conditions = :none))

(refinement task-build-house (task-build-house)
  (nodes
    (1 (setup))
    (2 (build house)))
  (orderings
    (1 2)))

(refinement task-build-secure-house (task-build-secure-house)
  (nodes
    (1 (setup))
    (2 (build house))
    (3 (check security)))
  (orderings
    (1 2 3))
  (constraints
    (resource overall (resource money) = 2000))) ; 2000 pounds

(refinement task-better-build-secure-house (task-better-build-secure-house)
  (nodes
    (1 (setup))
    (2 (build secure house)))
  (orderings
    (1 2))
  (constraints
    (resource overall (resource money) = 2000))) ; 2000 pounds

(refinement task-build-cheap-house (task-build-cheap-house)
  (nodes
    (1 (setup))
    (2 (build house)))
  (orderings
    (1 2))
  (constraints
    (resource overall (resource money) = 500))) ; 500 pounds

(refinement task-build-cheaper-house (task-build-cheaper-house)
  (nodes
    (1 (setup))
    (2 (build house)))
  (orderings
    (1 2))
  (constraints
    (resource overall (resource money) = 499))) ; 499 pounds

(refinement task-build-cheap-secure-house (task-build-cheap-secure-house)
  (nodes
    (1 (setup))
    (2 (build house))
    (3 (check security)))
  (orderings
    (1 2 3))
  (constraints
    (resource overall (resource money) = 500))) ; 500 pounds


(annotations (oplan-type-definitions = " \
types material = (straw sticks bricks); \
\
always {proof_against wolf bricks} = true; \
\
resource_units pounds = count; \
\
resource_types \
  consumable_strictly {resource money} = pounds, \
  consumable_strictly {resource straw}, \
  consumable_strictly {resource sticks}, \
  consumable_strictly {resource bricks}; \
"))

(refinement setup (setup)
  (constraints
    ;; Enumarate the "material" type
    (world-state effect (type straw) = material)
    (world-state effect (type sticks) = material)
    (world-state effect (type bricks) = material)
    ;; Properties of materials
    (world-state effect (proof-against wolf bricks) = true))
  (annotations
    ;; Keep this refinement from being used to achieve a condition.
    (use-for-world-state-effects = :none)))

(refinement house-builder (build house)
  (variables ?material) ;; = ?(type material);
  (nodes
    (1 (purchase ?material))
    (2 (make-walls-from ?material))
    (3 (install door))
    (4 (install windows)))
  (orderings
    (1 2 (3 4)))
  ;; conditions supervised (have ?material) at 2 from [1],
  ;;            supervised (walls built) at 3 from [2],
  ;;            supervised (walls built) at 4 from [2];
)

;;; /\/: Security-checker is a separate schema, because we can't
;;; put the materials check directly in the task schema.  Secure-
;;; house-builder is better way of doing this, because it doesn't
;;; lead to so much search.

(refinement security-checker (check security);
  (variables ?material) ;; = ?(type material), local
  (constraints
    (world-state condition (proof-against wolf ?material) = true)
    (world-state condition (walls-of ?material) = true)
    (world-state condition (wolf-proof door) = true)
    (world-state condition (wolf-proof windows) = true)))

(refinement secure-house-builder (build secure house)
  ;; /\/: "bricks" is built-in, which is too specific.  Blame the O-Plan one.
  (nodes
    (1 (build-walls bricks))
    (2 (install door))
    (3 (install windows))
    ;; 4 was a dummy in the O-Plan version, for use in supervised conditions.
    ;; We've replaced that by a security check.  /\/
    (4 (check security)))
  (orderings
    (1 (2 3) 4))
  (constraints
    (world-state condition (proof-against wolf bricks) = true))
  ;; conditions supervised (walls-of bricks) at 4 from [1],
  ;;            unsupervised (proof-against wolf bricks) at 1,
  ;;            supervised (walls built) at 2 from [1],
  ;;            supervised (walls built) at 3 from [1],
  ;;            supervised (wolf-proof door) at 4 from [2],
  ;;            supervised (wolf-proof windows) at 4 from [3];
)

(refinement build-walls (build-walls ?material)
  (variables ?material) ;; = ?(type material)
  (nodes
    (1 (purchase ?material))
    (2 (make-walls-from ?material)))
  (orderings
    (1 2))
  ;; conditions supervised (have ?material) at 2 from [1];
)

;;; Some primitive actions.

(refinement purchase-straw (purchase straw)
  (constraints
    (world-state effect (have straw) = true)  ; use for effect
    (resource consume (resource money) = 100)
    (resource consume (resource straw) = 1000)))

(refinement purchase-sticks (purchase sticks)
  (constraints
    (world-state effect (have sticks) = true) ; use for effect
    (resource consume (resource money) = 200)
    (resource consume (resource sticks) = 1000)))

(refinement purchase-bricks (purchase bricks)
  (constraints
    (world-state effect (have bricks) = true)  ; use for effect
    (resource consume (resource money) = 1000)
    (resource consume (resource bricks) = 1000)))

(refinement make-straw-walls (make-walls-from straw)
  (constraints
    (world-state condition (have straw) = true)   ; from a supervised
    (world-state effect (walls built) = true)     ; use for effect
    (world-state effect (walls-of straw) = true)  ; use for effect
    (resource consume (resource money) = 100)))

(refinement make-stick-walls (make-walls-from sticks)
  (constraints
    (world-state condition (have sticks) = true)   ; from a supervised
    (world-state effect (walls built) = true)      ; use for effect
    (world-state effect (walls-of sticks) = true)  ; use for effect
    (resource consume (resource money) = 200)))

(refinement make-brick-walls (make-walls-from bricks)
  (constraints
    (world-state condition (have bricks) = true)   ; from a supervised
    (world-state effect (walls built) = true)      ; use for effect
    (world-state effect (walls-of bricks) = true)  ; use for effect
    (resource consume (resource money) = 500)))

(refinement install-wolf-proof-door (install door)
  (constraints
    (world-state condition (walls built) = true)   ; from a supervised
    (world-state effect (door installed) = true)   ; use for effect
    (world-state effect (wolf-proof door) = true)  ; use for effect
    (resource consume (resource money) = 100)))

(refinement install-door (install door)
  (constraints
    (world-state condition (walls built) = true)   ; from a supervised
    (world-state effect (door installed) = true)   ; use for effect
    (resource consume (resource money) = 50)))

(refinement install-wolf-proof-windows (install windows)
  (constraints
    (world-state condition (walls built) = true)      ; from a supervised
    (world-state effect (windows installed) = true)   ; use for effect
    (world-state effect (wolf-proof windows) = true)  ; use for effect
    (resource consume (resource money) = 100)))

(refinement install-windows (install windows)
  (constraints
    (world-state condition (walls built) = true)     ; from a supervised
    (world-state effect (windows installed) = true)  ; use for effect
    (resource consume (resource money) = 50)))

;;; End
