/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Dec  2 15:39:08 2005 by Jeff Dalton
 * Copyright: (c) 2002 - 2005, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.*;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import java.io.*;

import java.util.*;

// Imports for using JDOM
// import java.io.IOException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Attribute;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import ix.icore.IXAgent;

import ix.ispace.ISpaceTool;

import ix.iface.util.ToolManager;
import ix.iface.util.ToolController;
import ix.iface.util.CatchingActionListener;

import ix.iface.util.*;
import ix.util.*;
import ix.util.lisp.*;

/** 
 * A frame that contains an XML editing panel.
 * 
 * @see XMLTreeEditor
 * @see XMLTreeEditPanel
 */
public class XMLTreeEditFrame extends ToolFrame implements ActionListener {

    protected IXAgent agent;

    protected Container contentPane = getContentPane();

    protected XMLTreeEditPanel editPanel;

    protected FrameGroup frameGroup;

    protected /* static */ TreeHelpFrame helpFrame = null;

    protected static FindFrame findFrame = null;

    protected static XMLTreeSendFrame sendFrame = null;

    protected static ToolManager toolManager;
    
    protected static FrameGroup globalFrameGroup;

    /** Manages namespaces for all tree editors and viewers. */
    static XMLTreeNamespaceManager namespaces;

    public XMLTreeEditFrame(IXAgent agent) {
	this(agent, agent.getAgentDisplayName() + " XML Tree Editor");
    }

    public XMLTreeEditFrame(IXAgent agent, String title) {
	// Default is BorderLayout.
	super(title);
	this.agent = agent;

	if (globalFrameGroup == null) {
	    globalFrameGroup = new FrameGroup(title);
	}
	globalFrameGroup.add(this);
	this.frameGroup = globalFrameGroup;

	ensureNamespaceManager();

	setIconImage(IconImage.getIconImage(this));
	setJMenuBar(makeMenuBar());

	editPanel = makeEditPanel();
	contentPane.add(editPanel, BorderLayout.CENTER);

	pack();
	// setSize(500, 600);	// width, height
	setSize(500, 550);	// width, height
	validate();
	editPanel.docSplit.setDividerLocation(0.55);

	// Make sure a "close" doesn't just leave it running in the
	// background...
	setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent e) {
		Debug.noteln("Main frame windowClosing event");
		closeFrame();
	    }
	});

    }

    public IXAgent getAgent() {
	return agent;
    }

    public FrameGroup getFrameGroup() {
	return frameGroup;
    }

    protected XMLTreeEditPanel makeEditPanel() { // factory method
	return new XMLTreeEditPanel();
    }

    protected void ensureNamespaceManager() {
	if (namespaces == null)
	    namespaces = new XMLTreeNamespaceManager
		(agent.getAgentSymbolName() + " XML Tree Editor Namespaces");
    }

    protected boolean isStandalone() {
	return agent instanceof XMLTreeEditor;
    }

    /**
     * Returns true if this frame allows editing and false
     * if it is a read-only viewer.
     */
    public boolean isEditor() {
	return true;
    }

    /**
     * Returns the negation of the value returned by
     * {@link #isEditor()}.
     */
    public boolean isViewer() {
	return !isEditor();
    }

    public void editDocument(Document doc) {
	editPanel.editDocument(doc);
    }

    public Document getDocument() {
	return editPanel.getDocument();
    }

    public void editObject(Object obj) {
	editPanel.editObject(obj);
    }

    public Object getObject() {
	return editPanel.getObject();
    }

    public void expandDocument(int depth) {
	editPanel.expandDocument(depth);
    }

    /**
     * A set of XMLTreeEditFrames that are logically part of the
     * same editor for selection, cut/paste, find, etc.
     */
    protected static class FrameGroup {
	int count = 0;			// total added
	String title;
	Map members = new WeakHashMap(); // Java doesn't have weak sets /\/
	FrameGroup(String title) {
	    this.title = title;
	}
	String getTitle() { return title; }
	int getCount() { return count; }
	int getNumberVisible() {
	    int numberVisible = 0;
	    for (Iterator i = memberSet().iterator(); i.hasNext();) {
		JFrame f = (JFrame)i.next();
		if (f.isVisible())
		    numberVisible++;
	    }
	    return numberVisible;
	}
	void add(XMLTreeEditFrame frame) {
	    count++;
	    members.put(frame, this);	// the value doesn't matter /\/
	}
	Set memberSet() {
	    return members.keySet();
	}
	public String toString() {
	    return "FrameGroup[" + title + "]";
	}
    }

    /**
     * Creates the menu bar
     */
    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	// "File" menu
	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(makeMenuItem("New Editor Window"));
	if (isStandalone() || isEditor()) {
	    // fileMenu.add(makeMenuItem("New Editor Window"));
	    fileMenu.add(makeMenuItem("Close Editor Window"));
	    if (!isStandalone())
		addLookAtMenu(fileMenu);
	    fileMenu.addSeparator();
	    fileMenu.add(makeMenuItem("Load From ..."));
	    fileMenu.add(makeMenuItem("Save As ..."));
	    if (isStandalone())
		fileMenu.add(makeMenuItem("Send To ..."));
	    fileMenu.addSeparator();
	    fileMenu.add(makeMenuItem("Reset"));
	    if (isStandalone())
		fileMenu.add(makeMenuItem("Exit"));
	}
	else {
	    // This frame is a viewer and part of another agent
	    fileMenu.add(makeMenuItem("Reset"));
	    fileMenu.add(makeMenuItem("Close"));
	}

	// "Edit" menu
	JMenu editMenu = new JMenu("Edit");
	bar.add(editMenu);
	editMenu.add(makeMenuItem("Check As Document"));
	editMenu.add(makeMenuItem("Check As Object"));
	editMenu.add(makeMenuItem("To Object And Back"));
	editMenu.addSeparator();
	editMenu.add(makeMenuItem("Find In Tree"));
	editMenu.addSeparator();
	editMenu.add(makeMenuItem("Namespaces"));

	// "View" menu
	JMenu viewMenu = new JMenu("View");
	bar.add(viewMenu);
	viewMenu.add(makeMenuItem("As XML"));

	// "Tools" menu
	if (isStandalone()) {
	    toolManager = makeToolManager();
	    bar.add(toolManager.getToolsMenu());
	}

	// "Help" menu
	JMenu helpMenu = new JMenu("Help");
	bar.add(helpMenu);
	helpMenu.add(makeMenuItem("Help"));

	return bar;
    }

    protected void addLookAtMenu(JMenu fileMenu) {
	JMenu menu = XML.config().makeTreeEditorLookAtMenu(this);
	if (menu != null) {
	    fileMenu.add(menu);
	}
    }

    ToolManager makeToolManager() {
	ToolManager tm = new ToolManager();
	tm.addTool(new ToolController("I-Space") {
	    public Object createTool() {
		return new ISpaceTool(agent);
	    }
	});
	return tm;
    }

    public void addTool(ToolController tc) {
	toolManager.addTool(tc);
    }

    // /\/: Must find a better way to do this rather than have
    // this method everywhere.
    protected JMenuItem makeMenuItem(String text) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(this));
	return item;
    }

    /**
     * Action interpreter for items in the frame's "File" menu.
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("XMLTreeEditFrame action:", command);

	if (command.equals("New Editor Window")) {
	    newEditorWindow();
	}
	else if (command.equals("Close Editor Window")) {
	    closeFrame();
	}
	else if (command.equals("Load From ...")) {
	    loadFrom();
	}
	else if (command.equals("Save As ...")) {
	    saveAs();
	}
	else if (command.equals("Send To ...")) {
	    sendTo();
	}
	else if (command.equals("Reset")) {
	    resetIfUserConfirms();
	}
	else if (command.equals("Close")) {
	    closeFrame();
	}
	else if (command.equals("Exit")) {
	    exitIfUserConfirms();
	}
	else if (command.equals("Check As Document")) {
	    editPanel.getDocument();
	}
	else if (command.equals("Check As Object")) {
	    editPanel.getObject();
	}
	else if (command.equals("To Object And Back")) {
	    editPanel.editObject(editPanel.getObject());
	}
	else if (command.equals("Find In Tree")) {
	    findInDocument();
	}
	else if (command.equals("As XML")) {
	    new TextViewFrame(editPanel.getDocument());
	}
	else if (command.equals("Namespaces")) {
	    namespaces.setVisible(true);
	}
	else if (command.equals("Help")) {
	    showHelp();
	}
	else
	    throw new ConsistencyException
		("Nothing to do for " + command);
    }

    protected void reset() {
	editPanel.reset();
    }

    protected void resetIfUserConfirms() {
	if (Util.dialogConfirms(null,
				"Are you sure you want to reset?"))
	    reset();
    }

    /**
     * Exit if this is the last visible frame for this frame's agent;
     * otherwise just become invisible.
     */
    protected void closeFrame() {
	if (isStandalone()) {
	    int numberVisible = frameGroup.getNumberVisible();
	    if (numberVisible == 1) {
		// If this is the last visible frame, treat a "close"
		// as a request to exit.
		exitIfUserConfirms();
		// If the user decides not to exit, don't close either.
		return;
	    }
	}

	// Plain close, rather than exit.

	// If this frame contains the most recently selected tree,
	// make the most recently selected tree null.
	XMLTreeEditPanel.EditorTree selected =
	    XMLTreeEditPanel.mostRecentlySelectedTree;
	if (selected == editPanel.docTree 
	        || selected == editPanel.templateTree) {
	    editPanel.setSelectedTree(null);
	}

	setVisible(false);
	// /\/: Check if any unsaved changes?
	// /\/: Dispose? -- what about message receiving frame?
    }

    public void exitIfUserConfirms() {
	if (Util.dialogConfirms(null,
				"Are you sure you want to exit?"))
	    System.exit(0);
    }

    protected void newEditorWindow() {
	String title = getBaseTitle() + ", Frame " + (1+frameGroup.getCount());
	XMLTreeEditFrame frame = new XMLTreeEditFrame(agent, title);
	frame.setVisible(true);
    }

    protected String getBaseTitle() {
	return frameGroup.getTitle();
    }

    protected void loadFrom() {
	Document doc = new XMLLoader(this).loadDocument();
	if (doc != null) {
	    editPanel.editDocument(doc);
	}
    }

    protected void saveAs() {
	Document doc = editPanel.getDocument();
	new XMLSaver(this).saveDocument(doc);
    }

    /*
     * Find
     */

    protected void findInDocument() {
	if (findFrame == null) {
	    findFrame = new FindFrame(getBaseTitle() + " Find");
	}
	findFrame.setVisible(true);
    }

    protected class FindFrame extends TextAreaFrame {
	FindFrame(String title) {
	    super(0, 0, title, new String[]{ "Find", "Cancel" });
	    addListener(new FindListener(this));
	}
	protected void finishFrame() {
	    // Doesn't make visible, unlike super method.
	    JFrame f = FindFrame.this.frame;	// /\/
	    f.pack();
	    f.setSize(400, 100);
	    f.validate();
	}
	public void setText(String text) {
	    super.setText(text);
	    setCaretPosition(0); // go to start of about text
	}
    }

    protected class FindListener implements TextAreaFrame.TListener {
	FindFrame frame;
	FindListener(FindFrame frame) {
	    this.frame = frame;
	}
	public void buttonPressed(String action) {
	    if (action.equals("Find")) {
		String text = frame.getText();
		Debug.noteln("Asking edit panel to find " +
			     Strings.quote(text));
		editPanel.findInTree(text);
	    }
	}
    }

    /*
     * Send
     */

    protected void sendTo() {
	if (sendFrame == null) {
	    sendFrame = new XMLTreeSendFrame(this, getBaseTitle() + " Send");
	}
	sendFrame.setEditFrame(this);
	sendFrame.setVisible(true);
    }

    /*
     * XML Text View
     */

    // /\/: Merge with the one in ItemEditor.
    // Or make TextAreaFrame more convenient.

    class TextViewFrame extends TextAreaFrame {
	TextViewFrame(Document doc) {
	    super(getBaseTitle() + " XML View",
		  new String[] {"Cancel"});
	    setText(XML.documentToXMLString(doc));
	    setVisible(true);
	}
	public void setText(String text) {
	    super.setText(text);
	    setCaretPosition(0); // scroll to beginning
	}
	public void whenClosed() {
	    frame.setVisible(false);
	    frame.dispose();
	}
    }

    /*
     * Help
     */

    protected void showHelp() {
	if (helpFrame == null) {
	    helpFrame = new TreeHelpFrame("Tree Editor Help");
	    helpFrame.setText(makeHelpText());
	}
	helpFrame.setVisible(true);
    }
	
    protected String makeHelpText() {
	return Strings.joinLines(sharedHelp) + Strings.joinLines(editHelp);
    }

    /** Help text shared by viewers and editors. */
    protected String[] sharedHelp = {
	"The left mouse button can be used to expand and",
	"contract nodes, and the right button produces a menu",
	"that can fully expand or contract the subtree rooted",
	"at the node, as well a performing other operations.",
	"",
	"The left button can also be used to select a node.",
	"When a tree node is selected, its text is displayed",
	"in a separate area at the bottom.  This makes it easier",
	"to read and edit long strings.",
	"",
	"Select \"Edit\" -> \"Namespaces\" from the menu bar",
	"to bring up a tool that allows you to define new",
	"namespaces and change color assignments.",
	"",
	"\"Edit\" -> \"Find In Tree\" can be used to find",
	"specified text, regardless of whether it appears in",
	"element tags, attributes, or element content."
    };

    /** Help text appended to {@link #sharedHelp} in editors. */
    protected String[] editHelp = {
	"", "",
        "Subtrees can be cut or copied using the right button.",
	"This produces a small frame containing the subtree.",
	"It can then be pasted into any editor frame by",
	"selecting a node and then using one of the \"Insert ...\"",
	"buttons at the bottom of the little frame.",
	"",
	"In some editor frames, \"Cut\" and \"Copy\" are also",
	"provided by buttons at the bottom of the frame.",
	"",
	"Templates representing common object types and",
	"suitable for copying and pasting can be found on the",
	"right of the vertical divider."
    };

    // /\/: This class should not have to exist.
    protected static class TreeHelpFrame extends TextAreaFrame {
	TreeHelpFrame(String title) {
	    super(0, 0, title, new String[]{"Cancel"});
	    setEditable(false);
	}
	protected void finishFrame() {
	    // Doesn't make visible, unlike super method.
	    JFrame f = TreeHelpFrame.this.frame;	// /\/
	    f.pack();
	    f.setSize(375, 400); // width, height
	    f.validate();
	}
	public void setText(String text) {
	    super.setText(text);
	    setCaretPosition(0); // go to start of about text
	}
    }

}

// Issues:
// * Some static fields should be per-agent instead.
// * FindFrame should (probably?) be per frame-group.
