/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Feb 24 13:20:43 2008 by Jeff Dalton
 * Copyright: (c) 2002 - 2008, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.io.*;
import java.util.*;

// JDOM imports
import org.jdom.*;

import ix.util.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

/**
 * Describes the (XML) syntax of I-X data objects as an XML schema.
 */
public class XMLSchemaSyntax extends XMLSyntax {

    // /\/: Dubious state variable
    Schema theSchema = null;

    public static final Namespace schemaNamespace
	= Namespace.getNamespace("xsd", "http://www.w3.org/2001/XMLSchema");

    public XMLSchemaSyntax() {
	super();
    }

    public XMLSchemaSyntax(XMLTranslator xmlt) {
	super(xmlt);
    }

    public Schema makeSchema(Class rootClass) {
	return makeSchema
	    (rootClass,
	     XML.config().xmlSyntaxClasses(classSyntax, rootClass));
    }

    Schema makeSchema(Class rootClass, List relevantClasses) {
	Schema schema = new Schema(rootClass, new String[][] {
	    {"targetNamespace", xmlt.getHomeNamespace().getURI()},
	    {"elementFormDefault", "qualified"}
	});

	theSchema = schema;	// /\/

	inheritance = new InheritanceTree(relevantClasses);

	// /\/: Modifies the value in the schema.
	collectListofClasses(relevantClasses, schema.getListofClasses());

	// We make the object namespace both the default and the target.
	// If we didn't make it the default, we'd have to write
	// things like type="ix:typename".

	// /\/: But we can't do it here, because of a bug in JDOM 1.0.
	// We would get:
	// Exception in thread "main" org.jdom.IllegalAddException:
	// The namespace xmlns="http://www.aiai.ed.ac.uk/project/ix/"
	// could not be added as a namespace to "schema": The namespace
	// prefix "" collides with the element namespace prefix
        //     at org.jdom.Element.addNamespaceDeclaration(Element.java:341)
	//     ...

	// schema.addNamespaceDeclaration(xmlt.getHomeNamespace());

	// Add top-level types and roots for substitutionGroups.
	addRootSyntax(schema);

	// Add general List and Map syntax
	addFrameworkDeclarations(schema, List.class);
	schema.addContent(makeListType());

	addFrameworkDeclarations(schema, Map.class);
	schema.addContent(makeMapType());

	// Add information from each class.
	for (Iterator i = relevantClasses.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    addClassSyntax(schema, c);
	}

	return schema;
    }

    void addRootSyntax(Schema schema) {
	// We add an "OBJECT" element, for use as a substitution
	// group.  For an element to be in the group, its type
	// must be a subtype.
	// We don't have an "object" element and "object" type
	// because we don't expect direct instances of Object.
	// But we do have an "object_as_element" type
	Element objectElt = makeXsdElement("element", new String[][] {
	    {"name", "OBJECT"},
//	    {"type", "xsd:anyType"},
	    {"abstract", "true"}
	});
	schema.addContent(new Comment("Prelude"));
	schema.addContent(objectElt);
	schema.addContent(makeClassAsElementType(getClassDescr(Object.class)));
    }

    void addClassSyntax(Schema schema, Class c) {
	ClassDescr cd = getClassDescr(c);
	schema.addContent(new Comment(cd.getExternalName()));
	if (cd.isStruct())
	    addStructSyntax(schema, c);
	else if (cd.isEnumeration())
	    addEnumerationSyntax(schema, c);
	else if (cd.isPrimitive())
	    addPrimitiveSyntax(schema, c);
	else if (cd.isXML())
	    addLiteralDocumentSyntax(schema, c);
	else
	    schema.addContent(new Comment("No syntax for " + c));
	// list-of element and type if needed
	if (schema.getListofClasses().contains(c)) {
	    schema.addContent(makeListofClassAsElementType(c));
	    schema.addContent(makeListofClassType(c));
	}
    }

    void addFrameworkDeclarations(Schema schema, Class c) {
	ClassDescr cd = getClassDescr(c);
	// Upper-case element declaration.
	schema.addContent(makeUpperCaseClassElement(cd));
	// Lower-case element declaration.
	if (cd.isAbstract() && !cd.isInterface()) {
	    // /\/: The !cd.isInterface() part is because classes
	    // such as List count as abstract, but we do want the
	    // lower-case declarations in that case.
	    schema.addContent
		(new Comment
		 ("Since " + getElementName(cd) + " is abstract, " +
		  "no " + Strings.quote(getElementName(cd)) +
		  " element or type declaration is needed."));
	}
	else {
	    schema.addContent(makeLowerCaseClassElement(cd));
	}
	// As-element type declaration.
	schema.addContent(makeClassAsElementType(cd));
    }

    Element makeUpperCaseClassElement(ClassDescr cd) {
	Class sup = inheritance.getSuperclass(cd.theClass);
	String substGroup = sup == null
	    ? "OBJECT"
	    : getClassDescr(sup).getUpperName();
	return makeXsdElement("element", new String[][] {
	    {"name", cd.getUpperName()},
	    {"abstract", "true"},
	    // No type, hence xsd:anyType
	    {"substitutionGroup", substGroup}
	});
    }

    Element makeLowerCaseClassElement(ClassDescr cd) {
	String className = cd.getExternalName();
	return makeXsdElement("element", new String[][] {
	    {"name", className},
	    {"type", className},
	    {"substitutionGroup", cd.getUpperName()}
	});
    }

    Element makeClassAsElementType(ClassDescr cd) {
	String typeName = asElementTypeName(cd.theClass);
	Element elementRef = makeXsdElement("element")
	    .setAttribute("ref", cd.getUpperName());
	return makeXsdElement("complexType")
	    .setAttribute("name", typeName)
	    .addContent(makeXsdElement("sequence")
			.addContent(elementRef));
    }

    String asElementTypeName(Class c) {
	// /\/: Used to use getElementName(c) here, but since
	// an element for a subclass can appear, the upper-case
	// name seems more appropriate.  Indeed, the ref produced
	// by makeClassAsElementType(ClassDescr) was always to
	// the upper-case element.
	return getUpperName(c) + "-as-element";
    }


    /*
     * Structs
     */

    void addStructSyntax(Schema schema, Class c) {
	addFrameworkDeclarations(schema, c);
	if (!getClassDescr(c).isAbstract())
	    schema.addContent(makeStructType(c));
    }

    Element makeStructType(Class c) {
	return makeXsdElement("complexType")
	    .setAttribute("name", getElementName(c))
	    .setContent(makeStructContents(c));
    }

    List makeStructContents(Class c) {
	ClassDescr cd = getClassDescr(c);
	List fields = cd.getFieldDescrs();
	List attrFields = attributeFields(fields);
	List eltFields = elementFields(fields);
	Debug.expect(fields.size() == attrFields.size() + eltFields.size());

	List fieldElts = makeStructFieldElements(eltFields);
	List attrElts = makeStructAttributeElements(attrFields);

	Element bodyElt = makeXsdElement("all")
	    .setContent(fieldElts);

	List result = new LinkedList();
	result.add(bodyElt);
	result.addAll(attrElts);

	return result;
    }

    List makeStructAttributeElements(List fields) {
	return (List)Collect.map(fields, new Function1() {
	    public Object funcall(Object fd) {
		return makeStructAttributeElement((FieldDescr)fd);
	    }
	});
    }

    Element makeStructAttributeElement(FieldDescr fd) {
	return makeXsdElement("attribute", new String[][] {
	    {"name", fd.getExternalName()},
	    {"type", getElementName(fd.getType())}
	});
    }

    List makeStructFieldElements(List fields) {
	return (List)Collect.map(fields, new Function1() {
	    public Object funcall(Object fd) {
		return makeStructFieldElement((FieldDescr)fd);
	    }
	});
    }

    Element makeStructFieldElement(FieldDescr fd) {
	return makeXsdElement("element")
	    .setAttribute("name", fd.getExternalName())
	    .setAttribute("type", getStructFieldValueType(fd))
	    .setAttribute("minOccurs", "0");
    }

    String getStructFieldValueType(FieldDescr fd) {
	ClassDescr valueDescr = fd.getTypeDescr();
	if (valueDescr.isList()) {
	    return getListValueType(valueDescr);
	}
	else if (valueDescr.isSet()) {
	    Debug.expect(false, "Can't handle Set values");
	    return null;
	}
	else if (valueDescr.isMap()) {
	    return getMapValueType(valueDescr);
	}
	else {
	    return getPlainValueType(valueDescr);
	}
    }

    String getPlainValueType(ClassDescr value) {
	return asElementTypeName(value.theClass);
    }

    String getListValueType(ClassDescr value) {
	ClassDescr eltType = value.getEltType();
	if (eltType == null)	// type is unknown so assume Object
	    return asElementTypeName(List.class);
	else {
	    // ensureListofClassTypes(eltType.theClass);
	    return ListofClassAsElementTypeName(eltType.theClass);
	}
    }

    String getMapValueType(ClassDescr value) {
	// /\/: Handles only Object -> Object Maps.
	return asElementTypeName(Map.class);
    }


    /*
     * List of Class
     */

    void ensureListofClassTypes(Class c) {
	String probe = ListofClassTypeName(c);
	Element exists = theSchema.getTypeDefinition(probe);
	if (exists != null)
	    return;
	theSchema.addContent(makeListofClassAsElementType(c));
	theSchema.addContent(makeListofClassType(c));
    }

    Element makeListofClassAsElementType(Class c) {
	return makeXsdElement("complexType")
	    .setAttribute("name", ListofClassAsElementTypeName(c))
	    .addContent
	        (makeXsdElement("sequence")
		 .addContent(makeXsdElement("element")
			     .setAttribute("name", "list")
			     .setAttribute("type", ListofClassTypeName(c))));
    }

    String ListofClassAsElementTypeName(Class eltClass) {
	// /\/: Used to use getElementName(eltClass) here, but
	// it should be consistent with ListofClassTypeName(Class).
	return "list-of-" + getUpperName(eltClass) + "-as-element";
    }

    Element makeListofClassType(Class c) {
	return makeXsdElement("complexType")
	    .setAttribute("name", ListofClassTypeName(c))
	    .addContent(makeSequenceOfClass(c));
    }

    String ListofClassTypeName(Class c) {
	// /\/: Used to use getElementName(c) here, but since
	// an element for a subclass can appear, the upper-case
	// name seems more appropriate.  Indeed, the ref produced
	// by makeSequenceOf(Class) was always to the upper-case element.
	return "list-of-" + getUpperName(c);
    }

    Element makeSequenceOfClass(Class c) {
	return makeSequenceOfElement
	    (makeXsdElement("element")
	     .setAttribute("ref", getClassDescr(c).getUpperName()));

    }

    Element makeSequenceOfElement(Element elt) {
	return makeXsdElement("sequence")
	    .addContent(elt
			.setAttribute("minOccurs", "0")
			.setAttribute("maxOccurs", "unbounded"));
    }

    Element makeListType() {	// general List of Object
	// /\/: Could perhaps have name list-of-object.
	return makeXsdElement("complexType")
	    .setAttribute("name", "list")
	    .addContent(makeSequenceOfClass(Object.class));
    }


    /*
     * Set of Class
     */

    // ... Parallels List of Class ...


    /*
     * Object -> Object Map
     */

    // /\/: We don't yet handle restricted Maps where the kays or
    // values are known to be instances of a class more specific
    // than Object.

    Element makeMapType() {
	return makeXsdElement("complexType")
	    .setAttribute("name", "map")
	    .addContent(makeSequenceOfElement(makeMapEntryElement()));
    }

    Element makeMapEntryElement() {
	return makeXsdElement("element")
	    .setAttribute("name", "map-entry")
	    .addContent
	        (makeXsdElement("complexType")
		 .addContent(makeXsdElement("sequence")
			     .addContent(makeObjectTypeElement("key"))
			     .addContent(makeObjectTypeElement("value"))));
    }

    Element makeObjectTypeElement(String name) {
	return makeXsdElement("element")
	    .setAttribute("name", name)
	    .setAttribute("type", asElementTypeName(Object.class));
    }


    /*
     * Enumerations
     */

    void addEnumerationSyntax(Schema schema, Class c) {
	addFrameworkDeclarations(schema, c);
	schema.addContent(makeEnumerationType(c));
    }

    Element makeEnumerationType(Class c) {
	return makeXsdElement("simpleType")
	    .setAttribute("name", getElementName(c))
	    .addContent(makeXsdElement("restriction")
			.setAttribute("base", "xsd:string")
			.setContent(makeEnumerationValueElements(c)));
    }

    List makeEnumerationValueElements(Class c) {
	List elts = new LinkedList();
	for (Iterator i = getEnumerationValues(c).iterator(); i.hasNext();) {
	    elts.add(makeXsdElement("enumeration", new String[][] {
		{"value", i.next().toString()}
	    }));
	}
	return elts;
    }

    /*
     * Literal Documents (embedded XML)
     */

    void addLiteralDocumentSyntax(Schema schema, Class c) {
	addFrameworkDeclarations(schema, c);
	schema.addContent(makeLiteralDocumentType(c));
    }

    Element makeLiteralDocumentType(Class c) {
	ClassDescr cd = getClassDescr(c);
	return makeXsdElement("complexType")
	    .setAttribute("name", getElementName(cd))
	    .addContent(makeXsdElement("sequence")
			.addContent(makeXsdElement("any")
				    // By default has namespace="##any"
				    .setAttribute("processContents",
						  "lax")));
    }

    /*
     * Primitive types
     */

    void addPrimitiveSyntax(Schema schema, Class c) {
	addFrameworkDeclarations(schema, c);
	schema.addContent(makePrimitiveType(c));
    }

    Element makePrimitiveType(Class c) {
	String type = "xsd:" + getSimpleType(c);
	return makeXsdElement("simpleType")
	    .setAttribute("name", getElementName(c))
	    .addContent(makeXsdElement("restriction")
			.setAttribute("base", type));
    }

    /** Maps classes to simple schema types. */
    static Object[][] simpleTypeTable = {
	{String.class,    "string"},
	{Symbol.class,    "string"},
	{ItemVar.class,   "string"},
	{Name.class,      "string"},
	{Byte.class,      null},
	{Character.class, null},
	{Short.class,     "short"},
	{Integer.class,   "int"},
	{Long.class,      "long"},
	{Float.class,     "float"},
	{Double.class,    "double"},
	{Boolean.class,   null}
    };

    /** Maps classes to simple schema types. */
    public static String getSimpleType(Class c) {
	String type = null;
	for (int i = 0; i < simpleTypeTable.length; i++) {
	    if (simpleTypeTable[i][0] == c) {
		type = (String)simpleTypeTable[i][1];
		break;
	    }
	}
	if (type != null)
	    return type;
	else
	    throw new IllegalArgumentException
		("Can't find XML Schema datatype for " + c);
    }

    /*
     * Some utilities.
     */

    public Element makeElement(String name, Namespace namespace,
			       String[][] attributes) {
	Element elt = new ExtendedElement(name, namespace);
	XML.setAttributes(elt, attributes);
	return elt;
    }

    public Element makeXsdElement(String name, String[][] attributes) {
	return makeElement(name, schemaNamespace, attributes);
    }

    public Element makeXsdElement(String name) {
	return new ExtendedElement(name, schemaNamespace);
    }

    static class ExtendedElement extends Element {
	ExtendedElement(String name) {
	    super(name);
	}
	ExtendedElement(String name, Namespace namespace) {
	    super(name, namespace);
	}
//  	ExtendedElement addChildren(List elements) {
//  	    for (Iterator i = elements.iterator(); i.hasNext();) {
//  		Element elt = (Element)i.next();
//  		addContent(elt);
//  	    }
//  	    return this;
//  	}
	public String toString() {
	    String name = getAttributeValue("name");
	    if (name != null)
		return super.toString() + ", name attribute = " + name;
	    else
		return super.toString();
	}
    }

    /** An XML schema. */
    public static class Schema extends ExtendedElement {
	Class rootClass;
	Set listofClasses = new HashSet();

	/** Create a schema with specified root class and attributes. */
	public Schema(Class rootClass, String[][] attributes) {
	    super("schema", schemaNamespace);
	    this.rootClass = rootClass;
	    // /\/: Have to add the default namespace before adding
	    // attributes because of a bug in JDOM 1.0.  See the related
	    // comment above.
	    this.addNamespaceDeclaration(XML.config().getHomeNamespace());
	    XML.setAttributes(this, attributes);
	}

	/** What this schema is about (domain, plan, or all objects). */
	public Class getRootClass() { return rootClass; }

	/** List of classes that require list-of types. */
	Set getListofClasses() { return listofClasses; }

	/** Finds a definition in the existing top-level content. */
	public Element getTypeDefinition(String typeName) {
	    for (Iterator i = getChildren().iterator(); i.hasNext();) {
		Element child = (Element)i.next();
		if (child.getName().endsWith("Type")
		      && child.getAttributeValue("name").equals(typeName))
		    return child;
	    }
	    return null;
	}
    }


    /**
     * Outputs a schema for I-X plans or for the class specified
     * by the "root" parameter.  
     *
     * @see ix.util.Parameters#getParameter(String)
     */
    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);

	XMLSchemaSyntax syntax = new XMLSchemaSyntax();
	Class root = syntax.classSyntax.
	    classForExternalName(Parameters.getParameter("root", "object"));
	Schema schema = syntax.makeSchema(root);
	Document doc = new Document(schema);
	XML.printXMLWithWhitespace(doc, 1);
    }

}
