/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Mar 18 02:55:46 2006 by Jeff Dalton
 * Copyright: (c) 2002 - 2006, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.lang.reflect.*;
import java.util.*;

import java.io.PrintStream;	// for testing

import java.awt.Color;		// to define a stringer

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;

import ix.util.*;
import ix.util.xml.XML;
import ix.util.xml.XMLSchemaSyntax;
import ix.util.lisp.*;

/**
 * Provides a syntax for objects by acting as a factory for class and
 * field descriptions.
 *
 * <p>For a class with fields, ClassSyntax provides the information
 * needed to process instances field-by-field.  It also embodies
 * decisions about which fields should be accessible in this way,
 * how their values are obtained or set, and the order in which
 * they should be visited.
 *
 * <p>A {@link ClassFinder} is used to map both ways between
 * external names and classes and to map between Java and external
 * names for classes and fields.
 *
 * @see ClassDescr
 * @see FieldDescr
 * @see ClassFinder
 */

public class ClassSyntax {

    protected ClassFinder classFinder;

    protected Map classToDescdCache = new HashMap();

    protected Map classToStringerMap = new HashMap();

    protected boolean inferElementClasses = false;

    public ClassSyntax() {
	this(XML.config().defaultClassFinder());
    }

    public ClassSyntax(ClassFinder finder) {
	Debug.expect(finder != null, "Null ClassFinder");
	this.classFinder = finder;
	initFieldCases();
	initStringConversions();
    }

    public ClassFinder getClassFinder() {
	return classFinder;
    }

    /**
     * Sets whether this ClassSyntax is allowed to infer List and Set
     * element classes from field names.  For example, if an object has
     * a List field named "refinements", the list's elements might all
     * be instances of a class named "Refinement".  This is not necessary,
     * for Lists, when {@link TypedList}s are used.
     */
    public void setInferElementClasses(boolean v) {
	this.inferElementClasses = v;
    }


    /*
     * External Names
     */

    // /\/: Maybe some of these should not be public so that things
    // have to go via the ClassDescr.

    public String externalNameForClass(Class c) {
	return classFinder.nameForClass(c);
    }

    public Class classForExternalName(String externalName) {
	return classFinder.classForName(externalName);
    }

    public String externalNameForField(String javaName) {
	return classFinder.externalFieldName(javaName);
    }

    public String upperNameForClass(Class c) {
	// /\/: If the external names are the same as the Java names,
	// the upper name loses too much information.  For example,
	// NodeEndRef becomes NODEENDREF.
	String externalName = externalNameForClass(c);
	return externalName.toUpperCase();
    }


    /*
     * Making ClassDescrs
     */

    // /\/: Note that only the one-argument case is made unique.
    // It's possible that we should have a separate class called
    // ValueDescr for field values so that only 1-arg ClassDescrs
    // are used.

    public ClassDescr getClassDescr(Class c) {
	ClassDescr cd = (ClassDescr)classToDescdCache.get(c);
	if (cd == null) {
	    cd = makeClassDescr(c);
	    classToDescdCache.put(c, cd);
	}
	return cd;
    }

    protected ClassDescr makeClassDescr(Class c) {
	if (TypedList.class.isAssignableFrom(c))
	    return new ClassDescr(this, c, ListOf.elementClass(c));
	else
	    return new ClassDescr(this, c);
    }

    public ClassDescr makeClassDescr(Class collectionClass,
				     Class eltClass) {
	return new ClassDescr(this, collectionClass, eltClass);
    }

    public ClassDescr makeClassDescr(Class mapClass,
				     Class keyClass, Class valClass) {
	return new ClassDescr(this, mapClass, keyClass, valClass);
    }


    /*
     * String representations
     */

    public String xmlSchemaDatatype(Class c) {
	// First try the Stringer, if one exists.
	Stringer s = getStringer(c);
	if (s != null) {
	    String result = s.xmlSchemaDatatype();
	    if (result != null)
		return result;
	}
	// Try the built-in mapping.
	return XMLSchemaSyntax.getSimpleType(c); // may throw IllegalArg
    }

    public Stringer getStringer(Class c) {
	return (Stringer)classToStringerMap.get(c);
    }
    protected void setStringer(Class c, Stringer s) {
	classToStringerMap.put(c, s);
    }

    protected void initStringConversions() {
	// Date
	setStringer(Date.class, new Stringer() {
	    ISODateFormat format = new ISODateFormat();
	    public String toString(Object obj) {
		return format.formatDateTime((Date)obj);
	    }
	    public Object fromString(String text) {
		return format.parseDateTime(text);
	    }
	    public String xmlSchemaDatatype() {
		return "dateTime";
	    }
	});
	// Duration
	setStringer(Duration.class, new Stringer() {
	    public String toString(Object obj) {
		return ((Duration)obj).toISOString();
	    }
	    public Object fromString(String text) {
		return new Duration(text);
	    }
	    public String xmlSchemaDatatype() {
		return "duration";
	    }
	});
	// Color
	setStringer(Color.class, new Stringer() {
	    public String toString(Object obj) {
		return Integer.toString(((Color)obj).getRGB());
	    }
	    public Object fromString(String text) {
		try {
		    return Color.decode(text);
		}
		catch (NumberFormatException e) {
		    throw new RethrownException(e);
		}
	    }
	    public String xmlSchemaDatatype() {
		return "int"; // /\/ ???
	    }
	});
    }

    /*
     * Finding relevant classes
     */

    /**
     * Returns a recursively composed list of classes that are related
     * to the specified class by being the types of fields, etc.
     */
    public List relevantClasses(Class c) {
	return relevantClasses(Lisp.list(c));
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified array by being the types of
     * fields, etc.
     */
    public List relevantClasses(Class[] topClasses) {
	return relevantClasses(Arrays.asList(topClasses));
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified list by being the types of
     * fields, etc.
     */
    public List relevantClasses(List topClasses) {
	List relevantClasses = new LinkedList();
	for (Iterator i = topClasses.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    collectRelevantClasses(c, relevantClasses, Lisp.NIL);
	}
	relevantClasses.remove(Object.class);	// /\/
	return relevantClasses;
    }

    /**
     * Returns a recursively composed list of classes that are related
     * to the classes in the specified list by being the types of
     * fields, etc.  This method retains the order of the original
     * list, but does preorder expansion of elements to include
     * any relevant classes not already there.
     */
    public List expandRelevantClasses(List topClasses) {
	// Debug.noteln("Expanding relevant classes in", topClasses);
	List collect = new LinkedList();
	for (LList tail = LList.newLList(topClasses);
	     !tail.isEmpty(); tail = tail.cdr()) {
	    Class c = (Class)tail.car();
	    Debug.expect(!collect.contains(c));
	    collectRelevantClasses(c, collect, tail.cdr());
	}
	collect.remove(Object.class); 		// /\/
	// Debug.noteln("Expanded to", collect);
	return collect;
    }

    protected void collectRelevantClasses(Class c, List result, List tail) {
	if (c == null || result.contains(c) || tail.contains(c))
	    return;
	result.add(c);

	ClassDescr cd = getClassDescr(c);
	if (cd.isStruct()) {
	    for (Iterator fi = cd.getFieldDescrs().iterator(); fi.hasNext();) {
		FieldDescr fd = (FieldDescr)fi.next();
		collectRelevantClasses(fd.getTypeDescr(), result, tail);
	    }
	}
    }

    protected void collectRelevantClasses
	           (ClassDescr ftype, List result, List tail) {
	if (ftype == null)
	    return;
	else if (ftype.isCollection())
	    collectRelevantClasses(ftype.getEltType(), result, tail);
	else if (ftype.isMap()) {
	    collectRelevantClasses(ftype.getKeyType(), result, tail);
	    collectRelevantClasses(ftype.getValType(), result, tail);
	}
	else
	    collectRelevantClasses(ftype.getDescribedClass(), result, tail);
    }


    /*
     * Making FieldDescrs
     */

    FieldMap makeFieldDescrs(Class c) { // deliberately not public
	return collectFieldInfo(c);
    }

    protected FieldDescr makeFieldDescr(String name, Class type) {
	return new FieldDescr(this, name, type);
    }

    protected TwoKeyHashMap fieldCaseMap =
	// Maps fieldName, containingClass -> ClassDescr
	// e.g. if the field contains the List of C, for
	// some class C, the descriptor will have C as its eltType.
	new TwoKeyHashMap();

    /** Initialize table of field special cases. */
    protected void initFieldCases() {
//  	setFieldCase(Refinement.class, "nodes",
//  		     makeClassDescr(List.class, NodeSpec.class));

//  	setFieldCase(Plan.class, "worldState",
//  		     makeClassDescr(List.class, PatternAssignment.class));

//  	setFieldCase(Plan.class, "constraints",
//  		     makeClassDescr(List.class, Constrainer.class));

//  	setFieldCase(PlanRefinement.class, "constraints",
//  		     makeClassDescr(List.class, Constrainer.class));

//  	// /\/: Want to avoid trying to find a class "Arg" in the applet.
//  	setFieldCase(ix.test.applet.AppletMessage.class, "args",
//  		     makeClassDescr(List.class));

//  	// /\/: Want to avoid trying to find a "Parameter" class.
//  	setFieldCase(Constraint.class, "parameters",
//  		     makeClassDescr(List.class));

    }

    public ClassDescr getFieldCase(Class aClass, String fieldName) {
	return (ClassDescr)fieldCaseMap.get(aClass, fieldName);
    }

    public void setFieldCase(Class aClass, String fieldName,
			     ClassDescr descr) {
	fieldCaseMap.put(aClass, fieldName, descr);
    }


    /*
     * Determining the visible fields of a class
     */

    protected FieldMap collectFieldInfo(Class objClass) {
	// Get a preliminaty list based on the field declarations
	// Entires will be removed if suitable "get" and "set"
	// methods cannot be found.
	FieldMap fields = collectFields(objClass, new FieldMap());

	// Look at the methods, starting with objClass and then
	// trying successive superclasses, taking the first
	// suitable "get" and "set" methods for each field.
	for (Class atClass = objClass; atClass != null;) {

	    // Look at the declared methods of the class
	    // Method[] methods = atClass.getDeclaredMethods();
	    Method[] methods = getDeclaredMethods(atClass);
	    for (int i = 0; i < methods.length; i++) {
		Method meth = methods[i];
		String methName = meth.getName();
		if (isGetName(methName))
		    tryGetMethod(meth, fields);
		else if (isSetName(methName)) {
		    trySetMethod(meth, fields);
		}
	    }

	    // Look at superclass methods
	    atClass = atClass.getSuperclass();
	}

	// Remove fields that don't have both "get" and "set",
	// and set the typeDescr of ones that do.
	// N.B. have to iterate over a copy because we remove as we go.
	for (Iterator i = LList.newLList(fields.getFields()).iterator();
	     i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    if (fd.getter == null || fd.setter == null)
		fields.remove(fd);
	    else
		fd.typeDescr = makeFieldTypeDescr(fd, objClass);
	}

	return fields;

    }

    /**
     * Returns the class's declared methods, or an empty array if
     * there's a security exception (as there might be in an applet).
     */
    protected Method[] getDeclaredMethods(Class c) {
	try {
	    return c.getDeclaredMethods();
	}
	catch (SecurityException e) {
	    Debug.noteln("Problem getting methods of " + c);
	    Debug.noteException(e, false);
	    return new Method[]{};
	}
    }

    /**
     * Constructs a {@link ClassDescr} that describes the value of
     * a field.
     *
     * @see #getFieldCase(Class, String)
     * @see #setFieldCase(Class, String, ClassDescr)
     */
    protected ClassDescr makeFieldTypeDescr(FieldDescr fd, Class fromClass) {
	String fieldName = fd.getName();
	Class fieldClass = fd.getType();

	// If fieldClass is a List or Map, we'd like to know
	// the content types (elt, or key + val).

	// Perhaps it's a known special case.
	// /\/: Perhaps we should also have a map from fieldClass.
	ClassDescr cd = getFieldCase(fromClass, fieldName);
	if (cd != null)
	    return cd;

	// For a TypedList class, we can just ask for its element class.
	if (TypedList.class.isAssignableFrom(fieldClass))
	    return makeClassDescr(fieldClass);

	cd = getClassDescr(fieldClass);

	// For a List or Set, we may be able to figure it out from
	// the fieldName.  For a List-valued field named "refinements",
	// for example, the element class might be Refinement.
	if (inferElementClasses
              && (cd.isList() || cd.isSet())
	      && fieldName.length() >= 2
	      && fieldName.endsWith("s")) {
	    String className = 
		Strings.capitalize
		    (fieldName.substring(0, fieldName.length() - 1));
	    // Find class from "internal" (Java) name, perhaps converting
	    // converting to external name to check imports.
	    Class c = classIfExists(fromClass, className);
	    if (c != null)
		cd = makeClassDescr(fieldClass, c);
	}

	// Return whatever we've managed to get
	return cd;
    }

    protected Class classIfExists(Class fromClass, String className) {
	// We're trying to find the element class of a
	// collection-valued field.
	// fromClass is the class that contained the field.
	// className is a name derived from the field name.
	// If a class of that name exists, we'll take it as the
	// element class.
	// The reason we have fromClass is that the element
	// class will often be in the same package.
	Package pack = fromClass.getPackage();
	String candidate = pack.getName() + "." + className;
	// Try the class in fromClass's package.  The ClassFinder
	// tryClassForName method expects an "internal" (Java syntax)
	// name.
	Class eltClass = classFinder.tryClassForName(candidate);
	if (eltClass != null)
	    return eltClass;
	else
	    // That didn't work, so try giving the ClassFinder the
	    // external name to see if it can be found from the
	    // imports.  (The ClassFinder will convert it back to
	    // a Java name at some point, but we don't have access
	    // to that point directly.)
	    return classFinder.classForName
		     (classFinder.externalName(className));
    }

    protected void tryGetMethod(Method meth, FieldMap fields) {
	Class[] ptypes = meth.getParameterTypes();
	int mod = meth.getModifiers();
	// Must be a public, 0-argument method
	if (!(Modifier.isPublic(mod) && ptypes.length == 0))
	    return;
	// Find corresponding field, if any
	String fieldName = fieldNameFromGetName(meth.getName());
	FieldDescr fd = fields.fieldForName(fieldName);
	// The field must exist and not already have a "get" method ...
	if (fd != null && fd.getter == null) {
	    // ... and this method must return the right type
	    Class fieldType = fd.getType();
	    if (meth.getReturnType() == fieldType)
		fd.getter = meth;
	}
    }

    protected void trySetMethod(Method meth, FieldMap fields) {
	Class[] ptypes = meth.getParameterTypes();
	int mod = meth.getModifiers();
	// Must be a public, 1-argument method
	if (!(Modifier.isPublic(mod) && ptypes.length == 1))
	    return;
	// Find corresponding field, if any
	String fieldName = fieldNameFromSetName(meth.getName());
	FieldDescr fd = fields.fieldForName(fieldName);
	// The field must exist and not already have a "set" method ...
	if (fd != null && fd.setter == null) {
	    // ... and this method must have an argument of the right type
	    Class fieldType = fd.getType();
	    // /\/: Must return-type be void?
	    if (ptypes[0] == fieldType)
		fd.setter = meth;
	}
    }

    public static final boolean isGetName(String name) {
	return name.length() > 3 && name.startsWith("get");
    }

    public static final boolean isSetName(String name) {
	return name.length() > 3 && name.startsWith("set");
    }

    public static final String fieldNameFromGetName(String name) {
	return Strings.uncapitalize(name.substring(3));
    }

    public static final String fieldNameFromSetName(String name) {
	return Strings.uncapitalize(name.substring(3));
    }

    /** Returns an initial List of {@link FieldDescr}s. */
    protected FieldMap collectFields(Class c, FieldMap fields) {

	// We look at fields in a class before looking at
	// its superclass, and keep track of names we've
	// already seem, so that a class's fields will hide
	// any inherited fields of the same name.  That ensures
	// the the most "local" field's type will be used.

	// We don't process interfaces, because they can provide
	// only static finals.

	// Non-inherited fields from this class.
	// Field[] localf = c.getDeclaredFields();
	Field[] localf = getDeclaredFields(c);
	for (int i = 0; i < localf.length; i++) {
	    if (isWanted(localf[i])) {
		String name = localf[i].getName();
		Class type = localf[i].getType();
		if (fields.fieldForName(name) == null) {
		    fields.add(makeFieldDescr(name, type));
		}
	    }
	}

	// Recursively process superclass, if it exists
	Class sup = c.getSuperclass();
	if (sup != null)
	    collectFields(sup, fields);

	return fields;
    }

    /**
     * Returns the class's declared fields, or an empty array if
     * there's a security exception (as there might be in an applet).
     */
    protected Field[] getDeclaredFields(Class c) {
	try {
	    return c.getDeclaredFields();
	}
	catch (SecurityException e) {
	    Debug.noteln("Problem getting fields of " + c);
	    Debug.noteException(e, false);
	    return new Field[]{};
	}
    }

    protected boolean isWanted(Field f) {
	int m = f.getModifiers();
	return !(Modifier.isFinal(m) || Modifier.isStatic(m) ||
		 Modifier.isTransient(m));
    }


    /*
     * Testing
     */

    /**
     * Prints a description of a class on <code>System.out</code>.
     */
    public void describeClass(Class c) {
	describeClass(c, System.out);
    }

    /**
     * Prints a description of a class on the designated PrintStream.
     */
    public void describeClass(Class c, PrintStream out) {
	ClassDescr cd = getClassDescr(c);
	List fields = cd.getFieldDescrs();

	out.println(cd.toString());
	out.println("Java name: " + cd.theClass.getName());
	out.println("External name: " + cd.getExternalName());
	out.println(fields.isEmpty() ? "No fields" : "Fields:");
	for (Iterator i = fields.iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    out.println(fd.getName() + ": " + fd.getTypeDescr());
	    out.println("   getter: " + fd.getter);
	    out.println("   setter: " + fd.setter);
	}
    }

    /**
     * Test loop that repeatedly asks the user for a class name
     * and prints a description.
     */
    public static void main(String[] argv) {
	ClassSyntax syntax = new ClassSyntax();
	for (;;) {
	    String in = Util.askLine("Class name:");
	    if (in.equals("bye"))
		return;

	    Class c = syntax.classFinder.classForName(in);
	    System.out.println(c);
	    if (c == null) continue;

	    syntax.describeClass(c, System.out);
	    System.out.println("");
	}
    }

}
