/* File: SimpleMatcher.java
 * Contains: A simple pattern-matcher
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Wed Feb 22 01:47:21 2006 by Jeff Dalton
 * Copyright: (c) 1998, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * A simple pattern-matcher.
 */

public final class SimpleMatcher {

    /**
     * emptyEnv is used to return non-null without allocating an env.
     */
    public static final MatchEnv emptyEnv = new MatchEnv() {
	public Object put(Object key, Object Value) {
	    Debug.expect(false, "emptyEnv can't be modified");
	    return null;
	}
    };

    public static final Symbol REST = Symbol.intern("&rest");

    public static final 
    MatchEnv mustMatch(Object pat, Object dat) {
	return mustMatch(pat, dat, null);
    }

    public static final
    MatchEnv mustMatch(Object pat, Object dat, MatchEnv env) {
	MatchEnv result = match(pat, dat, env);
	Debug.expect(result != null, "Data " + dat + " failed to match", pat);
	return result;
    }

    public static final
    MatchEnv matchAnyPattern(List patterns, Object data) {
	for (Iterator i = patterns.iterator(); i.hasNext();) {
	    MatchEnv result = match(i.next(), data);
	    if (result != null)
		return result;
	}
	return null;
    }

    public static final
    MatchEnv matchAnyValue(Object pat, List dataValues) {
	for (Iterator i = dataValues.iterator(); i.hasNext();) {
	    MatchEnv result = match(pat, i.next());
	    if (result != null)
		return result;
	}
	return null;
    }

    /**
     * Match tries to match a pattern against an object. <p>
     *
     * Only ItemVars are treated as variables, and even then only if
     * they appear in the pattern.  Otherwise, they are treated as data.
     * If a variable appears more than once in a pattern, the objects
     * it matches must be equal. <p>
     *
     * Numbers are compared using equal.
     *
     * @return a MatchEnv or null.
     *
     * @see MatchEnv
     * @see ix.util.lisp.ItemVar
     */

    public static final MatchEnv match(Object pat, Object dat) {
	return match(pat, dat, null);
    }

    public static final MatchEnv match(Object pat, Object dat, MatchEnv env) {
//  	Debug.noteln("Pat", pat.getClass() + " " + pat);
//  	Debug.noteln("Dat", dat.getClass() + " " + dat);
//  	Debug.noteln("Env", env);
	if (pat == Matcher.ITEM_WILD) {
	    return success(env);
	}
	else if (pat instanceof ItemVar) {
	    if (env == null)
		env = new MatchEnv();
	    Object existingValue = env.get(pat);
	    if (existingValue == null) {
		env.put(pat, dat);
		return env;
	    }
	    // Could call match recursively here.  /\/
	    return existingValue.equals(dat) ? env : null;
	}
	// /\/: Can't just return when ==, because the pattern
	// might contain variables that should be bound first.
//  	else if (pat == dat) {
//  	    return success(env);
//  	}
	else if (pat instanceof Cons && ((Cons)pat).car() == REST) {
	    return matchRest(pat, dat, env);
	}
	else if (pat instanceof Number) {
	    return when(pat.equals(dat), env);
	}
	else if (pat.getClass() != dat.getClass()) {
	    return null;
	}
	else if (pat instanceof Symbol || pat instanceof Null) {
	    if (pat == dat) return success(env); else return null;
	}
	else if (pat instanceof String) {
	    return when(((String)pat).equals((String)dat), env);
	}
	else if (pat instanceof Cons) {
	    Cons p = (Cons)pat;
	    Cons d = (Cons)dat;
	    MatchEnv e = match(p.car(), d.car(), env);
	    if (e == null)
		return null;
	    else if (e == emptyEnv)
		return match(p.cdr(), d.cdr(), null);
	    else
		return when(match(p.cdr(), d.cdr(), e) != null,
			    e);
	}
	else {
	    return null;
	}

    }

    private static final MatchEnv matchRest(Object pat, Object dat,
					    MatchEnv env) {
	if (!(dat instanceof LList))
	    return null;

	Cons p = (Cons)pat;
	Object var = p.cdr().car();

	if (var == Lisp.NIL)	 	// (&rest) matches any List
	    return success(env);

	Debug.expect(var instanceof ItemVar, "(&rest ?var ...)");
	Debug.expect(p.cdr().cdr() == Lisp.NIL, "(&rest ?var)");

	if (env == null)
	    env = new MatchEnv();
	env.put(var, dat);
	return env;
    }

    private static final MatchEnv when(boolean cond, MatchEnv env) {
	if (cond)
	    return success(env);
	else
	    return null;
    }

    private static final MatchEnv success(MatchEnv env) {
	if (env == null)
	    return emptyEnv;
	else
	    return env;
    }

    public static void main(String[] argv) {
	while (true) {
	    Object pat = Lisp.readFromString(Util.askLine("Pat:"));
	    Object dat = Lisp.readFromString(Util.askLine("Dat:"));
	    System.out.println(match(pat, dat));
	    System.out.println("");
	}
    }

}
