/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jan 17 00:38:39 2004 by Jeff Dalton
 */

package ix.util.lisp;

import java.util.*;

import ix.util.*;
import ix.util.xml.XML;

/**
 * Provides re-readable (within a JVM) names for objects.
 *
 * <p>The names have the form "{" class-name "-" hash-value "}",
 * where the hash-value is obtained by calling {@link #hash(Object)}.
 * These names are used by {@link Lisp#printToString(Object)} and
 * by {@link Lisp#readFromString(String)} and {@link LispReader}s
 * in general, but they use a particular ObjectHash object that
 * is private to the {@link Lisp} class.
 *
 * <p>Note that ordinary, strong references are used.  Objects that
 * have been given to {@link #hash(Object)} will therefore not be
 * garbage collected.  This is to ensure that hash-names in strings
 * will work even if no other reference to the object exists.
 * Indeed, the only ref might be the user's memory of some output
 * that contained a hash-name.  The user can type in the same
 * hash-name and get back the object.
 *
 * <p>The class-name part of hash-names is not checked.  It is there
 * only to make the names more meaningful.
 */
public class ObjectHash {

    protected EqMap objectToHash = new EqMap();

    protected List hashToObject = new ArrayList();

    public ObjectHash() {}

    /**
     * Maps an object to an int that can be given to {@link #unhash(int)}
     * to get back the same (==) object.
     */
    public int hash(Object obj) {
	Integer h = (Integer)objectToHash.get(obj);
	if (h == null) {
	    h = new Integer(hashToObject.size());
	    objectToHash.put(obj, h);
	    hashToObject.add(obj);
	}
	return h.intValue();
    }

    /**
     * The inverse of {@link #hash(Object)}.
     */
    public Object unhash(int h) {
	return hashToObject.get(h);
    }

    /**
     * Returns a name of the form "{" class-name "-" hash-value "}",
     * where the hash-value is obtained by calling {@link #hash(Object)}.
     */
    public String hashName(Object obj) {
	Class c = obj.getClass();
	String className = XML.nameForClass(c);
	return "{" + className + "-" + hash(obj) + "}";
    }

    /**
     * The inverse of {@link #hashName(Object)}.
     *
     * @throws IllegalArgumentException if a valid object hash value
     *    cannot be found in the name.
     */
    public Object parseHashName(String name) {
	int h = -1;
	if (name.startsWith("{") && name.endsWith("}")) {
	    int i = name.lastIndexOf("-");
	    if (i > 0) {
		String number = name.substring(i + 1, name.length() - 1);
		try {
		    h = Integer.valueOf(number).intValue();
		}
		catch (NumberFormatException e) {
		}
	    }
	}
	if (h >= 0 && h < hashToObject.size())
	    return unhash(h);
	else
	    throw new IllegalArgumentException
		("Can't find an object ref in " + Strings.quote(name));
    }

    /** Test loop */
    public static void main(String[] argv) {
	ObjectHash hasher = new ObjectHash();
	for (;;) {
	    String in = Util.askLine("Object:");
	    if (in.equals("bye"))
		return;
	    Object obj = Lisp.readFromString(in);
	    String name;
	    Object back;
	    Debug.noteln("Read", obj);
	    Debug.noteln("Hash name", name = hasher.hashName(obj));
	    Debug.noteln("Back to", back = hasher.parseHashName(name));
	    Debug.noteln("Eq " + (back == obj));
	}
	
    }

}

// Issues:
// * In LispStreamTokenizer, "{" and "}" were made word chars so
//   that Lisp.readFromString(String) would see hash-names as words
//   and thus have an easier time handling them.  Also, "{" and "}"
//   seem to be allowed in symbols in Common Lisp without needing
//   any escape characters.  But it still seems like they ought to
//   be single-character symbols instead of name-constituents.
