/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct  2 18:04:19 2006 by Jeff Dalton
 * Copyright: (c) 2004, 2006, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A context-dependent list or queue, often used for building lists.
 */
public class LLQueue<E> extends AbstractSequentialList<E> {

    // The state is a 2-element LList[] so that we can
    // lookup both parts in one "get".

    static final int		// state-value array indices
	IN = 0,			// new elts, backwards
	OUT = 1;		// older elts, forwards

    ContextValue<LList[]> __state;

    // N.B. The initialContents are always in the root-context.
    // If the LLQueue is never modified in the root context,
    // they act as a default.

    public LLQueue() {
	this(Context.getContextHolder(), Lisp.NIL);
    }

    public LLQueue(Collection<? extends E> initialContents) {
	this(Context.getContextHolder(), initialContents);
    }

    public LLQueue(ContextHolder holder) {
	this(holder, Lisp.NIL);
    }

    public LLQueue(ContextHolder holder, 
                   Collection<? extends E> initialContents) {
	super();
	__state = new ContextValue(holder, new LList[] {
	    Lisp.NIL,
	    LList.newLList(initialContents)
	});
    }

    public LList contents() {
	// Changes the nearest state to have all its elements in state[OUT].
	LList[] state = (LList[])__state.get();
	if (state[IN] != Lisp.NIL) {
	    state[OUT] = state[OUT].append(state[IN].reverse());
	    state[IN] = Lisp.NIL;
	}
	return state[OUT];
    }

    public void setContents(List l) {
	setLLContents(LList.newLList(l));
    }

    protected LList setLLContents(LList contents) {
	__state.set(new LList[] {Lisp.NIL, contents});
	return contents;
    }

    // /\/: LinkedList has getFirst, getLast, removeFirst,
    // removeLast(), addFirst, addLast.  addFirst is equivalent
    // to our push and removeFirst is equivalent to pop.

    /** Add an object to the front of the list. */
    public void push(E o) {
	setLLContents(new Cons(o, contents()));
    }

    /** Remove and return the first element of the list. */
    public E pop() {
	LList c = contents();
	if (c == Lisp.NIL)
	    throw new IndexOutOfBoundsException
		("LLQueue pop() when empty");
	setLLContents(c.cdr());
	return (E)c.car();
    }

    /** Remove all elements in all contexts. */
    public void clearCompletely() {
	__state = new ContextValue(__state.holder,
				   new LList[] {Lisp.NIL, Lisp.NIL});
    }

    /* Methods required by AbstractSequentialList */

    @Override
    public int size() {
	// /\/: Should do this from the IN and OUT lists w/o getting contents.
	return contents().size();
    }

    @Override
    public ListIterator listIterator(int index) {
	return new LLQueueListIterator(contents(), index);
    }

    /** ListIterator that doesn't allow any modification. */
    protected static class LLQueueListIterator extends LListListIterator {
	// The superclass already disallows add and remove.
	public LLQueueListIterator(LList list, int start) {
	    super(list, start);
	}
	/** Always throws an UnsupportedOperationException. */
	public void set(Object o) {
	    throw new UnsupportedOperationException
		("Cannot set using an LLQueue ListIterator");
	}
    }

    /* Override inherited methods that would be incorrect
     * or too slow. */

    @Override
    public boolean add(E o) {
	// Inherited method would call add(size(), o)
	// which would throw an UnsupportedOperationException.
	LList[] state = (LList[])__state.get();
	// /\/: We could avoid creating a new array if we
	// already have one in the right context.
	__state.set(new LList[] { new Cons(o, state[IN]), state[OUT] });
	return true;
    }

    /**
     * Removes all elements in the current context.  To remove all
     * elements in all contexts, call {@link #clearCompletely()}.
     */
    @Override
    public void clear() {
	// Inherited method does removeRange(0, size()).
	setLLContents(Lisp.NIL);
    }

    @Override
    public boolean remove(Object o) {
	// Inherited method uses the iterator.
	LList c = contents();
	LList revised = c.withoutFirstEqual(o);
	setLLContents(revised);
	return revised != c;	// they're == if no removal
    }

    @Override
    public boolean removeAll(Collection<?> c) {
	// Inherited method uses the iterator.
	LList old = contents();
	LList revised = contents().withoutAll(c);
	setLLContents(revised);
	return revised != old;	// they're == if no removal
    }

    @Override
    public Iterator iterator() {
	// The method inherited from AbstractSequentialList
	// returns a ListIterator, but we can get a plain Iterator
	// from the LList contents.
	return contents().iterator();
    }

    @Override
    public boolean isEmpty() {
	LList[] state = (LList[])__state.get();
	return state[IN] == Lisp.NIL && state[OUT] == Lisp.NIL;
    }

    @Override
    public E get(int index) {
	return (E)contents().elementAt(index);
    }

    @Override
    public int lastIndexOf(Object o) {
	// The inherited method iterates backwards over the list
	// which would be very slow.
	return contents().lastIndexOf(o);
    }

}

