/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 25 04:52:26 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.io.Serializable;

import java.util.*;

import ix.util.lisp.*;

public class StableHashMap extends AbstractMap implements Serializable {

    protected HashMap map;

    protected Entry keyChain = new Entry(null, null);   // header
    protected Entry keyChainLast = keyChain;

    /* * * Constructors * * */

    public StableHashMap() {
	this.map = new HashMap();
    }

    public StableHashMap(int initialCapacity) {
	this.map = new HashMap(initialCapacity);
    }

    public StableHashMap(int initialCapacity, float loadFactor) {
	this.map = new HashMap(initialCapacity, loadFactor);
    }

    public StableHashMap(Map m) {
	this.map = new HashMap();
	putAll(m);
    }

    /* * * Instance methods * * */

    // AbstractMap requires only entrySet() and, for a modifiable Map,
    // put(Object key, Object value) and an entrySet().iterator() that
    // supports remove().

    public Object put(Object key, Object value) {
	// Irritatingly complicated by the need to return the
	// previous value.
	Entry e = (Entry)map.get(key);
	if (e == null) {
	    // New key, not seen before.  Create a new entry for it
	    // and put it at the end of the key-chain.
	    Entry ent = new Entry(key, value);
	    ent.linkAfter(keyChainLast);
	    keyChainLast = ent;
	    map.put(key, ent);
	    return null;
	}
	else {
	    Object oldValue = e.value;
	    e.value = value;
	    return oldValue;
	}
    }

    public Object get(Object key) {
	Entry e = (Entry)map.get(key);
	return e == null ? null : e.value;
    }

    public Set entrySet() {
	// Because all the values in the underlying map are
	// already Map.Entries, we don't have to create any
	// entry obects.  But we can't quite use the underlying
	// map's entry set as-is.
	return new EntrySet();
    }

    public boolean containsKey(Object key) {
	// The method from AbstractMap iterates over the entry-set.
	return map.get(key) != null;
    }

    public void clear() {
	map.clear();
	keyChain = new Entry(null, null);
	keyChainLast = keyChain;
    }

    public Object remove(Object key) {
	// throw new UnsupportedOperationException();
	Entry e = (Entry)map.remove(key);
	if (e == null)
	    return null;	// There was no entry to remove.
	// We now have to remove the entry from the key chain.
	Entry chain = keyChain;
	while (chain != null) {
	    Entry next = chain.nextEntry;
	    if (next == e) {
		// Found entry to delete
		chain.nextEntry = next.nextEntry;
		if (next == keyChainLast) {
		    Debug.expect(chain.nextEntry == null);
		    keyChainLast = chain;
		}
		return e.value;
	    }
	    chain = next;
	}
	throw new ConsistencyException("Couldn't find entry to delete");
    }

    /**
     * The Map.Entry implementation for a StableHashMap.
     */
    protected class Entry implements Map.Entry, Serializable {

	Object key;
	Object value;
	Entry nextEntry = null;

	Entry(Object key, Object value) {
	    this.key = key;
	    this.value = value;
	}

	public Object getKey() {
	    return key;
	}
         
	public Object getValue() {
	    return value;
	}

	public Object setValue(Object value) {
	    Object oldValue = this.value;
	    this.value = value;
	    return oldValue;
	}

	Entry next() {
	    return nextEntry;
	}

	void linkAfter (Entry previous) {
	    Debug.expect(previous.nextEntry == null, "pre already linked");
	    Debug.expect(this.nextEntry == null, "this already linked");
	    previous.nextEntry = this;
	}

	public boolean equals(Object o) {
	    if (!(o instanceof Map.Entry))
		return false;
	    Map.Entry e = (Map.Entry)o;
	    Object e_key = e.getKey();
	    Object e_value = e.getValue();
	    Object k = key;
	    Object v = getValue();
	    return (k == null ? e_key == null : k.equals(e_key))
		&& (v == null ? e_value == null : v.equals(e_value));
	}
         
	public int hashCode() {
	    Object k = key;
	    Object v = getValue();
	    return (k == null ? 0 : k.hashCode()) ^
		   (v == null ? 0 : v.hashCode());
 	}
	
    }

    /**
     * The Set implementation for a StableHashMap entrySet.
     */
    protected class EntrySet extends AbstractSet {

	Entry chain;

	EntrySet() {
	    this.chain = keyChain;
	}

	public int size() {
	    return map.size();
	}

	public Iterator iterator() {
	    return new EntrySetIterator(chain);
	}

	public boolean contains(Object o) {
	    return containsKey(o);
	}

	public void clear() {
	    StableHashMap.this.clear();
	    chain = keyChain;
	}

	public boolean remove(Object o) {
	    throw new UnsupportedOperationException();
	}

	public boolean removeAll(Collection c) {
	    throw new UnsupportedOperationException();
	}

	public boolean retainAll(Collection c) {
	    throw new UnsupportedOperationException();
	}

    }

    /**
     * The Iterator implementation for a StableHashMap entrySet.
     */
    protected class EntrySetIterator implements Iterator {

	Entry chain;

	EntrySetIterator(Entry chain) {
	    this.chain = chain;
	}

	public boolean hasNext() {
	    return chain.next() != null;
	}

	public Object next() {
	    return chain = chain.next();
	}

	public void remove() {
	    throw new UnsupportedOperationException
		(this + " does not support remove()");
	}

    }

}
