/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Feb  8 01:43:56 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

/**
 * Computes a value in a separate thread.  Example:
 * <pre>
 *   FutureValue fv = new FutureValue("computes v") {
 *       public Object computeValue() {
 *           ...;
 *       }
 *   };
 *   fv.start();
 *   ...
 *   Object v = fv.getValue();
 * </pre>
 * {@link #getValue()} waits until the value is available.
 */
public abstract class FutureValue {

    private Thread thread;
    private ThreadCondition cond;
    private Object value;
    private Throwable except;

    /**
     * Construct a FutureValue that has a daemon thread for computing
     * the value.  This is normally called for a subclass that defines
     * the {@link #computeValue()} method.
     */
    public FutureValue(String name) {
	thread = new FutureValueThread(name);
	cond = new ThreadCondition("Future " + name);
    }

    /**
     * Start the thread that will compute the value.
     */
    public void start() {
	thread.start();
    }

    /**
     * Returns the computed value once it is available.
     *
     * @throws RethrownException if the computation of the value
     *    threw and error or exception.
     */
    public Object getValue() {
	cond.waitUntilTrue();
	if (except == null)
	    return value;
	else
	    throw new RethrownException(except);
    }

    public void waitUntilReady(long timeout) {
	cond.waitUntilTrue(timeout);
    }

    public boolean isReady() {
	return cond.isTrue();
    }

    private void setValue(Object v) {
	value = v;
	cond.setTrue();
    }

    private void setException(Throwable t) {
	except = t;
	cond.setTrue();
    }

    private class FutureValueThread extends Thread {
	FutureValueThread(String name) {
	    super(name);
	    setDaemon(true);
	}
	public void run() {
	    try { setValue(computeValue()); }
	    catch (Throwable t) {
		setException(t);
	    }
	}
    }

    public abstract Object computeValue();

}

// Issues:
// * Should there be a getValue(long timeout)
//   or a waitUntilReady(long timeout)?
// * Perhaps this class should just return a Runnable that could
//   then be given to a Thread (instead of having its own Thread).
