/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Dec  7 01:04:13 2007 by Jeff Dalton
 * Copyright: (c) 2002, 2003, AIAI, University of Edinburgh
 */

package ix.util;

import java.lang.reflect.*;
import java.util.*;

/**
 * Function utilities
 *
 * @see Collect
 * @see Strings
 * @see Util#makeInstance(Class)
 * @see Util#makeInstance(Class, Object...)
 */
public final class Fn {

    private Fn() { }	// block instantiation

    /**
     * Calls the indicated method.
     *
     * @see #getMethod(Class c, String methodName, Class[] argSig)
     */
    public static Object apply(Object object,
			       String methodName,
			       List arguments) {
	return apply(object, methodName, arguments.toArray());
    }

    /**
     * Calls the indicated method.
     *
     * @see #getMethod(Class c, String methodName, Class[] argSig)
     */
    public static Object apply(Object object,
			       String methodName,
			       Object[] arguments) {
	Method m = getMethod(object, methodName, arguments);
	if (m != null)
	    return apply(object, m, arguments);
	else
	    throw new IllegalArgumentException
		("Could not find a " + methodName + " method " +
		 "for " + object + " and arguments " + 
		 Arrays.asList(arguments));
    }

    /**
     * Calls the indicated method.
     */
    public static Object apply(Object object,
			       Method meth,
			       Object[] arguments) {
	try {
	    return meth.invoke(object, arguments);
	}
	catch (IllegalAccessException e) {
	    throw new Error("Cannot access method " + meth, e);
	}
	catch (InvocationTargetException e) {
	    throw new RuntimeException
		("Exception from method " + meth + ": " +
		 Debug.describeException(e.getTargetException()), e);
	}
    }

    /**
     * Calls the indicated static method.
     *
     * @see #getMethod(Class c, String methodName, Class[] argSig)
     */
    public static Object applyStatic(Class c,
				     String methodName,
				     List arguments) {
	return applyStatic(c, methodName, arguments.toArray());
    }

    /**
     * Calls the indicated static method.
     *
     * @see #getMethod(Class c, String methodName, Class[] argSig)
     */
    public static Object applyStatic(Class c,
				     String methodName,
				     Object[] arguments) {
	Class[] sig = new Class[arguments.length];
	for (int i = 0; i < sig.length; i++) {
	    sig[i] = arguments[i].getClass();
	}
	Method m = getMethod(c, methodName, sig);
	if (m != null)
	    return apply(null, m, arguments);
	else
	    throw new IllegalArgumentException
		("Could not find a " + methodName + " method " +
		 "for class " + c + " and arguments " + 
		 Arrays.asList(arguments));
    }

    /**
     * Returns the specified method if it exists, else null.
     *
     * @see #getMethod(Class c, String methodName, Class[] argSig)
     */
    public static Method getMethod(Object object,
				   String methodName,
				   Object[] arguments) {
	Class c = object.getClass();
	Class[] sig = new Class[arguments.length];
	for (int i = 0; i < sig.length; i++) {
	    sig[i] = arguments[i].getClass();
	}
	return getMethod(c, methodName, sig);
    }

    // /\/: Fn.getMethod(someObject, methName, Class[] sig) calls
    // the method above, NOT the one below.

    /**
     * Returns the specified method if it exists, else null.
     */
    public static Method getMethod(Class c,
				   String methodName,
				   Class[] argSig) {
	Method chosen = null;
	for (Method meth: c.getMethods()) {
	    if (meth.getName().equals(methodName)
		  && isApplicable(meth, argSig)
		  && (chosen == null || isMoreSpecific(meth, chosen)))
		chosen = meth;	    
	}
	return chosen;
    }

    private static boolean isApplicable(Method meth, Class[] argSig) {
	Class[] sig = meth.getParameterTypes();
	if (sig.length != argSig.length)
	    return false;
	for (int i = 0; i < sig.length; i++) {
	    if (!objectClass(sig[i]).isAssignableFrom(argSig[i]))
		return false;
	}
	return true;
    }

    private static Class objectClass(Class type) {
	if (!type.isPrimitive())
	    return type;
	else if (type == Boolean.TYPE) return Boolean.class;
	else if (type == Integer.TYPE) return Integer.class;
	else if (type == Long.TYPE) return Long.class;
	else if (type == Float.TYPE) return Float.class;
	else if (type == Double.TYPE) return Double.class;
	else if (type == Character.TYPE) return Character.class;
	else if (type == Short.TYPE) return Short.class;
	else if (type == Byte.TYPE) return Byte.class;
	else
	    throw new ConsistencyException("No object class for", type);
    }

    private static boolean isMoreSpecific(Method m1, Method m2) {
	Class[] sig1 = m1.getParameterTypes();
	Class[] sig2 = m2.getParameterTypes();
	Debug.expect(sig1.length == sig2.length);
	for (int i = 0; i < sig1.length; i++) {
	    if (!isMoreSpecific(sig1[i], sig2[i]))
		return false;
	}
	return true;
    }

    private static boolean isMoreSpecific(Class c1, Class c2) {
	// Is every instance of c1 an instance of c2?
	// We allow c1 == c2.
	return c2.isAssignableFrom(c1);
    }
 
    /**
     * Returns a function that calls the 0-argument method specified
     * by a class and method name.
     */
    public static Function1 accessor(final Class c,
				     final String methodName) {
	final Class[] empty = new Class[] {}; // doubles as sig and arglist
	try {
	    // Note that c.getMethod(...) goes up the superclass chain
	    // looking for the method.
	    final Method meth = c.getMethod(methodName, empty);
	    return new Function1() {
	        public Object funcall(Object a) {
		    try { 
                        // Cast to Object[] to get non-varargs call.
			return meth.invoke(a, (Object[])empty);
		    }
		    catch (IllegalAccessException e) {
			throw new Error("Cannot access method " + methodName +
					" for class " + c, e);
		    }
		    catch (InvocationTargetException e) {
			throw new RuntimeException
			    ("Exception from method " +
			       methodName + " from class " + c + ": " +
			       Debug.describeException
			         (e.getTargetException()),
			     e);
		    }
		}
	    };
	}
	catch (NoSuchMethodException e) {
	    throw new Error("Cannot find method " + methodName +
			    " for class " + c, e);
	}
    }

    /**
     * Returns a function that always returns the same value.
     */
    public static Function1 always(final Object value) {
	return new Function1() {
	    public Object funcall(Object arg) {
		return value;
	    }
	};
    }

    /**
     * Returns a predicate that returns false when the given
     * predicate returns true, and true when it returns false.
     */
    public static Predicate1 negate(final Predicate1 p) {
	return new Predicate1() {
	    public boolean trueOf(Object a) {
		return !p.trueOf(a);
	    }
	};
    }

    /**
     * Returns a predicate that test whether an object is an
     * instance of the specified class.
     */
    public static Predicate1 isInstanceOf(final Class c) {
	return new Predicate1() {
	    public boolean trueOf(Object a) {
		return c.isInstance(a);
	    }
	};
    }

    /**
     * A predicate that returns true for any object.
     */
    public static final Predicate1 alwaysTrue = new Predicate1() {
	public boolean trueOf(Object a) {
	    return true;
	}
    };

    /**
     * A predicate that returns false for any object.
     */
    public static final Predicate1 alwaysFalse = new Predicate1() {
	public boolean trueOf(Object a) {
	    return false;
	}
    };

    /**
     * The identity function.
     */
    public static final Function1 identity = new Function1() {
	public Object funcall(Object a) {
	    return a;
	}
    };

}

