/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 22 17:36:41 2007 by Jeff Dalton
 * Copyright: (c) 2003, 2005 - 2007, AIAI, University of Edinburgh
 */

package ix.util;

import java.io.Serializable;
import java.text.*;
import java.util.*;

import ix.util.lisp.*;

/**
 * The difference between two dates.  <i>(There must be a decent
 * Duration class out there somewhere, but I was unable to find
 * it and so wrote this one.  There is a lot that it doesn't do,
 * but it allows durations to be created and to be output in a
 * readable form.  It also supports most of the XML Schema / ISO 8601
 * syntax for durations.)</i>
 *
 * <p>Durations should be understood as periods of time without
 * fixed endpoints, such as 10 minutes or 3 hours.</p>
 *
 * <p>The full XML Schema duration datatype is difficult to work
 * with, because the number of days in a month varies.  Hence
 * durations aren't even totally ordered.  However, in the 1.1
 * version of the Schema definition, there are two derived types
 * that are each totally ordered: yearMonthDuration and dayTimeDuration.
 * That also means that full durations can be represented by two
 * values: months and seconds.  In the earlier 1.0 version,
 * durations were in a 6-dimensional space: years, months, days,
 * hours, minutes, and seconds.  That had the unfortunate
 * consequence that 1 day and 24 hours ("PT1D" and "PT24H")
 * were considered different values rather than two (lexical)
 * representations of the same value.</p>
 *
 * <p>This class avoids some of the complexity by disallowing
 * year and month values, thus making it close to the dayTimeDuration
 * type.  For long durations, the number of days can become large
 * instead.  However, this class retains the idea of a multi-dimensional
 * space and instances needn't always have the number of hours be
 * less than 24, the number of minutes less than 60, and so on.
 * Users should not reply on the details of this class's behaviour
 * in this area, because they are likely to change.</p> 
 *
 * <p>The canonical numeric representation of an instance of this
 * class is as a long holding the total number of milliseconds
 * in the period of time represented by the duration.  The breakdown
 * into days, hours, minutes, etc is normally visible only when
 * a Duration is converted to a String.</p>
 *
 * @author Jeff Dalton
 */
public class Duration implements SemiPrimitive, Comparable, Serializable {

    // Java Date getTime() returns the number of milliseconds
    // since January 1, 1970, 00:00:00 GMT.

    protected long milliseconds;

    protected boolean negative = false;
    protected long days;
    protected int hours;
    protected int minutes;
    protected int seconds;
    protected int ms;

    /**
     * Constructs a duration by taking the difference of two dates.
     * This is equivalent to
     * <pre>
     *    new Duration(to.getTime() - from.getTime())
     * </pre>
     */
    public Duration(Date from, Date to) {
	this.milliseconds = to.getTime() - from.getTime();
	breakup(this.milliseconds);
    }

    /**
     * Constructs a duration that corresponds to the specified
     * number of milliseconds.  This is the inverse of the
     * {@link #asMilliseconds()} method.  This constructor
     * always produces a duration in which the number of hours
     * is less than 24, the number of minutes is less than 60,
     * and so on.
     */
    public Duration(long milliseconds) {
	this.milliseconds = milliseconds;
	breakup(this.milliseconds);
    }

    /**
     * Constructs a duration from the specified component times.
     */
    public Duration(long days, int hours, int minutes, int seconds, int ms) {
	this.milliseconds = combine(days, hours, minutes, seconds, ms);
	// We still have to break up the total, because some of the
	// initial values may have been too large, e.g. minutes >= 60.
	// /\/: Should we do this?  Compare parseISOString.
	breakup(this.milliseconds);
    }

    /**
     * Constructs a duration from a string written in a subset of
     * the ISO 8601 notation used for the XML Schema duration datatype.
     *
     * <p>The full syntax is PnYnMnDTnHnMnS, optionally preceded by
     * a minus sign, where each n is a possibly different unsigned number,
     * 'Y', 'M', 'D', 'H', 'M, and 'S' stand for years, months, days, hours,
     * minutes, and seconds respectively, 'P' stands for "period",
     * and 'T' separates the year-month date portion from the time
     * portion.  'P' must always be present, and 'T' must be present
     * if any hours, minutes, or seconds are specified.  The number-letter
     * combinations are all optional, so long as at least one is
     * specified.  All of the numbers must have no decimal point
     * except for the number of seconds.  None of the numbers is
     * restricted in range.</p>
     *
     * <p>This class does not support year or month values and they
     * must not be given.  Second fractions are stored only to the
     * nearest millisecond, although they can be written to arbitrary
     * precision.</p>
     *
     * @throws Duration.SyntaxException if the string does not
     *    conform to the required syntax.
     *
     * @see #toISOString()
     * @see #parseISOString(String)
     */
    public Duration(String isoString) {
	parseISOString(isoString);
    }

    /** Separate the total milliseconds into days, hours, minutes, ... */
    protected void breakup(long total) {
	long t = total;
	negative = t < 0;
	if (negative)
	    t = -t;
	this.ms = (int)(t % 1000);
	t = t / 1000;
	this.seconds = (int)(t % 60);
	t = t / 60;
	this.minutes = (int)(t % 60);
	t = t / 60;
	this.hours = (int)(t % 24);
	t = t / 24;
	this.days = t;
	Debug.expect(hours < 24);
	Debug.expect(minutes < 60);
	Debug.expect(seconds < 60);
	Debug.expect(ms < 1000);
	Debug.expect(combine(days, hours, minutes, seconds, ms)
		     == Math.abs(total));
    }

    /** Combine the components into a total number of milliseconds. */
    protected long combine(long days, int hours, int minutes,
			   int seconds, int ms) {
	long total = days * 24 + hours;
	total = total * 60 + minutes;
	total = total * 60 + seconds;
	total = total * 1000 + ms;
	return total;
    }

    /**
     * Returns this Duration after rounding the time to the nearest
     * multiple of the specified granularity.  For example, to create
     * a duration rounded to the nearest minute, you could do
     * <pre>
     *   Duration d = new Duration(...).round(60 * 1000);
     * </pre>
     *
     * @param granularity  milliseconds, usually a value that represents
     *    one larger unit such as a day, hour, or minute.
     *
     * @see #roundToSeconds()
     * @see #roundToMinutes()
     */
    public Duration round(long granularity) {
	long t = milliseconds / granularity;
	long rem = milliseconds % granularity;
	if (rem >= granularity / 2)
	    t += 1;
	milliseconds = t * granularity;
	breakup(milliseconds);		// don't forget this!
	return this;
    }

    /** Equivalent to round(1000). */
    public Duration roundToSeconds() {
	return round(1 * 1000);
    }

    /** Equivalent to round(60 * 1000). */
    public Duration roundToMinutes() {
	return round(1 * 60 * 1000);
    }

    /**
     * Return the total number of milliseconds represented by this
     * Duration.  This method is the inverse of the {@link #Duration(long)}
     * constructor.
     */
    public long asMilliseconds() {
	return milliseconds;
    }

    /**
     * Returns a string that represents this duration in ISO 8601 syntax.
     * This method is a rough inverse of the {@link #Duration(String)}
     * constructor.  It will return only strings that can be parsed
     * by that constructor, but if s is the string returned for this
     * duration, it is not absolutely guaranteed that
     * <pre>
     *    new Duration(s).toISOString().equals(s)
     * </pre>
     * However, this duration and one greated from s will have
     * the same value as milliseconds.
     *
     * @see #toString()
     */
    public String toISOString() {
	if (milliseconds == 0) return "PT0S";
	StringBuffer s = new StringBuffer();
	if (negative) s.append("-");
	s.append("P");
	if (days != 0) { s.append(days); s.append("D"); }
	if (hours == 0 && minutes == 0 && seconds == 0 && ms == 0)
	    return s.toString();
	s.append("T");
	if (hours != 0) { s.append(hours); s.append("H"); }
	if (minutes != 0) { s.append(minutes); s.append("M"); }
	if (ms != 0) {
	    Debug.expect(ms < 1000, "ms >= 1000");
	    s.append(seconds);
	    s.append(".");
	    // Add 1000 to ms to provide any needed leading zeros,
	    // then delete the '1'.
	    s.append(1000 + ms); s.deleteCharAt(s.length() - 4);
	    s.append("S");
	}
	else if (seconds != 0) {
	    s.append(seconds);
	    s.append("S");
	}
	return s.toString();
    }

    /**
     * Describes a syntax error found when constructing a duration
     * from a string.
     *
     * @see Duration#Duration(String)
     * @see #parseISOString(String)
     */
    public static class SyntaxException extends RuntimeException {
	SyntaxException(String sourceText, String problem, int charPos) {
	    this(sourceText, problem + " at position " + charPos);
	}
	SyntaxException(String sourceText, String problem) {
	    this("Duration \"" + sourceText + "\" " + problem);
	}
	SyntaxException(String message) {
	    super(message);
	}
    }

    protected LList isoSequence =
	Lisp.list(new Character('Y'), new Character('D'),
		  new Character('T'), new Character('H'),
		  new Character('M'), new Character('S'));

    /**
     * Fills in fields in this duration by parsing a string
     * that is written in the subset of ISO 8601 notation that
     * is supported by this class.  This method is called by
     * the {@link #Duration(String)} constructor.
     *
     * @throws Duration.SyntaxException if the string does not
     *    conform to the required syntax.
     */
    protected void parseISOString(String s) {
	RefInt r = new RefInt(0);
	boolean inTime = false;
	boolean aNumberWasSeen = false;
	if (s.charAt(0) == '-') { negative = true; r.i++; }
	if (s.charAt(r.i) == 'P')
	    r.i++;
	else
	    throw new Duration.SyntaxException
		(s, "does not begin with \"P\".");
	LList seq = LList.newLList(isoSequence);
	while (r.i < s.length() /* && !seq.isEmpty() */ ) {
	    char c = s.charAt(r.i);
	    if (c == 'T') {
		inTime = true;
	    }
	    else {
		int n = scanInt(s, r);
		aNumberWasSeen = true;
		if (r.i >= s.length())
		    throw new Duration.SyntaxException
			(s, "is missing " + (inTime ? "H, M, or S" : "D") +
			    " after " + n);
		c = s.charAt(r.i);
		if (c == 'D') days = n;
		else {
		    if (c == 'H') hours = n;
		    else if (c == 'M') minutes = n;
		    else if (c == 'S') seconds = n;
		    else if (c == '.') {
			seconds = n;
			r.i++;
			int millis = scan1000ths(s, r);
			Debug.expect(millis <= 1000);
			if (millis < 1000) ms = millis; else seconds += 1;
			if (r.i >= s.length() || (c = s.charAt(r.i)) != 'S')
			    throw new Duration.SyntaxException
				(s, "is missing an 'S' after the number " +
				    "of seconds");
		    }
		    else
			throw new Duration.SyntaxException
			    (s, "has an invalid character, '"+c+"', ", r.i);
		    if (!inTime)
			throw new Duration.SyntaxException
			    ("'" + c + "' is not after 'T' in duration " +
			     Strings.quote(s));
		}
	    }
	    Character C = new Character(c); // inefficient /\/
	    if (!seq.contains(C))
		throw new Duration.SyntaxException
		    (s, "has '" + c + "' out of sequence", r.i);
	    seq = seq.dropTo(C).cdr();
	    r.i++;
	}
	if (r.i < s.length())
	    throw new Duration.SyntaxException
		(s, "has invalid characters at the end");
	if (!aNumberWasSeen)
	    throw new Duration.SyntaxException
		(s, "does not give a number");
	milliseconds = combine(days, hours, minutes, seconds, ms);
	// Note that hours may be > 24, minutes > 60, and so on.
	// That's probably closer to what's intended for the
	// "six-dimensional" XML Schema datatype, but it may
	// be surprising.  Calling breakup(milliseconds)
	// would put eveything into its limited range.
    }

    private class RefInt {
	int i; RefInt(int i) { this.i = i; }
    }

    private int scanInt(String s, RefInt r) {
	// Leaves r.i pointing to the first character after the digits.
	int value = 0;
	int start = r.i;
	while (r.i < s.length()) {
	    char c = s.charAt(r.i);
	    if (!Character.isDigit(c))
		break;
	    value = value * 10 + Character.digit(c, 10);
	    r.i++;
	}
	if (r.i == start)
	    throw new Duration.SyntaxException
	      (s, "lacks a positive integer", r.i);
	return value;
    }

    private int scan1000ths(String s, RefInt r) {
	// Leaves r.i pointing to the first character after the digits.
	int value = 0;
	int digitNumber = 0;
	int factor = 100;
	while (digitNumber < 3 && r.i < s.length()) {
	    char c = s.charAt(r.i);
	    if (!Character.isDigit(c))
		break;
	    Debug.expect(factor >= 1);
	    value += factor * Character.digit(c, 10);
	    factor = factor / 10;
	    digitNumber++;
	    r.i++;
	}
	if (digitNumber == 0)
	    throw new Duration.SyntaxException
	      (s, "lacks a fraction", r.i);
	else if (digitNumber == 3) {
	    // If there are more than 3 digits after the ".",
	    // use them only to round.
	    char next = s.charAt(r.i);
	    if (Character.isDigit(next)) {
		// Look at the first extra digit.
		int d = Character.digit(next, 10);
		if (d >= 5)
		    value += 1;
		// Scan off any remaining digits.
		do { r.i++; } while (Character.isDigit(s.charAt(r.i)));
	    }
	}
	return value;
    }

    /**
     * Returns an easy-to-read description of this duration.
     * For example, <tt>"10 days, 23 hours, 1 minute"</tt>.
     *
     * @see #toISOString()
     */
    public String toString() {
	if (milliseconds == 0) return "0 seconds";
	StringBuffer result = new StringBuffer();
	addValue(result, days, "day");
	addValue(result, hours, "hour");
	addValue(result, minutes, "minute");
	addValue(result, seconds, "second");
	addValue(result, ms, "millisecond");
	if (negative)
	    result.insert(0, "negative "); // must be after the above
	return result.toString();
    }

    private void addValue(StringBuffer b, long v, String singular) {
	if (v > 0) {
	    if (b.length() > 0)
		b.append(", ");
	    b.append(v).append(" ").append(singular);
	    if (v > 1)
		b.append("s");
	}
    }

    /**
     * Returns a hash value for this duration.  This is computed
     * as the exclusive OR of the two halves of the number of
     * milliseconds in this duration represented as a long.
     */
    public int hashCode() {
	return (int)(milliseconds ^ (milliseconds >> 32));
    }

    /**
     * Determines whether two durations represent the same amount of time.
     *
     * @return true if the argument duration represents the same amount
     *   of time as this duration and false if it does not.
     *
     * @throws ClassCastException if the argument is not a Duration.
     */
    public boolean equals(Object obj) {
	return this.asMilliseconds() == ((Duration)obj).asMilliseconds();
    }

    /**
     * Compares two durations.
     *
     * @return 0 if the durations are equal, a value less than 0
     *    if this duration is a shorter time than the argument duration,
     *    and a value greater than 0 if this duration is longer than
     *    the argument duration.
     *
     * @throws ClassCastException if the argument is not a Duration.
     */
    public int compareTo(Object obj) {
	return (int)(this.asMilliseconds() - ((Duration)obj).asMilliseconds());
    }

    /**
     * Simple test program.
     */
    public static void main(String[] argv) {
	System.out.println(new Duration(10, 23, 0, 59, 12).toISOString());
	System.out.println(new Duration(10, 23, 0, 59, 12));
	System.out.println(new Duration(10, 23, 0, 59, 12).roundToMinutes());
	System.out.println(new Duration(10, 23, 0, 30, 0).roundToMinutes());
	System.out.println(new Duration(10, 23, 0, 29, 500).roundToMinutes());
	System.out.println(new Duration(10, 23, 0, 29, 500).roundToSeconds());
	Duration d = new Duration(10, 23, 0, 59, 12);
	Debug.expect(d.asMilliseconds() ==
		     new Duration(d.asMilliseconds()).asMilliseconds());
	Duration nd = new Duration(-d.asMilliseconds());
	System.out.println(nd);
	Debug.expect(nd.asMilliseconds() ==
		     new Duration(nd.asMilliseconds()).asMilliseconds());
	d = new Duration("P1D");
	Debug.expectEquals(d, new Duration(d.asMilliseconds()));
	Debug.expectEquals(d, new Duration(d.toISOString()));
	Debug.expect(d.compareTo(new Duration(d.toISOString())) == 0);
	while (true) {
	    String in = Util.askLine("Duration: ");
	    try {
		Duration du = new Duration(in);
		System.out.println(du.toISOString() + " = " + du.toString());
		Duration d2 = new Duration(du.asMilliseconds());
		System.out.println(d2.toISOString() + " = " + d2.toString());
	    }
	    catch (Throwable t) {
		Debug.noteException(t, false);
	    }
	}
    }

}
