/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Apr 27 02:18:18 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.test;

import java.io.*;
import java.util.*;

import ix.iplan.*;
import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.plan.inspect.*;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * Converts a Plan to a format that can be understood by
 * etc/psgraph-ix-plan.
 */
public class OPlanPSGraph {

    public OPlanPSGraph() {
    }

    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);
	new OPlanPSGraph().run();
    }

    protected void run() {
	Plan plan = (Plan)XML.readObject(Plan.class,
					 Parameters.getParameter("plan"));
	outputPlanForOPlan(plan, Parameters.getParameter("output", "-"));
    }

    public void outputPlanForOPlan(Plan plan, String where) {
	if (where.equals("-")) {
	    outputPlanForOPlan(plan, System.out);
	    return;
	}
	try {
	    File file = new File(where);
	    FileOutputStream fos = new FileOutputStream(file);
	    PrintStream ps = new PrintStream(fos);
	    outputPlanForOPlan(plan, ps);
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    throw new RethrownException(e);
	}
    }

    public void outputPlanForOPlan(Plan plan, final PrintStream out) {
	PlanInspector inspect = new PlanInspector(plan);
	// Describe nodes
	out.println("(");
	inspect.walkPlanNodes(new PlanItemVisitor() {
	    public void visit(AbstractPlanItem pi,
			      AbstractPlanItem parent,
			      List children) {
		PlanNode node = (PlanNode)pi;
		LList description = nodeToList(node);
		out.println(Lisp.printToString(description));
	    }
	});
	out.println(")");
	// Describe orderings
	out.println("(");
	inspect.walkOrderings(new OrderingVisitor() {
	    public void visit(Ordering ord) {
		LList description = orderingToList(ord);
		out.println(Lisp.printToString(description));
	    }
	});
	out.println(")");
    }

    LList nodeToList(PlanNode node) {
	String id = node.getId().toString();
	LList pattern = node.getActivity().getPattern();
	return Lisp.list(id, pattern);
    }

    LList orderingToList(Ordering ord) {
	NodeEndRef from = ord.getFrom();
	NodeEndRef to = ord.getTo();
	return
	    Lisp.list(Lisp.list(from.getNode().toString(),
				endKeyword(from.getEnd())),
		      Lisp.list(to.getNode().toString(),
				endKeyword(to.getEnd())));
    }

    Symbol endKeyword(End end) {
	return Symbol.intern(":" + end.toString());
    }

}
