/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 31 22:14:15 2005 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.ip2.*;
import ix.icore.Status;
import ix.icore.domain.*;	// for PatternAssignment and friends
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

public class MatchStateHandler extends ActivityHandler {

    Symbol verb;

    static final Symbol
//  	S_MATCH_STATE = Symbol.intern("match-state"),
	S_COMMA       = Symbol.intern(","),
	S_EQUAL       = Symbol.intern("=");

    Ip2 ip2;

    public MatchStateHandler(Ip2 ip2) {
	this(ip2, "match-state");
    }

    public MatchStateHandler(Ip2 ip2, String verb) {
	super("Match state");
	this.ip2 = ip2;
	this.verb = Symbol.intern(verb);
    }

    public List getSyntaxList() {
	return (LList)Lisp.readFromString
	    // ("((match-state { ?pattern [ = ?value ] , } ...))");
	    ("((" + verb + " { (?pattern-item ...) [ = ?value ] } ...))");
    }

    public boolean appliesTo(AgendaItem item) {
	LList pattern = item.getPattern();
	return pattern.length() > 1
	    && pattern.get(0) == verb;
    }

    public void addHandlerActions(AgendaItem item) {
	// appliesTo will have already been called.
	item.addAction(new MatchStateAction(item));
    }

    /** Parser for comma-separated pattern = value pairs. */
    protected ListOfConstraint commaSyntax(LList itemPattern) {
	LListCollector tokens = new LListCollector(itemPattern);
	ListOfConstraint result = new LinkedListOfConstraint();
	if (tokens.popElement() != verb)
	    throw new ConsistencyException("No match-state", itemPattern);
	while (!tokens.isEmpty()) {
	    // Get the next pattern.  N.B. no parens around pattern.
	    LList pat = tokens.popToAny(Lisp.list(S_COMMA, S_EQUAL));
	    if (pat.isEmpty())
		throw new IllegalArgumentException
		    ("Empty pattern before " + tokens.contents() +
		     " in pattern " + itemPattern);
	    if (tokens.isEmpty())
		result.add(makeConstraint(pat));
	    else if (tokens.firstElement() == S_COMMA) {
		tokens.popElement();
		result.add(makeConstraint(pat));
	    }
	    else if (tokens.firstElement() == S_EQUAL) {
		tokens.popElement();
		LList val = tokens.popToAny(Lisp.list(S_COMMA));
		if (!tokens.isEmpty())
		    Debug.expect(tokens.popElement() == S_COMMA);
		if (val.size() == 1)
		    result.add(makeConstraint(pat, val.get(0)));
		else
		    throw new IllegalArgumentException
			("More than one value for pattern " + pat +
			 " in item pattern " + itemPattern);
	    }
	    else
		throw new IllegalArgumentException
		    ("Unexpected token " + tokens.firstElement() +
		     " at start of " + tokens.contents() +
		     " in " + itemPattern);
	}
	return result;
    }

    Constraint makeConstraint(LList pattern) {
	return makeConstraint(pattern, PatternAssignment.S_TRUE);
    }

    Constraint makeConstraint(LList pattern, Object value) {
	return new Constraint
	    (Refinement.S_WORLD_STATE,
	     Refinement.S_CONDITION,
	     Lisp.list(new PatternAssignment(pattern, value)));
    }

    /** Parser for pattern = value pairs with each pattern in parens. */
    protected ListOfConstraint parenSyntax(LList itemPattern) {
	LListCollector tokens = new LListCollector(itemPattern);
	ListOfConstraint result = new LinkedListOfConstraint();
	if (tokens.popElement() != verb)
	    throw new ConsistencyException("No match-state", itemPattern);
	while (!tokens.isEmpty()) {
	    Object patObj = tokens.popElement();
	    if (!(patObj instanceof Cons))
		throw new IllegalArgumentException
		    ("Illegal pattern before " + tokens.contents() +
		     " in item pattern " + itemPattern);
	    Cons pat = (Cons)patObj;
	    // Debug.noteln("Pattern", pat);
	    if (tokens.isEmpty())
		result.add(makeConstraint(pat));
	    else if (tokens.firstElement() == S_EQUAL) {
		tokens.popElement();
		Object val = tokens.popElement();
		result.add(makeConstraint(pat, val));
	    }
	    else
		result.add(makeConstraint(pat));
	    // Debug.noteln("Partial cond list", result);
	}
	return result;
    }

    class MatchStateAction extends HandlerAction {

	AgendaItem item;
	ListOfConstraint conditions;

	MatchStateAction(AgendaItem item) {
	    this.item = item;
	    this.shortDescription = "Match world state";
	    LList pat = item.getPattern();
	    this.conditions = pat.contains(S_COMMA)
		? commaSyntax(pat)
		: parenSyntax(pat);
	    Debug.noteln("Parsed conditions", conditions);
	}

	public boolean isReady() {
	    return true;
	}

	public void handle(AgendaItem item) {
	    Debug.expect(item == this.item, "Wrong item");

	    // We want to match each "condition" independently against
	    // the world state, creating a MatchChoice for each cond.
	    // But if we use the MM's filter-matching, it will produce
	    // a single list of MatchEnvs, one env for each way to match
	    // all of the conditions.  Since that should be equivalent,
	    // and is easier to do (since the MM does all the work),
	    // we'll do that, at least for now.  /\/

	    Debug.noteln("Conditions", conditions);

	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    List envs = mm.evalFilters(conditions, new MatchEnv());
	    if (envs.isEmpty()) {
		throw new FailureException
		    ("Cannot match everything");
	    }
	    else {
		MatchChoice mc = mm.addMatchChoice(envs);
		item.setStatus(Status.COMPLETE);
	    }
	}

    }

}
