/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  6 14:03:22 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.test;

import java.io.*;

import java.util.*;

import ix.iplan.*;
import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.plan.build.*;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * A Utility for running planning problems with files as input and output.
 *
 * <p>Parameters:
 * <dl>
 * <dt><tt>domain=</tt><i>filename</i></dt>
 * <dd>
 *   The domain to use when planning.  A {@link ParameterException}
 *   will be thrown if no domain is given.
 * </dd>
 *
 * <dt><tt>plan=</tt><i>filename</i></dt>
 * <dd>
 *   A initial plan that sets up the planning problem.  This is optional,
 *   but if it is not given, a task must be, or else a ParameterException
 *   will be thrown.
 * </dd>
 *
 * <dt><tt>task=</tt><i>activity-pattern</i></dt>
 * <dd>
 *   This is an alternative to specifying a <tt>plan</tt>, although
 *   both can be given.  The pattern is turned into an activity
 *   which is then added to the initial plan that's given to the
 *   planner.  The pattern is typically a sequence of words, numbers,
 *   etc separated by spaces, that is meaningful in the domain.
 *   For example, "<tt>test1</tt>", "<tt>drive-to Glasgow</tt>".
 * </dd>
 *
 * <dt><tt>number-of-plans=</tt><i>integer</i></dt>
 * <dd>
 *   The number of plans to generate (defaults to 1).
 * </dd>
 *
 * <dt><tt>output-directory=</tt><i>filename</i></dt>
 * <dd>
 *   The directory where output plans should be put.  If no directory
 *   is specified, the directory containing the initial plan is used,
 *   if an initial plan was given, or else the directory containing
 *   the domain.
 * </dd>
 *
 * <dt><tt>output-file-type=</tt><i>string</i></dt>
 * <dd>
 *   The file type to use for output plans.  If this is not specified,
 *   the type of the initial plan's file is used, if an initial plan
 *   was given; otherwise, the type defaults to "xml".
 * </dd>
 *
 * <dt><tt>test=</tt><i>true or false</i></dt>
 * <dd>
 *   If true, plans are generated as usual, but no files are created.
 *   If this parameter is not specified, it is assumed to be false.
 * </dd>
 * </dl>
 */
public class FilePlanner extends PlanningTest {

    protected String outputDirecoryName;
    protected String outputFileType;
    protected boolean testOnly;

    protected File domainFile;
    protected File planFile;
    protected File outputDirecoryFile;

    protected int numberOfPlans;
    protected int planNumber;

    public FilePlanner() {
    }

    public static void main(String[] argv) {
	Debug.off();
	try {
	    Parameters.setIsInteractive(false);
	    Parameters.processCommandLineArguments(argv);
	    FilePlanner fp = new FilePlanner();
	    fp.run();
	}
	catch (ParameterException pe) {
	    System.out.println(Debug.describeException(pe));
	    System.exit(1);
	}
    }

    @Override
    protected void run() {
	setup();
	planNumber = 1;		// number of the plan we're about to find
	planner.plan();
	outputPlan(planner.getPlan());
	while (planNumber <= numberOfPlans) {
	    planner.replan();
	    outputPlan(planner.getPlan());
	}
    }

    @Override
    protected void setup() {
	super.setup();
	numberOfPlans = Parameters.getInt("number-of-plans", 1);
	outputDirecoryName = Parameters.getParameter("output-directory");
	outputFileType = Parameters.getParameter("output-file-type");
	testOnly = Parameters.getBoolean("test", false);
	// See whether the user gave us everything we need.
	checkThatEnoughParametersWereGiven();
	// Turn filenames into abstract pathnames (Files).
	if (domainName != null)
	    domainFile = new File(domainName);
	if (planName != null)
	    planFile = new File(planName);
	// Determine what directory to use for output.
	if (outputDirecoryName != null)
	    outputDirecoryFile = new File(outputDirecoryName);
	else if (planFile != null)
	    outputDirecoryFile = planFile.getParentFile();
	else if (domainFile != null)
	    outputDirecoryFile = domainFile.getParentFile();
	else
	    throw new ConsistencyException("Can't determine output directory");
	// Determine output file type.
	if (outputFileType != null)
	    ; // an output-file-type parameter was given
	else if (planFile != null
		 && planFile.getName().indexOf(".") > 0)
	    outputFileType=Strings.afterLast(".", planFile.getName());
	else
	    outputFileType = "xml"; // default
    }

    protected void checkThatEnoughParametersWereGiven() {
	// Strictly speaking, we could plan without a domain, but
	// we need it when constructing filenames.
	List<String> problems = new LinkedList<String>();
	if (domainName == null)
	    problems.add("a domain must be given");
	// We must have at least one of task and initial plan
	if (task == null && planName == null)
	    problems.add("a task or an initial plan must be specified");
	if (!problems.isEmpty()) {
	    throw new ParameterException
		("Missing parameters: " + Strings.conjunction(problems));
	}
    }

    protected void outputPlan(Plan plan) {
	String savedPlanName = savedPlanName();
	File savedPlanFile = new File(outputDirecoryFile, savedPlanName);
	System.out.println("Have plan " + savedPlanFile);
	if (!testOnly)
	    XML.writeObject(plan, savedPlanFile.getPath());
	planNumber++;
    }

    /**
     * Constructs a file name for a plan based on the domain name
     * plus the task, if one was specified, or else the name of
     * the initial plan.  The parts are separated by dashes, and
     * any spaces in the task are replaced by dashes.  If the
     * initial plan's name begins with the domain name, followed
     * by a dash, that part is removed, so that it doesn't appear twice,
     * and replaced by "starting-from-".
     *
     * @see #taskDescription()
     */
    protected String savedPlanName() {
	// The task may be a multi-element pattern and hence contain
	// spaces; so we replace each space with "-".
	return namePart(domainFile) +
	    "-" + Strings.replace(" ", "-", taskDescription()) +
	    "-plan-" + planNumber + "." + outputFileType;
    }

    /**
     * Returns the name from a File, leaving out the directory and
     * any file type.
     */
    protected String namePart(File file) {
	return Strings.beforeLast(".", file.getName());
    }

    /**
     * Returns a short textual description of the task or problem
     * that's been given to the planner.  
     *
     * <p>The result is the task string, if one was specified,
     * or else the name of the initial plan.  If the initial-plan
     * string begins with the domain name (followed by a dash),
     * those characters are removed from the description, as are
     * any characters after and including any final ".".  The
     * domain part is replaced by "starting-from".  For example,
     * if the domain is "<tt>example</tt>" and initial-plan is
     * "<tt>example-init-plan.xml</tt>", the result will
     * be "<tt>starting-from-init-plan</tt>".</p>
     */
    protected String taskDescription() {
	if (task != null)
	    return task;
	else {
	    String domainPart = namePart(domainFile) + "-";
	    String initPlan =  namePart(planFile);
	    if (initPlan.startsWith(domainPart))
		return "starting-from-" + 
		    Strings.afterFirst(domainPart, initPlan);
	    else
		return initPlan;
	}
    }

}
