/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 16 14:39:18 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2003, 2006, AIAI, University of Edinburgh
 */

package ix.ispace;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.icore.IXAgent;

import ix.iface.util.ToolFrame;
import ix.iface.util.IconImage;
import ix.iface.util.SelectionPanel;
import ix.iface.util.ValueComboBox;
import ix.iface.util.IFUtil;
import ix.iface.util.KeyValueTable;

import ix.ispace.event.*;

import ix.util.*;

/**
 * The I-Space user interface.
 */
public class ISpaceTool implements ActionListener {

    IXAgent agent;
    ContactManager contactManager;

    JFrame frame;
    Container contentPane;
    JTabbedPane tabbedPane = new JTabbedPane();

    RelationPanel relationPanel;
    CapabilityPanel capabilityPanel;

    public ISpaceTool(IXAgent agent) {
	this.agent = agent;
	this.contactManager = agent.getContactManager();
	setupFrame();
    }

    void setupFrame() {		// or define a JFrame subclass??? /\/
	frame = new ToolFrame(agent.getAgentDisplayName() + " I-Space");
	// default is equiv to frame.setLayout(new BorderLayout());
	frame.setIconImage(IconImage.getIconImage(this));
	frame.setJMenuBar(makeMenuBar());

	contentPane = frame.getContentPane();
	contentPane.add(tabbedPane, BorderLayout.CENTER);

	// Relation panel
	relationPanel = new RelationPanel();
	contactManager.addContactListener(relationPanel);
	addTab("Relations", relationPanel); // calls frame.pack() /\/

	// Capability Panel
	capabilityPanel = new CapabilityPanel();
	contactManager.addContactListener(capabilityPanel);
	addTab("Capabilities", capabilityPanel);
	
	// frame.pack();
	// frame.setVisible(true);
    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    // /\/: This is needed so DispatchingStrategy can use the frame
    // when making its SelectionPanel subclass.
    public JFrame getFrame() {
	return frame;
    }

    /**
     * Adds a menu to the menu bar.
     */
    public void addMenu(JMenu menu) {
	frame.getJMenuBar().add(menu);
    }

    /**
     * Adds a new tab and corresponding component
     */
    public void addTab(String name, Component component) {
	tabbedPane.add(name, component);
	frame.pack();
    }

    /**
     * Creates the menu bar
     */
    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(IFUtil.makeMenuItem("Close", this));

	return bar;
    }

    /**
     * Action interpreter
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("I-Space Tool action:", command);
	if (command.equals("Close")) {
	    frame.setVisible(false);
	}
	else
	    throw new ConsistencyException
		("Nothing to do for " + command);
    }

    class RelationPanel extends SelectionPanel implements ContactListener {

	RelationPanel() {
	    super(frame, "Agent", "Relation");

	    // /\/: Can't do this in super constructor because
	    // getCurrentValueMap() will get a null ptr exception
	    // when it tries to refer to contactManager, because
	    // ISpaceTool.this does not yet have a value.
	    // Perhaps think of it like this: if the contact
	    // manager were passed as a parameter to this constructor,
	    // we couldn't do "this.contactManager = ..." until
	    // after the super constructor returned.
	    load();
	}

	// Define the SelectionPanel abstract methods

	protected ValueComboBox makeValueComboBox() {
	    return new RelationComboBox();
	}

	protected Map getCurrentValueMap() {
	    List dataList = contactManager.getAgentData();
	    Map valMap = new TreeMap();	// sorts
	    for (Iterator i = dataList.iterator(); i.hasNext();) {
		AgentData data = (AgentData)i.next();
		valMap.put(data.getName(), data.getRelationship());
	    }
	    return valMap;
	}

	protected void entryDeleted(String name) {
	    contactManager.deleteAgent(name);
	}

	protected void valueChanged(String name, Object value) {
	    contactManager.
		changeRelationship(name, (AgentRelationship)value);
	}

	protected void entryAdded(String name, Object value) {
	    AgentRelationship rel = (AgentRelationship)value;
	    Debug.noteln("Adding agent " + name + " as " + rel);
	    if (contactManager.getAgentData(name) != null) {
		JOptionPane.showMessageDialog
		    (frame,
		     "Agent " + name + " already exists",
		     "Error",
		     JOptionPane.ERROR_MESSAGE);
		return;
	    }
	    contactManager.addAgent(name, rel);
	}

	// ContactListener methods

	public void contactChange(ContactEvent e) {
	    if (e.isNewContact()) {
		String name = e.getNewData().getName();
		AgentRelationship value = e.getNewData().getRelationship();
		this.addTableEntry(name, value);
		return;
	    }
	    // /\/: Else for now just reload.  Eventually need to figure
	    // out just what has to change and warn the user if it affects
	    // any uncommitted changes made in the panel.
	    this.reload();
	}

    }

    /** A combo-box for selecting {@link AgentRelationship}s. */
    static class RelationComboBox extends ValueComboBox {

	public RelationComboBox() {
	    super();
	    for (Iterator i = AgentRelationship.values().iterator();
		 i.hasNext();) {
		AgentRelationship r = (AgentRelationship)i.next();
		addItem(Strings.capitalize(r.toString()));
	    }
	    addItem("Delete");
	    setSelectedValue(AgentRelationship.CONTACT);
	}

	/** Returns an {@link AgentRelationship}. */
	public Object getSelectedValue() {
	    String name = (String)getSelectedItem();
	    return name == "Delete"
		? null
		: AgentRelationship.valueOf(name.toLowerCase());
	}

	/** The item must be a {@link AgentRelationship}. */
	public void setSelectedValue(Object item) {
	    AgentRelationship r = (AgentRelationship)item;
	    setSelectedItem(Strings.capitalize(r.toString()));
	}

    }

    class CapabilityPanel extends JPanel implements ContactListener {

	CapabilityTable capabilityTable;

	CapabilityPanel() {

	    this.capabilityTable = new CapabilityTable(getCapabilityMap());

	    setLayout(new BorderLayout());
	    JScrollPane scroll = new JScrollPane(capabilityTable.getJTable());
	    add(scroll, BorderLayout.CENTER);

	}

	Map getCapabilityMap() {
	    List dataList = contactManager.getAgentData();
	    Map map = new HashMap();
	    for (Iterator i = dataList.iterator(); i.hasNext();) {
		AgentData data = (AgentData)i.next();
		map.put(data.getName(), data.getCapabilities());
	    }
	    return map;
	}

	// ContactListener methods

	public void contactChange(ContactEvent e) {
	    if (e.isNewContact() || e.isCapabilityChange()) {
		AgentData dat = e.getNewData();
		capabilityTable
		    .recordNewValue(dat.getName(), dat.getCapabilities());
	    }
	    else if (e.isDeletion()) {
		capabilityTable.deleteEntry(e.getOldData().getName());
	    }
	}

    }

    class CapabilityTable extends KeyValueTable {

	CapabilityTable(Map initialMap) {
	    super(initialMap, "Agent", "Verb Capabilities");
	    this.keySortComparator = new Comparator() {
		public int compare(Object k1, Object k2) {
		    return ((String)k1).compareTo(((String)k2));
		}
	    };
	}

	protected String keyToString(Object key) {
	    return (String)key;
	}

	protected String valueToString(Object value) {
	    return Collect.elementsToString((List)value, new Function1() {
		public Object funcall(Object a) {
		    return ((VerbCapability)a).getVerb();
		}
	    });
	}

	protected RowPopupMenu makePopupMenu() {
	    return new CapabilityPopupMenu();
	}

	protected void doPopupAction(ActionEvent event, int row, Object key) {
	    String command = event.getActionCommand();
	    Debug.expectEquals("Edit Capabilities", command);
	    String caps = valueToString(getValue(key));
	    String spec = Util.showInputDialog
		(getJTable(),
		 "Edit Capabilities of " + key,
		 caps);
	    if (spec == null)
		return;		// user cancelled
	    List verbs = Strings.breakAt(",", spec);
	    List capabilities = new LinkedList();
	    for (Iterator i = verbs.iterator(); i.hasNext();) {
		String verb = (String)i.next();
		capabilities.add(new VerbCapability(verb.trim()));
	    }
	    contactManager.setCapabilities((String)key, capabilities);
	}

	class CapabilityPopupMenu extends RowPopupMenu {
	    CapabilityPopupMenu() {
		super();
		add(makeMenuItem("Edit Capabilities"));
	    }
	}

    }

}
