package ix.isim.util;

/**
 * <p>This is a Red-Black-Tree-based implementation of a sorted map, mapping
 * <code>long</code>s to objects. This class guarantees that the map will be
 * in ascending key order and the <code>Iterator</code> returned by
 * <code>values()</code> will list the values (not the keys) in ascending key
 * order. </p>
 *
 * <p>This implementation provides guaranteed log(n) time cost for the
 * <code>containsKey(...)</code>, <code>get(...)</code>, <code>put(...)</code>
 * and <code>remove(...)</code> operations. Algorithms are adaptations of those
 * in Cormen, Leiserson, and Rivest's <i>Introduction to Algorithms</i>. </p>
 *
 * <p> This implementation should essentially behave like the one in the
 * java.util package. The main difference is that keys are <code>long</code>s
 * here which are not ojbects. The use of <code>long</code>s rather than
 * <code>Double</code>s greatly improves the efficiency of the implementation.
 * </p>
 *
 * @author Gerhard Wickler, based on the <code>TreeMap</code> implementation by
 * Josh Bloch and Doug Lea
 * @version 24.Mar.2005
 */

public class LongKeyTreeMap {

    /////////////////////
    // class variables //
    /////////////////////

    /** for tree balancing **/
    static private final boolean RED = false;
    /** for tree balancing **/
    static private final boolean BLACK = true;

    ////////////////////////
    // instance variables //
    ////////////////////////

    // protected //

    /** the root of the tree **/
    protected Entry root = null;

    // private //

    /** the number of entries in the tree **/
    private int size = 0;
    /** the number of structural modifications to the tree **/
    private int modCount = 0;

    /////////////////////////
    // public constructors //
    /////////////////////////

    /**
     * <p>This constructor creates a new, empty map. </p>
     */
    public LongKeyTreeMap() {
    }

    /////////////
    // cloning //
    /////////////

    /**
     * <p>This class does not support cloning and an exception will be thrown if
     * this method is called. </p>
     *
     * @return nothing
     * @exception CloneNotSupportedException will be thrown
     */
    protected Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    ///////////////////////////////
    // public instance functions //
    ///////////////////////////////

    // own (local) //

    /**
     * <p>This function returns the number of key-value mappings in this map.
     * </p>
     *
     * @return the number of key-value mappings in this map
     */
    public int size() {
        return size;
    }

    /**
     * <p>This function returns <code>true</code> if this map contains a mapping
     * for the specified key. </p>
     *
     * @param key the key whose presence in this map is to be tested
     *
     * @return <code>true</code> if this map contains a mapping for the
     * specified key
     */
    public boolean containsKey(long key) {
        return getEntry(key) != null;
    }

    /**
     * <p>This function returns the value to which this map maps the specified
     * key. It returns <code>null</code> if the map contains no mapping for this
     * key. A return value of <code>null</code> does not <i>necessarily</i>
     * indicate that the map contains no mapping for the key; it's also possible
     * that the map explicitly maps the key to <code>null</code>. The
     * <code>containsKey(...)</code> operation may be used to distinguish these
     * two cases. </p>
     *
     * @param key key whose associated value is to be returned
     * @return the value to which this map maps the specified key, or
     * <code>null</code> if the map contains no mapping for the key
     */
    public Object get(long key) {
        Entry p = getEntry(key);
        return (p == null ? null : p.value);
    }

    /**
     * <p>This function returns the first (lowest) key currently in this sorted
     * map. </p>
     *
     * @return the first (lowest) key currently in this sorted map
     * @throws NoSuchElementException if the map is empty
     */
    public long firstKey() throws java.util.NoSuchElementException {
        if (root == null) {
            throw new java.util.NoSuchElementException();
        }
        return firstEntry().key;
    }

    /**
     * <p>This function returns the last (highest) key currently in this sorted
     * map. </p>
     *
     * @return the last (highest) key currently in this sorted map
     * @throws NoSuchElementException if the map is empty
     */
    public long lastKey() throws java.util.NoSuchElementException {
        if (root == null) {
            throw new java.util.NoSuchElementException();
        }
        return lastEntry().key;
    }

    /**
     * <p>This function associates the specified value with the specified key in
     * this map. If the map previously contained a mapping for this key, the old
     * value is replaced. A <code>null</code> return may indicate that the map
     * previously associated <code>null</code> with the specified key. </p>
     *
     * @param key the key with which the specified value is to be associated
     * @param value the value to be associated with the specified key
     * @return previous value associated with specified key, or
     * <code>null</code> if there was no mapping for the key
     */
    public Object put(long key, Object value) {
        Entry t = root;

        if (t == null) {
            incrementSize();
            root = new Entry(key, value, null);
            return null;
        }

        while (true) {
            if (key == t.key) {
                return t.setValue(value);
            }
            else if (key < t.key) {
                if (t.left != null) {
                    t = t.left;
                }
                else {
                    incrementSize();
                    t.left = new Entry(key, value, t);
                    fixAfterInsertion(t.left);
                    return null;
                }
            }
            else
            /* (key > t.key) */
            {
                if (t.right != null) {
                    t = t.right;
                }
                else {
                    incrementSize();
                    t.right = new Entry(key, value, t);
                    fixAfterInsertion(t.right);
                    return null;
                }
            }
        }
    }

    /**
     * <p>This function removes the mapping for this key from this map if it was
     * present. A <code>null</code> return may indicate that the map previously
     * associated <code>null</code> with the specified key. </p>
     *
     * @param key the key to the mapping that is to be removed
     * @return the previous value associated with specified key, or
     * <code>null</code> if there was no mapping for the key
     */
    public Object remove(long key) {
        Entry p = getEntry(key);
        if (p == null) {
            return null;
        }

        Object oldValue = p.value;
        deleteEntry(p);
        return oldValue;
    }

    /**
     * <p>This function removes all mappings from this map. </p>
     */
    public void clear() {
        modCount++;
        size = 0;
        root = null;
    }

    /**
     * <p>This function returns an iterator for the objects in this map. They
     * will be returned in the (ascending) order of their keys. </p>
     *
     * @return an ordered <code>Iterator</code> of the values (not keys) in this
     * map
     */
    public java.util.Iterator values() {
        return new LongKeyTreeMapIterator();
    }

    ////////////////////////////////
    // private instance functions //
    ////////////////////////////////

    // own (local) //

    /**
     * <p>This function increments the size counter for this map and notes the
     * modification. </p>
     */
    private void incrementSize() {
        modCount++;
        size++;
    }

    /**
     * <p>This function decrements the size counter for this map and notes the
     * modification. </p>
     */
    private void decrementSize() {
        modCount++;
        size--;
    }

    /**
     * <p>This function returns this map's entry for the given key, or
     * <code>null</code> if the map does not contain an entry for the key. </p>
     *
     * @param key the key for which the entry is sought
     * @return this map's entry for the given key, or <code>null</code> if the
     * map does not contain an entry for the key
     */
    private Entry getEntry(long key) {
        Entry p = root;
        while (p != null) {
            if (key == p.key) {
                return p;
            }
            p = (key < p.key) ? p.left : p.right;
        }
        return null;
    }

    /**
     * <p>This function returns the first entry in this map.  It returns
     * <code>null</code> if the map is empty. </p>
     *
     * @return the <code>Entry</code> with the smallest key
     */
    private Entry firstEntry() {
        Entry p = root;
        if (p != null) {
            while (p.left != null) {
                p = p.left;
            }
        }
        return p;
    }

    /**
     * <p>This function returns the last entry in this map.  It returns
     * <code>null</code> if the map is empty. </p>
     *
     * @return the <code>Entry</code> with the largest key
     */
    private Entry lastEntry() {
        Entry p = root;
        if (p != null) {
            while (p.right != null) {
                p = p.right;
            }
        }
        return p;
    }

    /**
     * <p>This function deletes given entry from this map. If necessary, it also
     * rebalances the tree. </p>
     *
     * @param p the <code>Entry</code> to be deleted
     */
    private void deleteEntry(Entry p) {
        decrementSize();

        // If strictly internal, first swap position with successor.
        if (p.left != null && p.right != null) {
            Entry s = successor(p);
            swapPosition(s, p);
        }

        // Start fixup at replacement node, if it exists.
        Entry replacement = (p.left != null ? p.left : p.right);

        if (replacement != null) {
            // Link replacement to parent
            replacement.parent = p.parent;
            if (p.parent == null) {
                root = replacement;
            }
            else if (p == p.parent.left) {
                p.parent.left = replacement;
            }
            else {
                p.parent.right = replacement;
            }

            // Null out links so they are OK to use by fixAfterDeletion.
            p.left = p.right = p.parent = null;

            // Fix replacement
            if (p.color == BLACK) {
                fixAfterDeletion(replacement);
            }
        }
        else if (p.parent == null) { // return if we are the only node.
            root = null;
        }
        else { //  No children. Use self as phantom replacement and unlink.
            if (p.color == BLACK) {
                fixAfterDeletion(p);
            }

            if (p.parent != null) {
                if (p == p.parent.left) {
                    p.parent.left = null;
                }
                else if (p == p.parent.right) {
                    p.parent.right = null;
                }
                p.parent = null;
            }
        }
    }

    /**
     * <p>This function swaps the two give sub-trees which involve quite a few
     * special cases. </p>
     *
     * @param x the first sub-tree
     * @param y the second sub-tree
     */
    private void swapPosition(Entry x, Entry y) {
        // Save initial values.
        Entry px = x.parent, lx = x.left, rx = x.right;
        Entry py = y.parent, ly = y.left, ry = y.right;
        boolean xWasLeftChild = px != null && x == px.left;
        boolean yWasLeftChild = py != null && y == py.left;

        // Swap, handling special cases of one being the other's parent.
        if (x == py) { // x was y's parent
            x.parent = y;
            if (yWasLeftChild) {
                y.left = x;
                y.right = rx;
            }
            else {
                y.right = x;
                y.left = lx;
            }
        }
        else {
            x.parent = py;
            if (py != null) {
                if (yWasLeftChild) {
                    py.left = x;
                }
                else {
                    py.right = x;
                }
            }
            y.left = lx;
            y.right = rx;
        }

        if (y == px) { // y was x's parent
            y.parent = x;
            if (xWasLeftChild) {
                x.left = y;
                x.right = ry;
            }
            else {
                x.right = y;
                x.left = ly;
            }
        }
        else {
            y.parent = px;
            if (px != null) {
                if (xWasLeftChild) {
                    px.left = y;
                }
                else {
                    px.right = y;
                }
            }
            x.left = ly;
            x.right = ry;
        }

        // Fix children's parent pointers
        if (x.left != null) {
            x.left.parent = x;
        }
        if (x.right != null) {
            x.right.parent = x;
        }
        if (y.left != null) {
            y.left.parent = y;
        }
        if (y.right != null) {
            y.right.parent = y;
        }

        // Swap colors
        boolean c = x.color;
        x.color = y.color;
        y.color = c;

        // Check if root changed
        if (root == x) {
            root = y;
        }
        else if (root == y) {
            root = x;
        }
    }

    /*
     * Balancing operations.
     *
     * Implementations of rebalancings during insertion and deletion are
     * slightly different than the CLR version.  Rather than using dummy
     * nilnodes, we use a set of accessors that deal properly with null.  They
     * are used to avoid messiness surrounding nullness checks in the main
     * algorithms. (all from CLR)
     */

    private void rotateLeft(Entry p) {
        Entry r = p.right;
        p.right = r.left;
        if (r.left != null) {
            r.left.parent = p;
        }
        r.parent = p.parent;
        if (p.parent == null) {
            root = r;
        }
        else if (p.parent.left == p) {
            p.parent.left = r;
        }
        else {
            p.parent.right = r;
        }
        r.left = p;
        p.parent = r;
    }

    private void rotateRight(Entry p) {
        Entry l = p.left;
        p.left = l.right;
        if (l.right != null) {
            l.right.parent = p;
        }
        l.parent = p.parent;
        if (p.parent == null) {
            root = l;
        }
        else if (p.parent.right == p) {
            p.parent.right = l;
        }
        else {
            p.parent.left = l;
        }
        l.right = p;
        p.parent = l;
    }

    private void fixAfterInsertion(Entry x) {
        x.color = RED;

        while (x != null && x != root && x.parent.color == RED) {
            if (parentOf(x) == leftOf(parentOf(parentOf(x)))) {
                Entry y = rightOf(parentOf(parentOf(x)));
                if (colorOf(y) == RED) {
                    setColor(parentOf(x), BLACK);
                    setColor(y, BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    x = parentOf(parentOf(x));
                }
                else {
                    if (x == rightOf(parentOf(x))) {
                        x = parentOf(x);
                        rotateLeft(x);
                    }
                    setColor(parentOf(x), BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    if (parentOf(parentOf(x)) != null) {
                        rotateRight(parentOf(parentOf(x)));
                    }
                }
            }
            else {
                Entry y = leftOf(parentOf(parentOf(x)));
                if (colorOf(y) == RED) {
                    setColor(parentOf(x), BLACK);
                    setColor(y, BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    x = parentOf(parentOf(x));
                }
                else {
                    if (x == leftOf(parentOf(x))) {
                        x = parentOf(x);
                        rotateRight(x);
                    }
                    setColor(parentOf(x), BLACK);
                    setColor(parentOf(parentOf(x)), RED);
                    if (parentOf(parentOf(x)) != null) {
                        rotateLeft(parentOf(parentOf(x)));
                    }
                }
            }
        }
        root.color = BLACK;
    }

    private void fixAfterDeletion(Entry x) {
        while (x != root && colorOf(x) == BLACK) {
            if (x == leftOf(parentOf(x))) {
                Entry sib = rightOf(parentOf(x));

                if (colorOf(sib) == RED) {
                    setColor(sib, BLACK);
                    setColor(parentOf(x), RED);
                    rotateLeft(parentOf(x));
                    sib = rightOf(parentOf(x));
                }

                if (colorOf(leftOf(sib)) == BLACK &&
                    colorOf(rightOf(sib)) == BLACK) {
                    setColor(sib, RED);
                    x = parentOf(x);
                }
                else {
                    if (colorOf(rightOf(sib)) == BLACK) {
                        setColor(leftOf(sib), BLACK);
                        setColor(sib, RED);
                        rotateRight(sib);
                        sib = rightOf(parentOf(x));
                    }
                    setColor(sib, colorOf(parentOf(x)));
                    setColor(parentOf(x), BLACK);
                    setColor(rightOf(sib), BLACK);
                    rotateLeft(parentOf(x));
                    x = root;
                }
            }
            else { // symmetric
                Entry sib = leftOf(parentOf(x));

                if (colorOf(sib) == RED) {
                    setColor(sib, BLACK);
                    setColor(parentOf(x), RED);
                    rotateRight(parentOf(x));
                    sib = leftOf(parentOf(x));
                }

                if (colorOf(rightOf(sib)) == BLACK &&
                    colorOf(leftOf(sib)) == BLACK) {
                    setColor(sib, RED);
                    x = parentOf(x);
                }
                else {
                    if (colorOf(leftOf(sib)) == BLACK) {
                        setColor(rightOf(sib), BLACK);
                        setColor(sib, RED);
                        rotateLeft(sib);
                        sib = leftOf(parentOf(x));
                    }
                    setColor(sib, colorOf(parentOf(x)));
                    setColor(parentOf(x), BLACK);
                    setColor(leftOf(sib), BLACK);
                    rotateRight(parentOf(x));
                    x = root;
                }
            }
        }

        setColor(x, BLACK);
    }

    /////////////////////
    // class functions //
    /////////////////////

    // private //

    /**
     * <p>This function returns the successor of the specified entry, or
     * <code>null</code> if no such entry exists. </p>
     *
     * @param t the predecessor of the sought <code>Entry</code>
     * @return the <code>Entry</code> which succeeds the given one
     */
    static private Entry successor(Entry t) {
        if (t == null) {
            return null;
        }
        else if (t.right != null) {
            Entry p = t.right;
            while (p.left != null) {
                p = p.left;
            }
            return p;
        }
        else {
            Entry p = t.parent;
            Entry ch = t;
            while (p != null && ch == p.right) {
                ch = p;
                p = p.parent;
            }
            return p;
        }
    }

    static private boolean colorOf(Entry p) {
        return (p == null ? BLACK : p.color);
    }

    static private Entry parentOf(Entry p) {
        return (p == null ? null : p.parent);
    }

    static private void setColor(Entry p, boolean c) {
        if (p != null) {
            p.color = c;
        }
    }

    static private Entry leftOf(Entry p) {
        return (p == null) ? null : p.left;
    }

    static private Entry rightOf(Entry p) {
        return (p == null) ? null : p.right;
    }

    ///////////////////
    // inner classes //
    ///////////////////

    class LongKeyTreeMapIterator
            implements java.util.Iterator {
        protected Entry next;

        public LongKeyTreeMapIterator() {
            next = firstEntry();
        }

        public boolean hasNext() {
            return next != null;
        }

        public Object next() {
            if (next == null) {
                throw new java.util.NoSuchElementException();
            }
            Object result = next.value;
            next = successor(next);
            return result;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * <p>A node in the tree.  Doubles as a means to pass key-value pairs back
     * to user. </p>
     */
    static class Entry {
        protected long key;
        protected Object value;
        protected Entry left = null;
        protected Entry right = null;
        protected Entry parent;
        protected boolean color = BLACK;

        /**
         * <p>Make a new cell with given key, value, and parent, and with
         * <tt>null</tt> child links, and BLACK color. </p>
         *
         * @param key the key of the mapping
         * @param value the value to which the key is to be mapped
         * @param parent the parent <code>Entry</code> in the tree
         */
        Entry(long key, Object value, Entry parent) {
            this.key = key;
            this.value = value;
            this.parent = parent;
        }

        public Object setValue(Object newVal) {
            Object oldValue = value;
            value = newVal;
            return oldValue;
        }

        public boolean equals(Object o) {
            Entry e = (Entry) o;
            return (key == e.key) &&
                    (value == null ? e.value == null : value.equals(e.value));
        }

        public int hashCode() {
            int keyHash = (int) key;
            int valueHash = (value == null ? 0 : value.hashCode());
            return keyHash ^ valueHash;
        }

        public String toString() {
            return key + "=" + value;
        }
    }

}
