/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Apr 30 04:42:37 2005 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.iscript;

import java.util.*;

import ix.util.*;
import ix.util.lisp.Symbol;

/**
 * I-Script expression factory.
 */
public class IScript {

    public IScript() {
    }

    public Literal makeLiteral(Object value) {
	Literal lit = new Literal();
	lit.setValue(value);
	return lit;
    }

    public VarRef makeVarRef(Symbol name) {
	return makeVarRef(Name.valueOf(name));
    }

    public VarRef makeVarRef(Name name) {
	VarRef ref = new VarRef();
	ref.setName(name);
	return ref;
    }

    public Call makeCall(Expression fnExpr, ListOfExpression argExprs) {
	Call call = new Call();
	call.setFunction(fnExpr);
	call.setArguments(argExprs);
	return call;
    }

    public Assignment makeAssignment(Symbol name, Expression valExpr) {
	return makeAssignment(Name.valueOf(name), valExpr);
    }

    public Assignment makeAssignment(Name name, Expression valExpr) {
	Assignment a = new Assignment();
	a.setTo(name);
	a.setValue(valExpr);
	return a;
    }

    public If makeIf(Expression test, Expression ifTrue, Expression ifFalse) {
	If i = new If();
	i.setTest(test);
	i.setIfTrue(ifTrue);
	i.setIfFalse(ifFalse);
	return i;
    }

    public Sequence makeSequence(ListOfExpression exprs) {
	Sequence seq = new Sequence();
	seq.setOf(exprs);
	return seq;
    }

    public And makeAnd(ListOfExpression exprs) {
	And and = new And();
	and.setOf(exprs);
	return and;
    }

    public Or makeOr(ListOfExpression exprs) {
	Or or = new Or();
	or.setOf(exprs);
	return or;
    }

    public Let makeLet(Symbol[] vars, ListOfExpression valExprs,
		       Expression body) {
	ListOfBinding bindings = new LinkedListOfBinding();
	for (ListIterator i = valExprs.listIterator(); i.hasNext();) {
	    Name n = Name.valueOf(vars[i.nextIndex()]);
	    Expression e = (Expression)i.next();
	    bindings.add(makeBinding(n, e));
	}
	return makeLet(bindings, body);
    }

    public Let makeLet(ListOfBinding bindings, Expression body) {
	Let let = new Let();
	let.setBindings(bindings);
	let.setIn(body);
	return let;
    }

    public Binding makeBinding(Name name, Expression valExpr) {
	Binding b = new Binding();
	b.setName(name);
	b.setValue(valExpr);
	return b;
    }

    public Lambda makeLambda(Symbol[] parameters, Expression body) {
	ListOfName names = new LinkedListOfName();
	for (int i = 0; i < parameters.length; i++) {
	    names.add(Name.valueOf(parameters[i]));
	}
	return makeLambda(names, body);
    }

    public Lambda makeLambda(ListOfName parameters, Expression body) {
	Lambda l = new Lambda();
	l.setParameters(parameters);
	l.setIn(body);
	return l;
    }

    public While makeWhile(Expression test, Expression body) {
	While w = new While();
	w.setTest(test);
	w.setRepeat(body);
	return w;
    }

}
