/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed May 31 03:22:18 2006 by Jeff Dalton
 * Copyright: (c) 2005, 2006, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;

import ix.icore.process.PNode;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;

public class TechnicalPlanEvalManager extends PlanEvalManager {

    protected List planEvaluators = new LinkedList();

    public TechnicalPlanEvalManager() {
	super();
    }

    public void addPlanEvaluators() {
	addPlanEvaluator(new IssueCountEval());
	addPlanEvaluator(new NodeCountEval());
	addPlanEvaluator(new LongestPathLengthEval());
	addPlanEvaluator(new ObjectUseEval());
	addPlanEvaluator(new ObjectCountEval());
	addPlanEvaluator(new ChecksimEval());
    }

    public static class TaskDescriptionEval extends PlanEvaluator {

	public TaskDescriptionEval() {
	    super("Task");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    return new SimplePlanEvaluation(".");
	}

    }

    public static class IssueCountEval extends PlanEvaluator {

	public IssueCountEval() {
	    super("issues in plan");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    int n = ip2.getController().getIssueAgenda().getItems().size();
	    return new SimplePlanEvaluation(n);
	}

    }

    public static class NodeCountEval extends PlanEvaluator {

	public NodeCountEval() {
	    super("activities in plan");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    int n = ip2.getModelManager().getNodes().size();
	    return new SimplePlanEvaluation(n);
	}

    }

    public static class LongestPathLengthEval extends PlanEvaluator {

	public LongestPathLengthEval() {
	    super("longest path length");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    List nodeEnds = mm.getNodeEnds();
	    int n = new SlipStats(null).calcLongestPathLength(nodeEnds);
	    return new SimplePlanEvaluation(n);
	}

    }

    public static class ObjectUseEval extends PlanEvaluator {

	public ObjectUseEval() {
	    super("objects used in plan");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    Set objects = new HashSet();
	    for (Iterator i = mm.getNodes().iterator(); i.hasNext();) {
		PNode pnode = (PNode)i.next();
		collectObjects(objects, mm.getNodeConditions(pnode));
		collectObjects(objects, mm.getNodeEffects(pnode));
	    }
	    return new SimplePlanEvaluation(objects.size());
	}

	void collectObjects(Set objects, List constraints) {
	    // /\/: It's irritating that conditions are Constraint objects
	    // while effects are still just PatternAssignments.
	    if (constraints == null) return;
	    for (Iterator i = constraints.iterator(); i.hasNext();) {
		Object cond_or_effect = i.next();
		PatternAssignment pv;
		if (cond_or_effect instanceof PatternAssignment)
		    pv = (PatternAssignment)cond_or_effect;
		else {
		    Constraint c = (Constraint)cond_or_effect;
		    if (c.getType() == Refinement.S_WORLD_STATE)
			pv = c.getPatternAssignment();
		    else
			continue;
		}
		Object obj = pv.getPattern().get(1);
		if (obj != Lisp.NIL)
		    objects.add(obj);
	    }
	}

    }

    public static class ObjectCountEval extends PlanEvaluator {

	public ObjectCountEval() {
	    super("objects in state");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    // Builds a set of the objects to see how many different
	    // ones there are.
	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    Map map = mm.getWorldStateMap();
	    Set objects = new HashSet();
	    for (Iterator i = map.keySet().iterator(); i.hasNext();) {
		LList pattern = (LList)i.next();
		if (!pattern.cdr().isEmpty()) {
		    // An "object" is anything that appears as
		    // the second element of a pattern (ie, index = 1).
		    Object a = pattern.get(1);
		    objects.add(a);
		}
	    }
	    return new SimplePlanEvaluation(objects.size());
	}

    }

    public static class ChecksimEval extends PlanEvaluator {

	public ChecksimEval() {
	    super("simulated execution");
	}

	public PlanEvaluation evaluatePlan(Ip2 ip2, String optName) {
	    return new ChecksimPlanEvaluation(ip2, optName);
	}

    }

}
