/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Jul 10 04:07:39 2007 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Satisfies a node's preconditions.
 */
class SlipSatisfyConds implements Slip.Step {

    Slip slip;
    AgendaItem item;
    ListOfConstraint conds;

    SlipSatisfyConds(Slip slip) { // for cond sat in SlipFindExecutable /\/
	this.slip = slip;
    }

    SlipSatisfyConds(Slip slip, PNode node, ListOfConstraint conds) {
	this.slip = slip;
	this.item = (AgendaItem)node; // should use node instead /\/
	this.conds = conds;
    }

    public void run() {
	IPlanModelManager modelManager = slip.MM();

	// Make sure we respect the ways the filter conds
	// might bind now that the refinement has been
	// instantiated.
	// Don't need to pass anything in the env this time.
	// The model manager returns a list of envs
	List envs = modelManager.evalFilters(conds, new MatchEnv());
	checkCondsSatisfied(envs);
	// The envs tell us all the ways we could satisfy
	// all the filters by binding.  Each env is a way to
	// satisy all the filters.
	if (envs.size() == 1) {
	    // Only one way to satisfy the conds, so we can just
	    // do it.
	    satisfyCondsUsing((MatchEnv)envs.get(0));
	    adjustStatus(item);
	    slip.setNextStep(new SlipFindExecutable(slip));
	    return;
	}
	// Multiple envs.
	// /\/: Here we rely on the envs being a LinkedList so that
	// we can use them as a queue in alternatives.
	LinkedList condEnvs = (LinkedList)envs;
	MatchEnv e1 = (MatchEnv)condEnvs.removeFirst();
	slip.postAlternative(new ResumeSatisfaction(item, conds, condEnvs));
	satisfyCondsUsing(e1);
	adjustStatus(item);
	slip.setNextStep(new SlipFindExecutable(slip));
    }

    private void satisfyCondsUsing(MatchEnv env) {
	try {
	    slip.MM().satisfyConds(item, conds, env);
	}
	catch (FailureException fail) {
	    throw slip.poison(fail);
	}
    }

    void checkCondsSatisfied(List envs) {
	if (envs.isEmpty()) {
	    // We should get here only if the conds can all be
	    // satisfied.  Otherwise, the expander should have
	    // been an expand-later one, and we should have gone
	    // to SlipAchieveConds.
	    throw new ConsistencyException
		("Cannot match filter conditions", conds);
	    // throw slip.poison("cannot match filter conditions");
	}
    }

    void adjustStatus(AgendaItem item) {
	// slip.setStatus(item.getBegin(), Status.COMPLETE);
	slip.execBegin(item.getBegin());
    }

    class ResumeSatisfaction extends Alternative {

	AgendaItem item;
	ListOfConstraint conds;
	LinkedList condEnvs;

	ResumeSatisfaction(AgendaItem item, ListOfConstraint conds,
			   LinkedList condEnvs) {
	    this.item = item;
	    this.conds = conds;
	    this.condEnvs = condEnvs;
	}

	@Override
	public boolean isLocalChoice() {
	    return true;
	}

	public void run() {
	    MatchEnv e1 = (MatchEnv)condEnvs.removeFirst();
	    if (!condEnvs.isEmpty())
		slip.postAlternative
		    (new ResumeSatisfaction(item, conds, condEnvs));
	    slip.MM().satisfyConds(item, conds, e1);
	    adjustStatus(item);
	    // Now we're back to finding the next executable.
	    slip.setNextStep(new SlipFindExecutable(slip));
	}

    }

}
