/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Feb 14 18:21:28 2008 by Jeff Dalton
 * Copyright: (c) 2004 - 2005, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.process.*;
import ix.icore.domain.*;	// for domain utils
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

/**
 * Semi-linear planner.
 */
public class Slip extends PlannerBase implements ServiceSymbols {

    IPlanModelManager iplanMM;

    DomainAnalyser analysis;
    boolean useDomainAnalysis = true;

    boolean pickAlternativesDepthFirst = false;

    LListCollector alternatives = new LListCollector();

    ContextBoolean contextIsPoisoned = new ContextBoolean(false);

    ContextMap nodeEndStatusMap = new ContextHashMap();

    ContextMap isWaiting = new ContextHashMap();

    ContextMultiMap unwaitTable = new ContextMultiHashMap();

    LLQueue<Object[]> history = new LLQueue<Object[]>();

    Map savedInitialWorldState;

    SlipStats stats = new SlipStats(this);

    ContextGensym.Generator objectNameGenerator =
	new ContextGensym.Generator();

    int stepLimit = 100;
    int hardStepLimit = 500;

    SlipSingleStepper stepper = null;
    boolean singleStep = false;

    public Slip(boolean standAlone) {
	super(standAlone);	// true means make this "the" agent
    }

    IPlanModelManager MM() {
	return iplanMM;
    }

    /**
     * Standalone main program.  One plan in, one plan out;
     * To specify the initial plan, use the <tt>plan</tt>
     * command-line argument.  The result will be written to
     * the file specified by the <tt>output</tt> argument.
     * If no output is specified, the user is asked by a dialog.
     */
    public static void main(String[] argv) {
	Util.printGreeting("I-Plan");
	Parameters.setIsInteractive(false);
	Slip s = new Slip(true);
	s.mainStartup(argv);
	s.plan();
	s.outputPlan();
	s.exit();
    }

    /**
     * Completes setup and initialization.
     */
    public void startup() {
	super.startup();
	iplanMM = (IPlanModelManager)modelManager;
    }

    protected void processCommandLineArguments() {
	super.processCommandLineArguments();
	if (Parameters.haveParameter("step-limit"))
	    setStepLimit(Parameters.getInt("step-limit", stepLimit));
	singleStep = Parameters.getBoolean("single-step", singleStep);
	if (singleStep)
	    stepper = new SlipSingleStepper(this);
	useDomainAnalysis =
            Parameters.getBoolean("use-domain-analysis", useDomainAnalysis);
        if (useDomainAnalysis)
            analysis = new DomainAnalyser(domain);
    }

    /**
     * Called to restore the initial state.
     */
    public void reset() {
	super.reset();
	alternatives.clear();
	nodeEndStatusMap.clearCompletely();
	isWaiting.clearCompletely();
	unwaitTable.clearCompletely();
	history.clearCompletely();
	stats = new SlipStats(this);
    }

    public PlanStats getStatistics() {
	return stats;
    }

    Symbol generateObjectName(Symbol typeName) {
	return objectNameGenerator.nextSymbol(typeName.toString());
    }

    public void setStepLimit(int limit) {
	stepLimit = limit;
	hardStepLimit = Math.max(hardStepLimit, 3 * stepLimit);
    }

    public void setDomain(Domain dom) {
	super.setDomain(dom);
	if (useDomainAnalysis || true) { //\/ needed anyway in SlipAchieveConds
            analysis = new DomainAnalyser(domain);
	    analysis.analyse();
	}
	pickAlternativesDepthFirst = preferDepthFirst();
    }

    /**
     * Install any built-in issue and activity handlers.
     */ 
    protected void addHandlers() {
	// We have none and do things a different way.
    }

    /*
     * Finally, the actual planner
     */

    // Here's a rough sketch of the basic idea:

    // If all nodes are complete, we're done; otherwise:
    // Find an executable node, N, that either (a) can be expanded now
    //   or (b) has no refinements that apply and all conds satisfiable.
    //   That it can be expanded now means using a refinement
    //   whose conds can all be satisfied now.  If N has already
    //   been expanded when we consider it, treat it as if no refinements
    //   apply.
    // (Really this is done with node-ends, where expansion is considered
    //   only for BEGIN ends.)
    // If no such node can be found, poison; otherwise:
    // Save an alt that would find another node instead.
    // If N could be expanded, expand it after saving an alt
    //   that could expand it using a different refinement.
    // Pick a way to satisfy N's conditions, saving an alt that
    //   would satisfy them different ways.
    // Go back to the top.

    // When there's a poison, pick an alt and resume at that point.
    // If we can't pick one, because there are none, no plan can
    // be found.

    // The above is modified to handle "achieve" in a limited way.
    // A node N can be expanded even using refinements whose conds
    // cannot all be satisfied at that time.  New nodes can be introduced
    // to satisfy the conds.  This can greatly increase the search
    // space.  However, the achieving nodes aren't introduced until
    // we consider N for execution, which limits how early they can
    // occur in the plan.  This makes them significantly more
    // limited than in O-Plan.

    static interface Step extends Runnable { }

    Step invalidStep = new Step() {
	public void run() {
	    throw new ConsistencyException("Invalid algorithm step");
	}
	public String toString() { return "INVALID STEP"; }
    };

    private Step nextStep = invalidStep;

    void setNextStep(Step s) {
	nextStep = s;
    }

    Step testStep = new Step() {
	// A loop that sets the status of an executable node-ends
	// to complete each time around.  It's just a way to
	// test that this will get us through all the node-ends.
	public void run() {
	    PNodeEnd next = findExecutable();
	    if (next == null) {
		// nextStep = null;
		throw new HavePlanException();
	    }
	    else {
		Debug.noteln("Executing:", next);
		setStatus(next, Status.COMPLETE);
		nextStep = this;
	    }
	}
    };

    public void plan() {
	Debug.noteln("Plan requested from", this);
	history("Plan");
	if (useDomainAnalysis)
	    analysis.analyseIfNeeded();
	Map ws = iplanMM.getWorldStateMap();
	// /\/: We need a snapshot, not the whole context-layered map.
	// Perhaps the MM's getWorldStateMap() should do this for us.
	savedInitialWorldState = new StableHashMap(ws);
	// allComputeStatus(iplanMM.getNodeEnds());
	computeProperStatusValues();
	initStatusTable();
	nextStep = new SlipFindExecutable(this);
	planLoop();
    }

    public void replan() {
	Debug.noteln("Replan requested from", this);
	history("Replan");
	stats = new SlipStats(this);
	// /\/: Can't just call pickAlternative(), because
	// it could throw a NoPlanException, and we want all
	// those to go through the finishedPlanning method.
	// So can't do: nextStep = pickAlternative(); // like a poison
	nextStep = new ReplanStep();
	planLoop();
    }

    class ReplanStep implements Step {
	public void run() {
	    nextStep = pickAlternative("Replan"); // like a poison
	}
    }

    boolean findFirstPlan() {
	// This is an alternative to plan() that returns false
	// instead of throwing a NoPlanException.
	try { plan(); return true; }
	catch (NoPlanException e) { return false; }
    }

    boolean findNextPlan() {
	// This is an alternative to replan() that returns false
	// instead of throwing a NoPlanException.
	try { replan(); return true; }
	catch (NoPlanException e) { return false; }
    }

    void planLoop() {
	try {
	    while (true) {	// was (nextStep != null) /\/
		try {
		    Step step = nextStep;
		    nextStep = invalidStep;
		    Debug.noteln("Step " + (stats.numberStepsTaken + 1) +": "+
				 Context.getContext() + " " + step);
		    if (singleStep)
			stepper.step();
		    stats.stepTaken();
		    step.run();
		}
		catch (Poison p) {
		    Debug.noteln(Context.getContext(), p);
		    nextStep = pickAlternative(p);
		}
		if (stats.numberStepsTaken >= hardStepLimit)
		    stepLimitReached(true);
		if (stats.numberStepsTaken % stepLimit == 0)
		    stepLimitReached(false);
	    }
	}
	catch (HavePlanException e) {
	    createPlanContext();
	    finishedPlanning(true);
	}
	catch (NoPlanException e) {
	    finishedPlanning(false);
	    throw e;
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    finishedPlanning(false);
	    throw new RethrownException(t);
	}
    }

    // static class NoPlanException extends RuntimeException { }

    static class HavePlanException extends RuntimeException { }

    static class StepLimitException extends RuntimeException {
	public StepLimitException(String message) { super(message); }
    }

    void stepLimitReached(boolean isHardLimit) {
	if (!isHardLimit && Parameters.isInteractive()) {
	    if (Util.dialogConfirms(IPlan.displayFrame(), new String[] {
		  "The planner has not found a plan after " +
		      stats.numberStepsTaken + " planning steps.",
		  "Do you want to continue for another " +
                      stepLimit + " steps?"}))
		return;
	}
	throw new StepLimitException
	    ("Step limit exceeded after " + stats.numberStepsTaken +
	     " planning steps.");
    }

    void finishedPlanning(boolean success) {
	Debug.noteln("Finished planning, success = " + success);
	stats.fillInValues();
	if (success) {
	    Debug.expect(isWaiting.isEmpty(), "nonempty isWaiting map");
	    // /\/: Any other checks we ought to make?
	}
	if (Debug.on) {
	    Debug.noteln(success ? "Plan found" : "No plan");
	    Debug.noteln("Current context", Context.getContext());
	    Context.printContextTree();
	    stats.report(Debug.out);
	    describeAlternatives();
	    // if (success) describeNodeEnds(iplanMM.getNodeEnds());
	    // if (success) iplanMM.getTPNManager().describeTPN(Debug.out);
	    if (success) noteHistory();
	}
    }

    void createPlanContext() {
	// In the contexts we show externally some things are different
	// than in the contexts used while planning.
	// /\/: We should have O-Plan-style options.
	Context.pushContext();
	// Go back to the initial world state rather than the
	// one from the end of planning.
	iplanMM.setWorldStateMap(savedInitialWorldState);
	// The values in the nodeEndStatusMap are also now incorrect,
	// because they're from the end of planning, and so will
	// (usually?) be complete.
	computeProperStatusValues();
	// Record inputs and outputs so we can recover the data-flow.
	recordInputsAndOutputs();
	// /\/:
	for (PNode n: iplanMM.getNodes())
	    if (n.isExpanded())
		((ActivityItem)n).recordExpansionRefinementName();
    }

    /*
     * Alternatives
     */

    void postAlternative(Alternative alt) {
	alt.setCost(this);
	Debug.noteln("Posting", alt);
	stats.altPosted();
	alternatives.pushElement(alt);
	Context.pushContext();
    }

    Alternative pickAlternative(Object reason) {
	// Returns the lowest cost alt, preferring more recent alts
	// when costs are the same.  In some cases, a depth-first
	// choice is made instead.
	if (alternatives.isEmpty())
	    throw new NoPlanException();
	// Start with the most recent alternative.
	Iterator i = alternatives.iterator();
	Alternative best = (Alternative)i.next();
	boolean depthFirst = pickAlternativesDepthFirst
	                     || best.isLocalChoice();
	if (!i.hasNext()) {
	    // There is only one.
	    Debug.noteln("Only one alternative, depth-first = " + depthFirst);
	}
	else if (depthFirst) {
	    Debug.noteln("Picking alternative depth-first");
	    // Stick with most recent.
	}
	else {
	    // Look for lowest cost alternative.
	    double bestCost = best.getCost();
	    while (i.hasNext()) {
		Alternative alt = (Alternative)i.next();
		double altCost = alt.getCost();
		if (altCost < bestCost) {
		    best = alt;
		    bestCost = altCost;
		}
	    }
	}
	Debug.noteln("Picking", best);
	stats.altPicked();
	alternatives.deleteElement(best);
	Context altContext = best.getContext();
	Context.setContext(altContext);
	Debug.expect(contextIsPoisoned.isFalse(),
		     "context is poisoned", altContext);
	Context.pushContext();
	history("Picked", best, "because", reason);
	best.whenPicked();
	return best;
    }

    Poison poison(Object reason) {
	// Always call using: throw poison(...);
	contextIsPoisoned.set(true);
	return new Poison(reason);
    }

    /*
     * History
     */

    void history(Object... words) {
	history.add(words);
    }

    void noteHistory() {
	Debug.noteln("Route to this plan:");
	for (Object[] words : history) {
	    Debug.note("-->");
	    for (Object word : words) {
		Debug.note(" "); Debug.note(word.toString());
	    }
	    Debug.noteln("");
	}
	Debug.noteln("");
    }

    /*
     * Parameters / properties
     */

    static final Symbol S_TRUE = Symbol.intern("true");

    boolean domainSays(String propname) {
	return domain.getAnnotation(Symbol.intern(propname)) == S_TRUE;
    }

    static final Symbol SEARCH_PREFERENCE = Symbol.intern("search-preference");
    static final Symbol DEPTH_FIRST = Symbol.intern(":depth-first");

    boolean preferDepthFirst() {
	return domain.getAnnotation(SEARCH_PREFERENCE) == DEPTH_FIRST;
    }

    // Domains and refienments can specify achievable-world-state-conditions;
    // refinements can also specify use-for-world-state-effects.

    // /\/: Achievable-world-state-conditions doesn't yet work in refinements.

    public static final Keyword K_ALL = (Keyword)Symbol.intern(":all");
    public static final Keyword K_NONE = (Keyword)Symbol.intern(":none");
    public static final Symbol S_ACHIEVABLE_CONDS =
	Symbol.intern("achievable-world-state-conditions");
    public static final Symbol S_USE_FOR_EFFECTS =
	Symbol.intern("use-for-world-state-effects");

    static boolean isAchievableCond(LList pattern, Annotated decider) {
	Object achievable = decider.getAnnotation(S_ACHIEVABLE_CONDS);
	return achievable == null ? true
	    :  achievable == K_ALL ? true
	    :  achievable == K_NONE ? false
	    :  ((List)achievable).contains(pattern.get(0));
    }

    static boolean canBeUsedForEffect(LList pattern, Refinement r) {
	Object usable = r.getAnnotation(S_USE_FOR_EFFECTS);
	return usable == null ? true
	    :  usable == K_ALL ? true
	    :  usable == K_NONE ? false
	    :  ((List)usable).contains(pattern.get(0));
    }

    /*
     * Status propagation
     */

    void testStatusPropagation() {
	while (!planIsComplete()) {
	    PNodeEnd next = findExecutable();
	    Debug.expect(next != null, "nothing to execute");
	    Debug.noteln("Executing:", next);
	    setStatus(next, Status.COMPLETE);
	}
    }

    PNodeEnd findExecutable() {
	return findExecutable(iplanMM.getNodeEnds().iterator());
    }

    PNodeEnd findExecutable(Iterator i) {
	while (i.hasNext()) {
	    PNodeEnd end = (PNodeEnd)i.next();
	    // Debug.noteln("Considering", end);
	    // Debug.noteln("Status", getStatus(end));
	    if (getStatus(end) == Status.POSSIBLE)
		return end;
	}
	return null;
    }

    boolean planIsComplete() {
	return allHaveStatus(iplanMM.getNodeEnds(), Status.COMPLETE);
    }

    Status getStatus(PNodeEnd end) {
	Status s = (Status)nodeEndStatusMap.get(end);
	return s != null ? s: Status.BLANK;
    }

    void setStatus(PNodeEnd end, Status status) {
	if (status == Status.COMPLETE)
	    history(status, end);
	nodeEndStatusMap.put(end, status);
	allComputeStatus(end.getSuccessors());
    }

    void computeStatus(PNodeEnd end) {
	if (getStatus(end) == Status.BLANK) {
	    if (allHaveStatus(end.getPredecessors(), Status.COMPLETE))
		setStatus(end, Status.POSSIBLE);
	}
    }

    void allComputeStatus(List nodeEnds) {
	Iterator i = nodeEnds.iterator();
	while (i.hasNext()) {
	    PNodeEnd e = (PNodeEnd)i.next();
	    computeStatus(e);
	}
    }

    boolean allHaveStatus(List nodeEnds, Status status) {
	Iterator i = nodeEnds.iterator();
	while(i.hasNext()) {
	    PNodeEnd e = (PNodeEnd)i.next();
	    if (getStatus(e) != status)
		return false;
	}
	return true;
    }

    /*
     * Working with the model-manager's / node-ends' own status values.
     */

    void initStatusTable() {
	Iterator i = iplanMM.getNodeEnds().iterator();
	while (i.hasNext()) {
	    PNodeEnd e = (PNodeEnd)i.next();
	    nodeEndStatusMap.put(e, e.getStatus());
	}
    }

    void computeProperStatusValues() {
	// Node-ends
	Iterator i = iplanMM.getNodeEnds().iterator();
	while (i.hasNext()) {
	    PNodeEnd e = (PNodeEnd)i.next();
	    computeProperStatus(e);
	}
	// Nodes
	Iterator j = iplanMM.getNodes().iterator();
	while (j.hasNext()) {
	    PNode n = (PNode)j.next();
	    setProperStatus(n);
	}
    }

    void computeProperStatus(PNodeEnd end) {
	Status status = end.getStatus();
	// Debug.noteln("CPS " + end + " " + status);
        if (status == Status.BLANK) {
	    if (PNode.allHaveStatus(end.getPredecessors(), Status.COMPLETE))
		setProperStatus(end, Status.POSSIBLE);
	}
	else if (status == Status.POSSIBLE) {
	    if (!PNode.allHaveStatus(end.getPredecessors(), Status.COMPLETE))
		setProperStatus(end, Status.BLANK);
	}
	else if (status != Status.COMPLETE)
	    throw new IllegalArgumentException
		("Cannot handle status " + status + " for " + end);
    }

    void setProperStatus(PNodeEnd end, Status status) {
	Debug.noteln("Setting proper status of " + end +
		     " from " + end.getStatus() +
		     " to " + status);
	end.setStatus(status);
	Debug.noteln("Status is now", end.getStatus());
    }

    void setProperStatus(PNode node) {
	Status nodeStatus = node.getStatus();
	Status correctNodeStatus = node.statusFromNodeEnds();
	if (nodeStatus != correctNodeStatus) {
	    Debug.noteln("Setting proper status of " + node +
			 " from " + nodeStatus +
			 " to " + correctNodeStatus);
	    node.setStatus(correctNodeStatus);
	    Debug.noteln("Node status is now", node.getStatus());
	}
    }

    /*
     * "Executing" node-ends
     */

    boolean canExecBegin(PNodeEnd ne) {
	Debug.expectSame(End.BEGIN, ne.getEnd());
	// Assumes that a way to satisfy the filter conds has been
	// chosen and that the corresponding bindings have been made.
	// /\/: Time constraints have also already been done,
	// and it was before the satisfaction bindings,
	// which limits the use of vars in time constraints.
	PNode node = ne.getNode();
	List temporal = iplanMM.getNodeTimeConstraints(node);
	List other = iplanMM.getOtherNodeConstraints(node);
	return Variable.isFullyBoundEverywhere(temporal)
	    && Variable.isFullyBoundEverywhere(other);
    }

    void execBegin(PNodeEnd ne) {
	Debug.noteln("execBegin", ne);
	Debug.expectSame(End.BEGIN, ne.getEnd());
	Debug.expect(canExecBegin(ne)); // /\/
	PNode node = ne.getNode();
	List<Constraint> other = iplanMM.getOtherNodeConstraints(node);
	Debug.noteln("Other constraints", other);
	if (other != null) {
	    try {
		iplanMM.evalAtBegin(ne, other);
	    }
	    catch (FailureException fail) {
		throw poison(fail);
	    }
	}
	releaseWaitingEnds(ne);
	setStatus(ne, Status.COMPLETE);
    }

    boolean canExecEnd(PNodeEnd ne) {
	Debug.expectSame(End.END, ne.getEnd());
	// Looks at the effects and assumes other constraints
	// were checked at the BEGIN end.
	PNode node = ne.getNode();
	List effects = iplanMM.getNodeEffects(node);
	return Variable.isFullyBoundEverywhere(effects);
    }

    void execEnd(PNodeEnd ne) {
	Debug.noteln("execEnd", ne);
	Debug.expectSame(End.END, ne.getEnd());
	PNode node = ne.getNode();
	List effects = iplanMM.getNodeEffects(node);
// 	List temporal = iplanMM.getNodeTimeConstraints(node);
	List<Constraint> other = iplanMM.getOtherNodeConstraints(node);
	Debug.noteln("Effects", effects);
	Debug.noteln("Other constraints", other);
	try {
	    if (effects != null)
		iplanMM.handleEffects(node, effects);
	    if (other != null)
		iplanMM.evalAtEnd(ne, other);
	}
	catch (FailureException fail) {
	    throw poison(fail);
	}
	releaseWaitingEnds(ne);
	setStatus(ne, Status.COMPLETE);
    }

    void releaseWaitingEnds(PNodeEnd ne) {
	List<PNodeEnd> waiting = (List<PNodeEnd>)unwaitTable.get(ne);
	if (waiting != null)
	    for (PNodeEnd w: waiting)
		isWaiting.remove(w);
    }

    /*
     * Inputs and outputs for OWL-S processes and similar
     */

    void recordInputsAndOutputs() {
	for (Iterator i = iplanMM.getNodes().iterator(); i.hasNext();) {
	    AgendaItem node = (AgendaItem)i.next();
	    Activity act = (Activity)node.getAbout();
	    if (act.getAnnotation(S_INPUTS) != null
		  || act.getAnnotation(S_OUTPUTS) != null)
		// Assume they were done in earlier planning session.
		continue;
	    recordInputsAndOutputs(act);
	}
    }

    void recordInputsAndOutputs(Activity act) {
	// /\/: For now we can assume that an activity with
	// inputs or outputs has a pattern of the form
	// (service_name input_var... TO output_Var)
	// where the service_name is also the name of a
	// refinement.  By matching that to the activity's
	// by now instantiated pattern, we can recover the
	// values of the input and output vars.  The activity's
	// or the refinement's conds and effects can then
	// give us the types.
	// /\/: Unfortunately, whenever the refinement name
	// and the first word of the activity pattern match,
	// we'll try to find inputs and outputs even if
	// there aren't meant to be any.  We could check the
	// refinement to see if any outputs were intended,
	// but there's no corresponding annotation for inputs.
	String possibleName = act.getPattern().car().toString();
	Refinement ref = domain.getNamedRefinement(possibleName);
	if (ref == null)
	    // No refinement with the expected name, so assume
	    // no inputs or outputs.  /\/
	    return;
	Debug.expect(Variable.isFullyBound(act.getPattern()),
		     "Not fully bound", act.getPattern());
	MatchEnv env = Matcher.match(ref.getPattern(), act.getPattern());
	if (env == null)
	    // The refinement's pattern doesn't match the activity's,
	    // so it was presumably not used to expand the activity.
	    return;
	List conds = ref.getConditions();
	List effects = ref.getEffects();
	if (!conds.isEmpty()) {
	    List inputs = makeParameters(conds, env, makeInput);
	    if (!inputs.isEmpty())
		act.setAnnotation(S_INPUTS, inputs);
	}
	if (!effects.isEmpty()) {
	    List outputs = makeParameters(effects, env, makeOutput);
	    if (!outputs.isEmpty())
		act.setAnnotation(S_OUTPUTS, outputs);
	}
    }

    List makeParameters(List patternAssignments,
			MatchEnv env,
			ParameterMaker m) {
	List result = new LinkedList();
	for (Iterator i = patternAssignments.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    LList pat = pv.getPattern();
	    if (pat.length() == 2 && pat.get(0) == S_TYPE
		  && pat.get(1) instanceof ItemVar
                  && pv.getValue() instanceof Symbol) {
		ItemVar var = (ItemVar)pat.get(1);
		Symbol name = var;
		Symbol type = (Symbol)pv.getValue();
		Symbol value = (Symbol)env.get(var);
		// /\/: Sometimes type constraints aren't for inputs
		// or outputs.  See recordInputsAndOutputs(Activity act)
		// above.  So the var may not be in the env, and
		// then the value will be null.
//  		if (value != null)
		    result.add(m.makeParameter(name, type, value));
	    }
	}
	return result;
    }

    interface ParameterMaker {
	public ProcessParameter makeParameter
	         (Symbol name, Symbol type, Symbol value);
    }

    ParameterMaker makeInput = new ParameterMaker() {
	public ProcessParameter makeParameter
	         (Symbol name, Symbol type, Symbol value) {
	    return new Input(name, type, value);
	}
    };

    ParameterMaker makeOutput = new ParameterMaker() {
	public ProcessParameter makeParameter
                 (Symbol name, Symbol type, Symbol value) {
	    return new Output(name, type, value);
	}
    };

    /*
     * Achieving and goal node utilities
     */

    // /\/: Should make it more regular, or at least clearer,
    // when the argument is the node and when it's the node-end,
    // and also when it's referred to as an ActivityItem or
    // AgendaItem and when it's referred to as a PNode.

    static final Symbol S_ACHIEVE_GOAL = Symbol.intern("__achieve-goal__");

    static final Symbol S_ACHIEVE_PV = Symbol.intern("__achieve-pv__");

    static final Symbol S_AT_END = Symbol.intern("__at-end___");

    static final Symbol S_EQL = Symbol.intern("=");

    boolean isGoalNode(PNodeEnd ne) {
	LList pattern = ne.getNode().getPattern();
	return pattern.car() == S_ACHIEVE_GOAL;
    }

    boolean isGoalNode(PNode n) {
	LList pattern = n.getPattern();
	return pattern.car() == S_ACHIEVE_GOAL;
    }

    PatternAssignment getGoalCond(PNodeEnd ne) {
	ActivityItem item = (ActivityItem)ne.getNode();
	Activity act = (Activity)item.getAbout();
	PatternAssignment pv =
	    (PatternAssignment)act.getAnnotation(S_ACHIEVE_PV);
	Debug.expect(pv != null, "no goal cond for", act);
	return pv;
    }

    ActivityItem makeGoalNode(PatternAssignment goalCond,
			      List achievers,
			      PNodeEnd atNodeEnd) {
	LList pattern = Lisp.list
	    (S_ACHIEVE_GOAL, goalCond.getPattern(), S_EQL,
	                     goalCond.getValue());
	Activity act = new Activity(pattern);
	ActivityItem item = new ActivityItem(act);
	act.setAnnotation(S_ACHIEVE_PV, goalCond);
	// /\/: We have to record the achievers someplace,
	// and for now that's here:
	act.setAnnotation(S_ACHIEVE_GOAL, achievers);
	// /\/: Note that achievers can't be converted to XML
	// or processed by an ObjectCopier.  That's one reason
	// why we lose the annotation in normalizeGoalNode.
	// /\/: The "at" annotation is in a similar boat.
	act.setAnnotation(S_AT_END, atNodeEnd);
	return item;
    }

    List getGoalAchievers(PNode goalNode) {
	Debug.expectSame(S_ACHIEVE_GOAL, goalNode.getPattern().get(0));
	ActivityItem item = (ActivityItem)goalNode;
	List achievers = (List)item.getAbout().getAnnotation(S_ACHIEVE_GOAL);
	Debug.expect(achievers != null, "no achievers for", goalNode);
	return achievers;
    }

    PNodeEnd getAtEndForGoal(PNodeEnd goalEnd) {
	return getAtEndForGoal((ActivityItem)goalEnd.getNode());
    }

    PNodeEnd getAtEndForGoal(ActivityItem goalNode) {
	Debug.expectSame(S_ACHIEVE_GOAL, goalNode.getPattern().get(0));
	PNodeEnd end = (PNodeEnd)goalNode.getAbout().getAnnotation(S_AT_END);
	Debug.expect(end != null, "no at-end for", goalNode);
	return end;
    }

    void normalizeGoalNode(ActivityItem goalNode) {
	Refinement r = goalNode.getRefinement();
	PNodeEnd atEnd = getAtEndForGoal(goalNode);
	// Make the goal node look like a normal node.
	// We both change the pattern to the one from the refinement
	// and get rid of the annotations listing the achievers,
	// and the end the cond must be true "at",
	// by replacing the Activity the ActivityItem is "about". /\/
	LList pattern = r.getPattern();
	Activity act = new Activity(pattern);
	// We also change the meaning of the annotation to record
	// the information that's in the pattern.
	act.setAnnotation(S_ACHIEVE_GOAL, getGoalCond(goalNode.getBegin()));
	// Now we can change what the goal node is "about".
	goalNode.setAbout(act);
	// Re-record which refinement we used, because it's
	// also stored in the "about" activity, and we've just
	// put a new activity object in as the "about".  /\/
	goalNode.setRefinement(r);
	// /\/: Here's where we finally put a TPN constraint between
	// the ends of the node.  We can't easily do it earlier because
	// if we deleted the goal node, it would be a pain to remove
	// the constraint.
	TimePointNet tpnm = iplanMM.getTPNManager();
	tpnm.addTimeConstraintElseFail(goalNode.getBegin(), goalNode.getEnd());
	// /\/: Also add a TPN constraint to the node-end that wanted
	// the condition satisfied.
	tpnm.addTimeConstraintElseFail(goalNode.getEnd(), atEnd);
    }

    // /\/: Used only as a set.
    ContextHashMap waitingForAchieveTable = new ContextHashMap();

    void recordAsWaitingForAchieve(PNode node) {
	waitingForAchieveTable.put(node, node);
    }

    boolean isWaitingForAchieve(PNode node) {
	return waitingForAchieveTable.get(node) == node;
    }

    /*
     * Domain utilities
     */

    /**
     * Returns a list of ways to achieve a condition by introducing a node.
     */
    List getAchieversForCond(PatternAssignment pv) {
	VariableManager vm = iplanMM.getVariableManager();
	AdviceManager advice = iplanMM.getAdviceManager();
	List result = new LinkedList();
	LList condPat = pv.getPattern();
	Object condVal = pv.getValue();
	// First check whether the domain says the cond allows achieving.
	if (!isAchievableCond(condPat, domain))
	    return result;	// will be empty
	// Now look at all the refinements.
	for (Iterator ri = getDomain().getRefinements().iterator()
		 ; ri.hasNext();) {
	    Refinement r = (Refinement)ri.next();
	    ListOfConstraint cs = r.getConstraints();
	    if (cs == null)
		continue;	// next Refinement
	    if (!advice.isUsableRefinement(r))
		continue;
	    for (Iterator ci = cs.iterator(); ci.hasNext();) {
		Constraint c = (Constraint)ci.next();
		if (c.getType() == r.S_WORLD_STATE
		      && c.getRelation() == r.S_EFFECT) {
		    Constraint effect = c;
		    PatternAssignment effect_pv =
			(PatternAssignment)effect.getParameter(0);
		    LList ePat = effect_pv.getPattern();
		    // Compare first word of cond and effect patterns,
		    // in case we can skip proper matching.
		    if (condPat.car() != ePat.car())
			continue; // next effect
		    // See of the effect can be used to achieve
		    if (!canBeUsedForEffect(ePat, r))
			continue; // next effect
		    Object eVal = effect_pv.getValue();
		    MatchEnv pEnv = Matcher.match(condPat, ePat);
		    if (pEnv == null)
			continue; // next effect
		    MatchEnv vEnv = Matcher.match(condVal, eVal, pEnv);
		    if (vEnv == null)
			continue; // next effect
		    // We've found a matching effect.
		    // But check with existing constraints. /\/
		    if (!vm.consistentBindings(vEnv))
			continue;
		    result.add(new SlipAchiever(pv, r, effect_pv, vEnv));
		}
	    }
	}
	return result;
    }

    boolean existsAchieversForCond(PatternAssignment pv) {
	VariableManager vm = iplanMM.getVariableManager();
	AdviceManager advice = iplanMM.getAdviceManager();
	LList condPat = pv.getPattern();
	Object condVal = pv.getValue();
	if (!isAchievableCond(condPat, domain))
	    return false;
	for (Iterator ri = getDomain().getRefinements().iterator()
		 ; ri.hasNext();) {
	    Refinement r = (Refinement)ri.next();
	    ListOfConstraint cs = r.getConstraints();
	    if (cs == null)
		continue;	// next Refinement
	    if (!advice.isUsableRefinement(r))
		continue;
	    for (Iterator ci = cs.iterator(); ci.hasNext();) {
		Constraint c = (Constraint)ci.next();
		if (c.getType() == r.S_WORLD_STATE
		      && c.getRelation() == r.S_EFFECT) {
		    Constraint effect = c;
		    PatternAssignment eff =
			(PatternAssignment)effect.getParameter(0);
		    LList ePat = eff.getPattern();
		    // Compare first word of cond and effect patterns,
		    // in case we can skip proper matching.
		    if (condPat.car() != ePat.car())
			continue; // next effect
		    // See of the effect can be used to achieve
		    if (!canBeUsedForEffect(ePat, r))
			continue; // next effect
		    Object eVal = eff.getValue();
		    MatchEnv pEnv = Matcher.match(condPat, ePat);
		    if (pEnv == null)
			continue; // next effect
		    MatchEnv vEnv = Matcher.match(condVal, eVal, pEnv);
		    if (vEnv == null)
			continue; // next effect
		    // We've found a matching effect
		    // But check with existing constraints. /\/
		    if (!vm.consistentBindings(vEnv))
			continue;
		    return true;
		}
	    }
	}
	return false;
    }

    /*
     * Potential conditions and effects
     */

    Set getPossibleConditions(PNodeEnd ne) {
	// We need whatever conds are already on the node-end in the MM
	// (which - if we have O-Plan-style refinements - could have come
	// from refining the node above), plus, if the node hasn't been
	// expanded, the possible conds found by domain-analysis.
	// Note that it's ok to have too many conds, but not to miss any.
	// For goal nodes, we need to include the cond the node's supposed
	// to satisfy (since the node will be eliminated if it's already
	// satisfied) plus the possible conds of the applicable refinements
	// for the goal node.
	return possibleCondFinder.getPossibles(ne);
    }

    Set getPossibleEffects(PNodeEnd ne) {
	// Similar to conds except that it's about effects.
	return possibleEffectFinder.getPossibles(ne);
    }

    Set getPossibleOtherConstraints(PNodeEnd ne) {
	return possibleConstraintFinder.getPossibles(ne);
    }

    PossibleConditionFinder possibleCondFinder =
	new PossibleConditionFinder();
    PossibleEffectFinder possibleEffectFinder =
	new PossibleEffectFinder();
    PossibleConstraintFinder possibleConstraintFinder =
	new PossibleConstraintFinder();

    class PossibleConditionFinder extends PossiblesFinder {
	List getInstalled(PNode node) {
	    return constraintsToPVs(iplanMM.getNodeConditions(node));
	}
	Set getForPattern(LList pattern) {
	    return analysis.getPossibleConditions(pattern);
	}
	void addExtraForAchiever(SlipAchiever a, Set result) {
	    result.add(a.cond);
	}
    }

    class PossibleEffectFinder extends PossiblesFinder {
	List getInstalled(PNode node) {
	    return iplanMM.getNodeEffects(node);
	}
	Set getForPattern(LList pattern) {
	    return analysis.getPossibleEffects(pattern);
	}
	void addExtraForAchiever(SlipAchiever a, Set result) {
	    // The a.cond we're trying to achieve should be among
	    // the possible effects.
	    Debug.expect(analysis.haveCommonPatterns
			 (result, new HashSet(Lisp.list(a.cond))));
	}
    }

    class PossibleConstraintFinder extends PossiblesFinder {
	List getInstalled(PNode node) {
	    return iplanMM.getOtherNodeConstraints(node);
	}
	Set getForPattern(LList pattern) {
	    return analysis.getPossibleConstraints(pattern);
	}
    }

    abstract class PossiblesFinder {

	Set getPossibles(PNodeEnd ne) {
	    // Normally gets Constraints; sometimes PatternAssignments /\/
	    if (isGoalNode(ne))
		return getForGoalNode(ne);
	    PNode node = ne.getNode();
	    List installed = getInstalled(node);
	    if (node.isExpanded())
		return installed == null ? Collections.emptySet()
		    :  new HashSet(installed);
	    Set constraints = getForPattern(node.getPattern());
	    if (installed != null)
		constraints.addAll(installed);
	    return constraints;
	}

	abstract List getInstalled(PNode node);
	abstract Set getForPattern(LList pattern);
	void addExtraForAchiever(SlipAchiever a, Set result) {
	}

	Set getForGoalNode(PNodeEnd ne) {
	    Debug.expect(!ne.getNode().isExpanded());
	    Set result = new HashSet();
	    List achievers = getGoalAchievers(ne.getNode());
	    for (Iterator i = achievers.iterator(); i.hasNext();) {
		SlipAchiever a = (SlipAchiever)i.next();
		LList p = a.refinement.getPattern();
		result.addAll(getForPattern(p));
		addExtraForAchiever(a, result);
	    }
	    return result;
	}

    }

    private List<PatternAssignment> constraintsToPVs(List constraints) {
	if (constraints == null) return Collections.emptyList();
	List<PatternAssignment> result = new LinkedList<PatternAssignment>();
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    result.add(((Constraint)i.next()).getPatternAssignment());
	}
	return result;
    }

    /*
     * Debugging utilities
     */

    void describeAlternatives() {
	Debug.noteln("Alternatives:");
	if (alternatives.isEmpty())
	    Debug.noteln("None");
	else {
	    for (Iterator i = alternatives.iterator(); i.hasNext();) {
		Alternative alt = (Alternative)i.next();
		describeAlternative(alt);
	    }
	}
    }

    void describeAlternative(Alternative alt) {
	Debug.noteln("   " + alt);
    }

    void describeNodeEnds() {
	describeNodeEnds(iplanMM.getNodeEnds());
    }

    void describeNodeEnds(List nodeEnds) {
	for (Iterator i = nodeEnds.iterator(); i.hasNext();) {
	    PNodeEnd end = (PNodeEnd)i.next();
	    describeNodeEnd(end);
	}
    }

    void describeNodeEnd(PNodeEnd end) {
	Debug.noteln(getStatus(end) + " " + end);
	describeLinks(end, "after", end.getPredecessors());
	describeLinks(end, "before", end.getSuccessors());
    }

    void describeLinks(PNodeEnd at, String relation, List nodeEnds) {
	if (nodeEnds.isEmpty())
	    Debug.noteln("  No " + relation + " links");
	else {
	    Debug.noteln("  Linked " + relation + ":");
	    for (Iterator i = nodeEnds.iterator(); i.hasNext();) {
		PNodeEnd otherEnd = (PNodeEnd)i.next();
		if (otherEnd.getNode() == at.getNode())
		    Debug.noteln("     " + otherEnd.getEnd() + " of self");
		else
		    Debug.noteln("    ", otherEnd);
	    }
	}
    }

}
