/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 19 02:27:22 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.Plan;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * A plan-checker that looks for inconsistencies in the details.
 *
 * <p><i>This class was inspired by the sanity-checker in O-Plan.</i></p>
 *
 * @see AutoTester
 * @see PlanCheckingSimulator
 *
 */
public class SanityChecker {

    protected static final Symbol TRUE = Symbol.intern("true");
    protected static final Symbol FALSE = Symbol.intern("false");

    protected PlanCheckingSimulator sim;
    protected Ip2ModelManager mm;

    /**
     * Creates a sanity-checker for the specified similator.
     */
    public SanityChecker(PlanCheckingSimulator sim) {
	this.sim = sim;
	this.mm = sim.getModelManagerBeingChecked();
    }

    public void checkPlan() {
	sim.traceln("Running sanity check.");
	checkNodesVsNodeEnds();
	checkExpansionRefinements();
	checkForMinimalNodes();
	checkForMaximalNodes();
	checkStartAndFinishTimes();
	checkOrderingLinkConnectivity();
	checkTemporalConstraintConnectivity();
	checkTemporalConstraintsAgainstLinks();
    }

    public void checkNodesVsNodeEnds() {
	Set ends = new HashSet(mm.getNodeEnds());
	for (PNode n: mm.getNodes()) {
	    if (!ends.remove(n.getBegin()))
		sim.problem("Missing " + n.getBegin());
	    if (!ends.remove(n.getEnd()))
		sim.problem("Missing " + n.getEnd());
	}
	if (!ends.isEmpty())
	    sim.problem("Stray ends " + ends);
    }

    /**
     * Checks whether the correct expansion refinement was recorded for each
     * expanded activity in the plan.
     */
    public void checkExpansionRefinements() {
	Domain dom = mm.getIp2().getDomain();
	for (ActivityItem act: mm.getNodes()) {
	    if (!act.isExpanded())
		continue;
	    if (dom.isEmpty()
		  && dom.getName() != null
		  && dom.getName().equals("empty-domain")) {
		// Special case for when we're meant to check a plan
		// without having a meaningful domain to use.
		// The activity might not even have a recorded
		// expansion-refinement name.
		continue;
	    }
	    String name = act.getExpansionRefinementName();
	    if (name == null) {
		sim.problem("No expansion refinements for " + act);
		continue;
	    }
	    Refinement r = dom.getNamedRefinement(name);
	    if (r == null) {
		sim.problem("No refinement named " + Strings.quote(name) +
			    " for expanding " + act);
		continue;
	    }
	    // We now have the refinement that was supposedly
	    // used to expand the activity.
	    MatchEnv env = Matcher.match(r.getPattern(), act.getPattern());
	    // The refinement's pattern should match the activity's.
	    if (env == null)
		sim.problem(r + " doesn't match " + act);
	    // Every subnode specified by the refinement should
	    // exist as a subactivity and (for now /\/) vice versa.
	    // For now, we also expect them to be in the same order. /\/
	    ListOfNodeSpec specs = r.getNodes();
	    List<PNode> children = act.getChildren();
	    if (specs == null || specs.isEmpty()) {
		if (!(children == null || children.isEmpty()))
		    sim.problem("Children of " + act + 
				" that weren't in " + r + ": " + children);
		continue;
	    }
	    if (children == null || children.isEmpty()) {
		sim.problem("Subnodes of " + r +
			    " that weren't subactivities of " + act +
			    ": " + specs);
		continue;
	    }
	    if (specs.size() != children.size()) {
		sim.problem(act + " has the wrong number of subnodes for " +
			    r);
		continue;
	    }
	    Iterator si = specs.iterator();
	    for (PNode child: children) {
		NodeSpec spec = (NodeSpec)si.next();
		if (Matcher.match(spec.getPattern(), child.getPattern())==null)
		    sim.problem(act + " child " + child + " does not match " +
				spec + " from " + r);
	    }
	}
    }

    /*
     * Check for minimal and maximal nodes
     */

    public void checkForMinimalNodes() {
	for (PNode node: mm.getNodes())
	    if (node.getBegin().getPredecessors().isEmpty())
		return;
	sim.problem("No minimal nodes.");
    }

    public void checkForMaximalNodes() {
	for (PNode node: mm.getNodes())
	    if (node.getEnd().getSuccessors().isEmpty())
		return;
	sim.problem("No maximal nodes.");
    }

    /*
     * Check start and finish times.
     */

    public void checkStartAndFinishTimes() {
	for (PNode node: mm.getNodes()) {
	    mustNotBeLater(node.getBegin(), node.getEnd());
	    PNode parent = node.getParentPNode();
	    if (parent == null)
		continue;
	    mustNotBeLater(parent.getBegin(), node.getBegin());
	    mustNotBeLater(node.getEnd(), parent.getEnd());
	}
    }

    private void mustNotBeLater(PNodeEnd a, PNodeEnd b) {
	mustBeLE(a, b, "start", a.getMinTime(), b.getMinTime());
	mustBeLE(a, b, "finish", b.getMaxTime(), a.getMaxTime());
    }

    private void mustBeLE(PNodeEnd a, PNodeEnd b, String action,
			  long t1, long t2) {
	if (!(t1 <= t2))
	    sim.problem(a + " at " + Inf.asString(t1) +
			" can " + action + " after " +
			b + " at " + Inf.asString(t2));
    }

    /*
     * Check ordering link connectivity.
     */

    public void checkOrderingLinkConnectivity() {
	checkOrderingLinkConnectivityForward();
	checkOrderingLinkConnectivityBackward();
    }

    public void checkOrderingLinkConnectivityForward() {
	new ForwardOrderingLinkChecker().checkConnectivity();
    }

    private class ForwardOrderingLinkChecker
	    extends NodeEndConnectivityChecker<Object> {
	PNodeEnd getStartingEndForWalk(PNode n) {
	    return n.getBegin();
	}
	Collection<Object> getSuccessorLinks(PNodeEnd at) {
	    return at.getSuccessors();
	}
	PNodeEnd getSuccessor(Object link) {
	    return (PNodeEnd)link;
	}
    }

    public void checkOrderingLinkConnectivityBackward() {
	new BackwardOrderingLinkChecker().checkConnectivity();
    }

    private class BackwardOrderingLinkChecker 
	    extends ForwardOrderingLinkChecker {
	@Override
	PNodeEnd getStartingEndForWalk(PNode n) {
	    return n.getEnd();
	}
	Collection<Object> getSuccessorLinks(PNodeEnd at) {
	    return at.getPredecessors();
	}
    }

    /*
     * Check temporal constraint connectivity.
     */

    public void checkTemporalConstraintConnectivity() {
	checkTemporalConstraintConnectivityForward();
	// checkTemporalConstraintConnectivityBackward();
    }

    public void checkTemporalConstraintConnectivityForward() {
	new ForwardTemporalLinkChecker().checkConnectivity();
    }

    private class ForwardTemporalLinkChecker
	    extends NodeEndConnectivityChecker<TimePointNet.TimeConstraint> {
	PNodeEnd getStartingEndForWalk(PNode n) {
	    return n.getBegin();
	}
	Collection<TimePointNet.TimeConstraint> getSuccessorLinks(PNodeEnd at) {
	    return (Collection<TimePointNet.TimeConstraint>)
		      at.getPostConstraints();
	}
	PNodeEnd getSuccessor(TimePointNet.TimeConstraint link) {
	    return Util.mustBe(PNodeEnd.class, link.getPostPoint());
	}
    }

    /*
     * Check temporal constraints against ordering links.
     */

    public void checkTemporalConstraintsAgainstLinks() {
	for (PNodeEnd from: mm.getNodeEnds()) {
	    successor_loop:
	    for (Object to_i: from.getSuccessors()) {
		PNodeEnd to = (PNodeEnd)to_i;
		for (Object c_i: from.getPostConstraints()) {
		    TimePointNet.TimeConstraint c =
			(TimePointNet.TimeConstraint)c_i;
		    if (c.getPrePoint() == from && c.getPostPoint() == to)
			continue successor_loop;
		}
		sim.problem("No temporal constraint from "+from+" to "+to);
	    }
	}
    }

    /*
     * Graph utilities
     */

    protected abstract class NodeEndConnectivityChecker<L>
	               extends ReachableWithin<PNodeEnd,L> {

	boolean isWithin(Object top, PNodeEnd e) {
	    return e.getNode() == top || e.getNode().getParentPNode() == top;
	}

	abstract PNodeEnd getStartingEndForWalk(PNode n);

	void checkConnectivity() {
	    for (PNode n: mm.getNodes()) {
		PNodeEnd start = getStartingEndForWalk(n);
		PNodeEnd otherEnd = start.getOtherNodeEnd();
		Set<PNodeEnd> r = collect(n, start);
		if (!r.contains(otherEnd))
		    doesntReach(start, otherEnd);
		for (PNode c: n.getChildren()) {
		    if (!r.contains(c.getBegin()))
			doesntReach(start, c.getBegin());
		    if (!r.contains(c.getEnd()))
			doesntReach(start, c.getEnd());
		}
	    }
	}

	void doesntReach(PNodeEnd from, PNodeEnd to) {
	    String className = Strings.afterLast("$", getClass().getName());
	    sim.problem(className + ": " + from + " doesn't connect to " + to);
	}

    }

    protected static abstract class ReachableWithin<T,L> {

	abstract Collection<L> getSuccessorLinks(T at);
	abstract T getSuccessor(L link);
	abstract boolean isWithin(Object top, T at);

	public Set<T> collect(Object top, T start) {
	    Set<T> result = new HashSet<T>();
	    walk(start, top, result);
	    return result;
	}

	private void walk(T at, Object top, Set<T> result) {
	    if (!isWithin(top, at) || result.contains(at))
		return;
	    result.add(at);
	    for (L link: getSuccessorLinks(at)) {
		walk(getSuccessor(link), top, result);
	    }
	}

    }

}
