/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 22 16:59:17 2007 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import ix.icore.domain.Domain;
import ix.icore.plan.Plan;

/**
 * Planner / planning algorithm.  This interface provides a standard
 * way to use a planner from within Java.  One way to create a planner
 * is to call the static {@link IPlan#makeDefaultPlanner()} method.
 * A convenient way to use a planner (created by thet method) from
 * outside Java is to use the class {@link ix.test.FilePlanner}
 * as a main program.
 *
 * Here is how a planner might be used in Java:
 * <pre>
 *   import ix.icore.plan.Plan;
 *   import ix.icore.domain.Domain;
 *   import ix.iplan.IPlan;
 *   import ix.iplan.Planner;
 *   import ix.iplan.NoPlanException;
 *
 *   ...
 *
 *   Planner planner = IPlan.makeDefaultPlanner();
 *   planner.setDomain(...);
 *   planner.loadPlan(...);
 *   try {
 *       planner.plan();
 *       Plan plan = planner.getPlan();
 *       while (!isGoodEnoughPlan(plan)) {
 *           planner.replan();
 *           plan = planner.getPlan();
 *       }
 *   }
 *   catch (NoPlanException e) {
 *       ...
 *   }
 * </pre>
 */
public interface Planner {

    /**
     * Sets this planner's domain.  This is normally done before
     * plan() or replan() are called.  The consequences of calling
     * this method at other times are unspecified.
     */
    public void setDomain(Domain domain);

    /**
     * Adds the contents of the specified plan to this planner's
     * current plan.  This is normally done before plan() or
     * replan() are called.  If it is done later, it should
     * be followed by a call to plan() rather than to replan().
     * The consequences of calling this method at other times
     * are unspecified.
     */
    public void loadPlan(Plan plan);

    /**
     * Finds a plan that is a completion of this planner's current
     * plan and makes it the new current plan.  The precise meaning
     * of "completion" depends on the planner in question, but
     * typically it will mean that all activities in the initial
     * plan that might be expanded into subactions, using refinements
     * in this planner's domain, will be expanded, and that all
     * constraints in the initial plan or introduced by expansion
     * would be satisfied if the plan were executed in a way that
     * respected all of the ordering constraints.   (Here, the
     * <i>initial plan</i> is the plan that was current when
     * plan() was called.)
     *
     * <p>A successful call to plan() leaves this planner in a
     * state in which {@link #replan()} can be called.</p>
     *
     * @throws NoPlanException if no plan can be found by this planner.
     */
    public void plan();

    /**
     * Finds the next in a series of plans that are completions
     * of the plan that was current when {@link #plan()} was
     * most recently called.  That completion then becomes the
     * new current plan.  You can think of this as reconsidering
     * some choice made when producing earlier plans in the sequence
     * and making different decision at that point.  However, it
     * is not guaranteed that the plan found in this way will
     * be interestingly different from all of the earlier plans
     * in the sequence.
     *
     * <p>The replan() method should not be called if plan() has
     * not been called or if this planner has thrown an exception.</p>
     *
     * <p>A successful call to replan() leaves this planner in a
     * state in which {@link #replan()} can be called again.
     * plan() may be called instead, but this will begin a new
     * sequence of plans based on this planner's now current plan,
     * rather than on the plan that was the base for the previous
     * sequence.</p>
     *
     * @throws NoPlanException if no new plan can be found by this planner.
     */
    public void replan();

    /**
     * Returns this planner's current plan.  This value is
     * well-defined only if this planner has not thrown
     * an exception.
     */
    public Plan getPlan();

    /**
     * Returns statistics gathered by the most recent {@link #plan()}
     * or {@link #replan()}.
     */
    public PlanStats getStatistics();

}
