/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 10 21:15:20 2007 by Jeff Dalton
 * Copyright: (c) 1992, 1993, 2004 - 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import ix.util.*;

/**
 * Specifies a planning test.
 *
 * <p>A test specifies a domain, a task, a number of plans, and whether
 * or not the test is exhaustive.  It also gives the number of randomly
 * reordered simulations to try per plan, and a "step limit" that the
 * planner can use as a limit on how much work it should do.
 *
 * <p>The task can be specified either by an activity pattern
 * in the {@link #task} field or by a resource name (filename
 * or URL) in the {@link #initialPlan} field.  The activity
 * pattern is given as a string without initial and final parens,
 * for example <tt>"test1"</tt> or <tt>"paint block-1 green"</tt>.
 * In some cases, it may make sense to specify both an initial plan
 * and a task.
 *
 * <p>If the test is exhaustive, the auto-tester will expect to
 * find exactly the specified number of plans, and it will consider
 * the test a failure if it doesn't.  If you just want to check
 * the first solution and completely ignore the possibility of
 * replanning, use 1 as the number of plans and don't make the
 * test exhaustive.  (That is also the default.)  If you want
 * to look at the first k solutions when there are more than k,
 * use k as the number of plans and again make it not exhaustive.
 *
 * <p>For an exhaustive test, the number of solutions (plans) is also
 * the number of replans.  An exhaustive test should continue until
 * we run out of alternative plans.  So if there are k solutions, we
 * should replan k times (there's the first plan, then k-1 replans to
 * get the k-1 other solutions, then one more replan to get the
 * out-of-alternatives exception).  This is a nice coincidence,
 * but may not be exactly what's expected.
 *
 * <p>Each plan is tested, when found, by running an execution
 * simulator.  The plan is first executed in whatever order
 * the simulator naturally selects.  That order will always
 * be the same for the same plan.  If this PlanTest's
 * {@link #randomSimulations} field has a value greater than
 * zero, the similator will be used to simulate the plan's
 * execution that many more times in a way that randomly
 * permutes the execution order (subject to any ordering
 * constraints).  The simulator does this is a way that always
 * generates the same sequence of permutations for the same
 * plan, so that the simulations are repeatable.
 * By default no random simulations will be done.
 *
 * <p>When the planner finds it unusually difficult to find a plan
 * (or to determine that it can't find one), this PlanTest's
 * {@link #stepLimit} field can be set to a value large enough
 * to let the planner do enough work.  The required value is
 * necessarily implementational and must be determined empirically.
 *
 * <p><i>The PlanTest class is very like O-Plan's plan-test class, and
 * parts of this doc comment are based on the one for O-Plan.</i></p>
 *
 * @see AutoTester
 * @see PlanCheckingSimulator
 * @see PlanCheckingSimulator#setShuffle(boolean)
 */
public class PlanTest implements java.io.Serializable {

    protected String domain;
    protected String task;
    protected String initialPlan;
    protected int plans = 1;
    protected boolean isExhaustive = false;
    private boolean isExhaustiveWasSet = false;
    protected int randomSimulations = -1;
    protected int stepLimit = -1;

    public PlanTest() {
        super();
    }

    public String getDomain() {
        return domain;
    }

    public void setDomain(String domain) {
        this.domain = domain;
    }

    public String getTask() {
        return task;
    }

    public void setTask(String task) {
        this.task = task;
    }

    public String getInitialPlan() {
        return initialPlan;
    }

    public void setInitialPlan(String initialPlan) {
        this.initialPlan = initialPlan;
    }

    public int getPlans() {
        return plans;
    }

    public void setPlans(int plans) {
        this.plans = plans;
    }

    public boolean getIsExhaustive() {
        return isExhaustive;
    }

    public void setIsExhaustive(boolean isExhaustive) {
        this.isExhaustive = isExhaustive;
	this.isExhaustiveWasSet = true;
    }

    public int getRandomSimulations() {
        return randomSimulations;
    }

    public void setRandomSimulations(int randomSimulations) {
        this.randomSimulations = randomSimulations;
    }

    public int getStepLimit() {
	return stepLimit;
    }

    public void setStepLimit(int stepLimit) {
	this.stepLimit = stepLimit;
    }

    /**
     * Takes field values from the specified PlanTest which will
     * usually be an instance of {@link PlanTestDefaults}.
     * Only some fields can be defaulted; currently they are
     * {@link #domain}, {@link #randomSimulations}, {@link #stepLimit},
     * and {@link #isExhaustive}.
     */
    void takeDefaults(PlanTest defaults) {
	// Only some fields can be defaulted.
	if (domain == null)
	    setDomain(defaults.getDomain());
	if (randomSimulations == -1)
	    setRandomSimulations(defaults.getRandomSimulations());
	if (stepLimit == -1)
	    setStepLimit(defaults.getStepLimit());
	if (!isExhaustiveWasSet && defaults.isExhaustiveWasSet)
	    setIsExhaustive(defaults.getIsExhaustive());
    }

    /**
     * Factory method that returns a new TestRunner that should
     * be used to run this test.  This allows PlanTest subclasses
     * to vary the way in which tests are processed.
     */
    public AutoTester.TestRunner makeTestRunner(AutoTester auto) {
	return auto.new TestRunner(this);
    }

    /**
     * Returns a short textual description of this test.
     * It should include the domain name and a description
     * of the task or problem.  Examples:
     * <pre>
     *   Domain duplicate-effect-test, Task test1
     *   Domain blocks-pure-htn-1, Problem ca,b-abc
     * </pre>
     *
     * <p>The method defined by the PlanTest class calls
     * {@link #getDomain()} and {@link #taskDescription()}.</p>
     *
     * <p>The testDescription() method is called by {@link #toString()}.</p>
     */
    public String testDescription() {
	return
	    "Domain " + getDomain() +
	    ", Task " + taskDescription();
    }

    /**
     * Returns a short textual description of the task or problem
     * that this test gives to the planner.
     *
     * <p>The method defined by the PlanTest class returns the
     * result of calling {@link #getTask()}, if that is not null,
     * and otherwise calls {@link #getInitialPlan()}.  If the
     * initial-plan string begins with this test's domain name,
     * those characters are removed from the description, as
     * are any characters after and including any final ".".
     * For example, if the test has domain "<tt>example</tt>"
     * and initial-plan "<tt>example-init-plan-1.xml</tt>",
     * the result will be "<tt>-init-plan-1</tt>".</p>
     *
     * @see #testDescription()
     */
    public String taskDescription() {
	if (getTask() != null)
	    return getTask();
	else {
	    String initPlan =  Strings.beforeLast(".", getInitialPlan());
	    String domainName = getDomain();
	    if (initPlan.startsWith(domainName))
		return Strings.afterFirst(domainName, initPlan);
	    else
		return initPlan;
	}
    }

    /**
     * Returns "className[...]" wrapped around the result of calling
     * {@link #testDescription()}.
     */
    public String toString() {
	String className = Strings.afterLast(".", this.getClass().getName());
	return className + "[" + testDescription() +"]";
    }

}
