/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue May 22 01:01:00 2007 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

// /\/: The division of labour between this and the model manager
// still isn't right.

/**
 * Stores the current world state and certain things about it.
 */
public class IPlanWorldStateManager extends Ip2WorldStateManager {

    // The fields in Facts are context-dependent, so the patternToFactMap
    // doesn't need to be.  However, we'd have to check for getValue()
    // returning null, which might instead be a bug.  So we make the
    // mapping context-dependent anyway.
    // Map entries can be added but should never be changed.
    protected ContextMap patternToFactMap = new ContextHashMap();

    public IPlanWorldStateManager(Ip2ModelManager modelManager) {
	super(modelManager);
    }

    @Override
    public void reset() {
	super.reset();
	patternToFactMap.clearCompletely();
    }

    @Override
    public void clear() {
	super.clear();
	patternToFactMap.clear();
    }

    @Override
    public void deleteEffect(PatternAssignment pv) {
	// /\/: This should happen only because the IPlanModelManager's
	// setWorldState(Map desiredState) method was called.
	patternToFactMap.remove(pv.getPattern());
	super.deleteEffect(pv);
    }

    public void satisfyConds(PNode at, ListOfConstraint conds, MatchEnv env) {
	// conds is a list of PatternAssignments.
	Debug.noteln("Using env "+ env +" to satisfy "+ conds +" at "+ at);
	modelManager.bindVariables(env);
	// /\/: Have to remove the Variables somewhere.
	conds = (ListOfConstraint)Variable.removeAllVars(conds);
	for (Iterator i = conds.iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() != Refinement.S_WORLD_STATE)
		continue;
	    PatternAssignment pv = c.getPatternAssignment();
	    Debug.noteln("Finding contrib for", pv);
	    Debug.noteln("At", at);
	    satisfy1Cond(pv.getPattern(), pv.getValue(), at);
	}
    }

    protected void satisfy1Cond(LList p, Object v, PNode at) {
	// The pattern and value should be fully bound, with all
	// variables replaced by their values, and should match
	// a world-state entry.  Our job is just to determine
	// the contributor.
	Fact f = (Fact)patternToFactMap.get(p);
	Debug.expect(f != null, "can't find a fact for", p);
	Debug.expectEquals(v, f.getValue(), "fact value doesn't match");
	PNode contrib = f.getContributor();
	if (contrib == null)
	    Debug.noteln("No contributor so presumably initial state.");
	else
	    modelManager.linkAfter(at.getBegin(), contrib.getEnd());
	f.addUser(at);
    }

    @Override
    protected void assign(LList p, Object v, PNode at) {
	// /\/: Assumes conds are all at begin and effects all at end.
	// This should be redone with at, contribs, and users all as
	// node-ends.
	// Note that "at" might be null, and so a Fact's contributor
	// might be null.  In I-Plan, this should happen only for
	// for a p = v that we consider initial state for the planning
	// problem.
	Fact f = (Fact)patternToFactMap.get(p);
	if (f == null) {
	    // A new pattern.  Therefore nothing now depends on its value.
	    Debug.expect(worldStateMap.get(p) == null);
	    f = new Fact(p, v, at);
	    patternToFactMap.put(p, f);
	}
	else if (f.getValue().equals(v)) {
	    // Re-assertion of the same value.
	    PNode c = f.getContributor();
	    if (c != null && at != null && c != at)
		modelManager.linkBefore(c.getEnd(), at.getEnd());
	    f.setContributor(at);
	}
	else {
	    // A change in value.
	    PNode c = f.getContributor();
	    if (at != null) {
		if (c != null && c != at)
		    modelManager.linkBefore(c.getEnd(), at.getEnd());
		for (Iterator i = f.getUsers().iterator(); i.hasNext();) {
		    PNode u = (PNode)i.next();
		    // u.begin -> at.end
		    modelManager.linkBefore(u.getBegin(), at.getEnd());
		}
	    }
	    f.clearUsers();
	    f.setContributor(at);
	    f.setValue(v);
	}
	worldStateMap.put(p, v);
    }

    protected static class Fact {

	protected LList pattern;
	protected ContextValue value =
	    new ContextValue(null);
	protected ContextValue contributor =
	    new TypedContextValue(PNode.class, null);
	protected LLQueue users =
	    new LLQueue();

	public Fact() {
	}

	public Fact(LList pattern, Object value, PNode at) {
	    // N.B. The field initializers have already been executed.
	    this.pattern = pattern;
	    this.value.set(value);
	    this.contributor.set(at);
	}

	public LList getPattern() {
	    return pattern;
	}

	public Object getValue() {
	    return value.get();
	}

	public void setValue(Object v) {
	    value.set(v);
	}

	public PNode getContributor() {
	    return (PNode)contributor.get();
	}


	public void setContributor(PNode at) {
	    contributor.set(at);
	}

	public List getUsers() {
	    return users.contents();
	}

	public void addUser(PNode u) {
	    users.add(u);
	}

	public void clearUsers() {
	    users.clear();
	}

    }

}

