/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 23 03:04:27 2007 by Jeff Dalton
 * Copyright: (c) 2004 - 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;		// for reading initial options
import java.util.*;

import ix.icore.Annotations;
import ix.icore.Report;
import ix.icore.IXAgent;
import ix.icore.plan.Plan;
import ix.iplan.event.*;
import ix.ip2.*;
import ix.iface.util.Reporting;
import ix.util.*;
import ix.util.xml.*;
import ix.util.context.*;
import ix.test.DigestSet;

/**
 * Manages options for an instance of {@link IPlan} or {@link Ip2}.
 */
public class IPlanOptionManager {

    Ip2 ip2;
    SortedMap nameToOptionMap = new TreeMap();
    Opt currentOption;
    Opt optionForInput;
    boolean warnUserOfDelayedInput = true;
    boolean warnUserOfDelayedReport = true;
    boolean useOnlyOneOption = false;
    boolean planSplitsOption =
	Parameters.getBoolean("plan-splits-option", false);
    List optionListeners = new LinkedList();
    boolean noticePlanChangeEvents = true; // for use with reloadViewers() /\/
    int inUndoableTransaction = 0; 	   // nesting level
    boolean nowUndoing = false;
    Gensym.Generator nameGen = new Gensym.Generator();
    PlanEvalManager planEvalManager = new TechnicalPlanEvalManager();

    public IPlanOptionManager(Ip2 ip2) {
	this.ip2 = ip2;
    }

    public List getPlanEvaluators() {
	return planEvalManager.getPlanEvaluators();
    }

    public void connectYourself() {
	new CombinedPlanChangeListener() {
	    protected void eventReceived(EventObject e) {
		planChangeEvent(e);
	    }
	}.connectYourself(ip2);
	ip2.addResetHook(new ResetHook());
    }

    void planChangeEvent(EventObject e) {
	Debug.expect(currentOption != null,
		     "State modified before there's a current option");
	if (noticePlanChangeEvents) {
	    Debug.noteln("Option manager sees", e);
	    currentOption.noteChange(e);
	}
	else
	    Debug.noteln("Option manager ignores", e);
    }

    class ResetHook implements Runnable {
	// Contexts have already been cleared by the time
	// is is called.
	public void run() {
	    Debug.noteln("Resetting", IPlanOptionManager.this);
	    nameToOptionMap.clear();
	    currentOption = null;
	    optionForInput = null;
	    setInitialCurrentOption();
	}
    }

    Predicate1 makePlanFilter() {
	return Parameters.getBoolean("filter-duplicate-plans", true)
	    ? new DigestSet().new IsNewPredciate()
	    : Fn.alwaysTrue;
    }

    /*
     * Option initialization
     */

    public void initOneOption() {
	// Called by agents that just want one option.
	useOnlyOneOption = true;
	if (Parameters.haveParameter("option-directory"))
	    Debug.warn("The option-directory parameter is not supported " +
		       "by this agent.");
	setInitialCurrentOption();
    }

    public void initOptions() {
	// Called by agents that want however many options the user requests.
	useOnlyOneOption = false;
	String optDir = Parameters.getParameter("option-directory");
	if (optDir == null) {
	    setInitialCurrentOption();
	}
	else {
	    if (Parameters.haveParameter("plan"))
		Debug.warn("The plan and option-directory parameters " +
			   "should not be used together.");
	    loadOptions(optDir);
	    String inName = Parameters.getParameter("option-for-input");
	    if (inName != null) {
		Opt inOpt = (Opt)nameToOptionMap.get(inName);
		if (inOpt == null)
		    inOpt = makeTopLevelOption(inName, null);
		setInitialCurrentOption(inOpt);
	    }
	    else if (nameToOptionMap.isEmpty()) { // nothing from opt dir
		setInitialCurrentOption();
	    }
	    else {
		String name1 = (String)nameToOptionMap.firstKey();
		Opt opt1 = (Opt)nameToOptionMap.get(name1);
		setInitialCurrentOption(opt1);
	    }
	}
    }

    void setInitialCurrentOption() {
	String name = Parameters.getParameter("option-for-input", "Option-1");
	setInitialCurrentOption(makeTopLevelOption(name, null));
    }

    void setInitialCurrentOption(Opt option) {
	// setOption(Opt) does too much, so we don't want to use
	// it at this point.
	Debug.noteln("Setting initial current option to", option);
	Debug.expect(currentOption == null,
		     "initial current option set twice");
	option.makeYourselfTheCurrentOption();
	checkContext();
	// Also make it the initial option for input
	optionForInput = option;
    }

    public void loadOptions(String directoryName) {
	Debug.noteln("Reading options from", directoryName);
	// Makes options without setting the current option.
	try {
	    SortedMap plans = readPlans(directoryName);
	    makeTopLevelOptions(directoryName, plans);
	}
	catch (Throwable t) {
	    Debug.displayException("Problem reading options", t);
	    if (!Parameters.isInteractive())
		throw new RethrownException(t);
	}
    }

    void makeTopLevelOptions(String directoryName, SortedMap plans) {
	Debug.noteln("Option plans", plans.keySet());
	// /\/: Should we check using optionNameIsAvailable?
	// /\/: N.B. a KeySet() may be an inner class that lacks
	// a 0-arg constructor.  But conjunction needed a list anyway.
	List conflicts =
	    (List)Collect.intersection
		   (new LinkedList(nameToOptionMap.keySet()),
		    plans.keySet());
	if (!conflicts.isEmpty())
	    throw new IllegalStateException
		("Some options in " + directoryName + 
		 " have the same name as existing options: " +
		 Strings.conjunction(conflicts) + ".");
	for (Iterator i = plans.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    String name = (String)e.getKey();
	    Plan plan = (Plan)e.getValue();
	    makeTopLevelOption(name, plan);
	}
    }

    private Opt makeTopLevelOption(String name, Plan plan) {
	// Create a context for the option, but delay loading the
	// plan until the option first becomes the current option.
	// That way, we avoid bothering the viewers before we have to.
	// We could probably get away without the try-finally,
	// but having it makes the uses of this method more transparent.
	synchronized(ip2) {
	    Context saved = Context.getContext();
	    try {
		Context.setContext(Context.rootContext);
		Context.pushContext();
		return new Opt(name, plan);
	    }
	    finally {
		Context.setContext(saved);
	    }
	}
    }

    /*
     * Option names
     */

    public SortedMap getNameToOptionMap() {
	return nameToOptionMap;
    }

    Opt getOption(String name) {
	Opt opt = (Opt)nameToOptionMap.get(name);
	if (opt != null)
	    return opt;
	else
	    throw new IllegalArgumentException
		("There is no option named " + Strings.quote(name));
    }

    String childName(String parentName, int i) {
	char c = parentName.charAt(parentName.length() - 1);
	return c == '-' || Character.isDigit(c)
	    ? parentName + "." + i
	    : parentName + "-" + i;
    }

    String parentName(String childName) {
	for (int i = childName.length() - 1; i >0; i--) {
	    char c = childName.charAt(i);
	    if (c == '.' || c == '-')
		return childName.substring(0, i);
	}
	throw new ConsistencyException
	    ("Can't find parent name for", childName);
    }

    String nextSiblingName(String baseName) {
	for (int i = 1; ; i++) {
	    String candidate = childName(baseName, i);
	    if (optionNameIsAvailable(candidate))
		return candidate;
	}
    }

    boolean optionNameIsAvailable(String name) {
	if (nameToOptionMap.containsKey(name))
	    return false;
	for (Iterator i = nameToOptionMap.keySet().iterator(); i.hasNext();) {
	    String usedName = (String)i.next();
	    if (usedName.startsWith(name))
		return false;
	}
	return true;
    }

    String wantOptionName(String name) {
	if (!optionNameIsAvailable(name))
	    throw new IllegalArgumentException
		("The option name \"" + name + "\" has already been used.");
	return name;
    }

    /*
     * The current option
     */

    public Opt getOption() {
	checkContext();
	return currentOption;
    }

    public void setOption(String name) {
	setOption(getOption(name));
    }

    protected void setOption(Opt option) {
	Debug.noteln("Setting option to", option);
	Debug.expect(nameToOptionMap.values().contains(option));
	checkContext();
	if (useOnlyOneOption) {
	    Debug.expectSame(currentOption, option);
	    // /\/: We still might need to load a new plan after a replan
	    // and so fall through to do makeYourselfTheCurrentOption().
	}
	else if (option == currentOption) {
	    Debug.noteln(option + " is already current");
	    return;
	}
	option.makeYourselfTheCurrentOption();
	checkContext();
    }

    // /\/: Here are some context utilities indended to help us
    // deal with the problem mentioned at the start of the
    // recordDelayedInput method.

    // /\/: Note that deliverEnqueuedInput will be called
    // when planning finishes, but before the resulting plan
    // is installed in the option.  So things in the delayed
    // messages will be lost in that option.

    String contextDepartureReason = null;
    List specialInputQueue = null;

    Context departContext(String reason) {
	Context current = Context.getContext();
	Debug.noteln("Departing " + current + " because " + reason);
	contextDepartureReason = reason;
	return current;
    }

    void restoreContext(Context saved) {
	Context.setContext(saved);
	Debug.noteln("Returning to " + saved +
		     " after " + contextDepartureReason);
	contextDepartureReason = null;
	if (specialInputQueue != null) {
	    deliverEnqueuedInput();
	}
    }

    void enqueueInput(IPC.InputMessage message) {
	Debug.noteln("Specially delayed input", message.getContents());
	if (specialInputQueue == null)
	    specialInputQueue = new LinkedList();
	specialInputQueue.add(message);
    }

    void deliverEnqueuedInput() {
	checkContext();
	synchronized(ip2) {
	    try {
		for (Iterator i = specialInputQueue.iterator(); i.hasNext();) {
		    IPC.InputMessage m = (IPC.InputMessage)i.next();
		    Object contents = m.getContents();
		    Debug.noteln("Delivering specially delayed", contents);
		    ip2.handleInput(m);
		}
	    }
	    finally {
		specialInputQueue = null;
	    }
	}
    }

    /*
     * The option for input
     */

    public Opt getOptionForInput() {
	return optionForInput;
    }

    public void setOptionForInput(String name) {
	setOptionForInput(getOption(name));
    }

    public void setOptionForInput(Opt option) {
	Debug.noteln("Setting option for input to", option);
	Debug.expect(nameToOptionMap.values().contains(option),
		     "unknown option", option);
	optionForInput = option;
    }

    public boolean canTakeInput() {
	Debug.expect(optionForInput != null, "No option can receive input");
	return optionForInput == currentOption
	    && currentOption.expectsContext();
    }

    public void recordDelayedInput(IPC.InputMessage message) {
	// We get here for one of two reasons.  Either the current option
	// is not the option for input, or else the current context is
	// not what the current option expects.  The context can be
	// wrong if we're planning, or something else that temporarily
	// puts us into different contexts.  /\/: You'd think this
	// couldn't happen, because the thread is busy planning or
	// whatever, but it can if a dialog (even a modal one) is up.
	// Then the SwingUtilities.invokeAndWait in IXAgent that
	// takes input processing into the event thread (which is
	// also where I-X runs) is let through, for some reason. :(
	if (!currentOption.expectsContext()) {
	    // Handle this silently
	    enqueueInput(message);
	    return;
	}
	Debug.expect(optionForInput != null, "No option can receive input");
	String inputOptName = optionForInput.getName();
	if (Parameters.isInteractive() && warnUserOfDelayedInput) {
	    String[] note = {
		"Received " + Reporting.messageDescription(message),
		"It will be processed when option currently selected" +
		" for input,",
		inputOptName + ", becomes the current option.",
		"",
		"Do you want to continue to receive these warnings?"
	    };
	    warnUserOfDelayedInput = 
		Util.dialogConfirms(ip2.getFrame(), note);
	}
	else {
	    Debug.noteln("Received ", Reporting.messageDescription(message));
	    Debug.noteln("It will be processed when " + inputOptName +
			 " becomes the current option.");
	}
	optionForInput.recordDelayedInput(message);
    }

    public void handleReportWhenOptions(IPC.InputMessage message) {
	if (!currentOption.expectsContext()) {
	    // /\/: See recordDelayedInput above.
	    enqueueInput(message);
	    return;
	}
	checkContext();
	Report report = (Report)message.getContents();
	boolean takenNow = false;
	// First see if the current option wants the report.
	// Note that the option for input will always be given
	// the report, but may have to give it to a "Note other
	// reports" activity.
	if (canTakeInput() || ip2.getController().wantsReport(report)) {
	    ip2.handleInputDirectly(message);
	    takenNow = true;
	}
	if (nameToOptionMap.size() < 2)
	    return;
	// Tell the user what's happening
	String[] note = {
	    "Received " + Reporting.messageDescription(message),
	    "It was " + (takenNow ? "delivered" : "not relevant") +
	    " in option " + currentOption.getName(),
	    "It will be processed in other options when they next " +
	    "become the current option."
	};
	if (Parameters.isInteractive() && warnUserOfDelayedReport) {
	    String[] question = {
		"", "Do you want to continue to receive these warnings?"
	    };
	    warnUserOfDelayedReport = 
		Util.dialogConfirms(ip2.getFrame(),
				    Util.appendArrays(note, question));
	}
	else {
	    Debug.notelines(note);
	}
	// Now let other options see the report when they become
	// current. This is a bit like syncState() and will use
	// the pseudo-message mechanism developed for "sync".
	ReportMessage reportMessage = new ReportMessage(report);
	for (Iterator i = nameToOptionMap.values().iterator(); i.hasNext();) {
	    Opt option = (Opt)i.next();
	    if (option != currentOption)
		option.recordDelayedInput(reportMessage);
	}
	checkContext();
    }

    class ReportMessage extends PseudoMessage {
	ReportMessage(Report report) {
	    super(report);
	}
	public void receivedBy(Opt option, Ip2 modelHolder) {
	    Debug.noteln(option + " trying " + 
			 Reporting.messageDescription(this));
	    Debug.noteln("Using agendas of", modelHolder);
	    Debug.expectSame(currentOption, option);
	    Report report = (Report)getContents();
	    // /\/: To be consistent with bahaviour elsewhere, we
	    // should really check whether the option was the option
	    // for input at the time when the report was received.
	    if (modelHolder.getController().wantsReport(report)
		  || option == optionForInput) {
		Debug.noteln("Giving report to", option);
		modelHolder.handleInputDirectly(this);
	    }
	}
    }

    /*
     * Operations on options
     */

    public void newOption(String name) {
	Debug.noteln("New option", name);
	wantOptionName(name);
	setOption(makeTopLevelOption(name, null));
    }

    public void copyOption(String newName) {
	Debug.noteln("Copy", currentOption);
	wantOptionName(newName);
	checkContext();
	setOption(makeTopLevelOption(newName, currentOption.asPlan()));
    }

    public void renameOption(String newName) {
	Debug.noteln("New name for " + currentOption, newName);
	wantOptionName(newName);
	checkContext();
	// A user-requested rename counts as a change,
	// and (/\/) we awkwardly have to create an event
	// to pass to noteChange.
	currentOption.setName(newName);
	currentOption.noteChange(new OptionEvent(this, currentOption));
    }

    public void splitOption() {
	Debug.noteln("Split", currentOption);
	Opt newSibling = currentOption.splitYourself();
	// checkContext(); // not needed since we call setOption.
	setOption(newSibling);
    }

    public boolean plan() {
	Debug.noteln("Plan in", currentOption);
	checkContext();
	Opt hasPlan = currentOption.plan();
	checkContext();
	if (hasPlan == null)
	    return false;	// no plan was found
	else if (hasPlan == currentOption) {
	    Debug.expect(!planSplitsOption);
	    // /\/: We need to load the new plan and get the viewers
	    // to show it, and the easiest way to so that is to call
	    // the current option's makeYourselfTheCurrentOption method.
	    currentOption.makeYourselfTheCurrentOption();
	    checkContext();
	}
	else {
	    Debug.expect(planSplitsOption);
	    setOption(hasPlan);
	}
	return true;		// found a plan
    }

    public void replan() {
	Debug.noteln("Replan in", currentOption);
	Debug.expect(currentOption.allowsReplan(), "Can't replan now");
	checkContext();
	Opt newSibling = currentOption.replan();
	checkContext();
	if (newSibling != null)
	    setOption(newSibling);
    }

    public PlanStats getStats() {
	return currentOption.getStats();
    }

    public void clearOption() {
	Debug.noteln("Clear", currentOption);
	checkContext();
        currentOption.clear();  // not clear how much this should do /\/
	ip2.clearModel();
	ip2.resetViewers();
	reloadViewers();
	checkContext();
    }

    public void clearOptionAllButState() {
	Debug.noteln("Clear all but state", currentOption);
	checkContext();
        currentOption.clearAllButState();  // not clear what this should do /\/
	ip2.clearAllButState();
	ip2.resetViewers();
	reloadViewers();
	checkContext();
    }

    public void deleteOption() {
	Debug.noteln("Delete", currentOption);
	checkContext();
	if (currentOption == optionForInput)
	    throw new IllegalStateException
		(currentOption + " is currently selected for input " +
		 "and so cannot be deleted.");
	// Which option should become the new current option?
	String currentName = currentOption.getName();
	// Get entries for keys strictly less than the current option's
	// name.
	SortedMap head = nameToOptionMap.headMap(currentName);
	// Get entries for keys strictly greater than the current option's
	// name.  The +"\0" trick was taken fromthe tailMap javadoc.
	SortedMap tail = nameToOptionMap.tailMap(currentName+"\0");
	// Take the next option in name order, if it exists,
	// else the previous option in name order, if there
	// is one, else create a new option.
	String newName =
	    (String)(!tail.isEmpty() ? tail.firstKey()
	             : !head.isEmpty() ? head.lastKey()
	             : null);
	currentOption.deleteYourself();
	if (newName != null)
	    setOption(newName);
	else
	    newOption("Option-1");
    }

    public void deleteOptions(List selected) {
	Debug.noteln("Delete Options", selected);
	checkContext();
	// /\/: To avoid manipulating the viewers for each deletion,
	// we don't call deleteOption for each selected option.
	// Besides, that would require making each one the current option.
	boolean deleteCurrentOption = false;
	for (Iterator i = selected.iterator(); i.hasNext();) {
	    String name = (String)i.next();
	    Opt opt = getOption(name);
	    if (opt == currentOption)
		deleteCurrentOption = true;
	    else
		opt.deleteYourself();
	}
	if (deleteCurrentOption)
	    deleteOption();
    }

    /*
    public void syncState() {
	Debug.noteln("Sync state from", currentOption);
	checkContext();
	// This is tricky.  To avoid possibly confusing any world-state
	// viewers, we don't want to change the other options now.
	// However, we can't just give them a plan to load (or modify
	// any existing plans to load) because some options might
	// have delayed messages waiting.  Hence pseudo-messages.
	Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	Map state = mm.getWorldStateMap();
	Plan plan = new Plan();
	plan.setWorldState(state);
	SyncMessage syncMessage = new SyncMessage(plan);
	for (Iterator i = nameToOptionMap.values().iterator(); i.hasNext();) {
	    Opt option = (Opt)i.next();
	    if (option != currentOption)
		option.recordDelayedInput(syncMessage);
	}
	checkContext();
    }
    */

    public void syncState(List selectedRecipients) {
	Debug.noteln("Sync state to", selectedRecipients);
	Debug.noteln("Syna state from", currentOption);
	checkContext();
	// This is tricky.  To avoid possibly confusing any world-state
	// viewers, we don't want to change the other options now.
	// However, we can't just give them a plan to load (or modify
	// any existing plans to load) because some options might
	// have delayed messages waiting.  Hence pseudo-messages.
	Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	Map state = mm.getWorldStateMap();
	Plan plan = new Plan();
	plan.setWorldState(state);
	SyncMessage syncMessage = new SyncMessage(plan);
	for (Iterator i = selectedRecipients.iterator(); i.hasNext();) {
	    String name = (String)i.next();
	    Opt option = getOption(name);
	    Debug.expect(option != currentOption,
			 "appempt to sync with self:", option);
	    option.recordDelayedInput(syncMessage);
	}
	checkContext();
    }

    public static abstract class PseudoMessage extends IPC.BasicInputMessage {
	public PseudoMessage(Object contents) { super(contents); }
	public abstract void receivedBy(Opt option, Ip2 modelHolder);
    }

    static class SyncMessage extends PseudoMessage {
	SyncMessage(Plan plan) {
	    super(plan);
	}
	public void receivedBy(Opt option, Ip2 modelHolder) {
	    Debug.noteln("Syncing state in", option);
	    Debug.noteln("Loading sync state in model of", modelHolder);
	    modelHolder.loadPlan((Plan)getContents());
	}
    }

    /*
     * Undo
     */

    public void undo() {
        Debug.noteln("Undo in", currentOption);
        checkContext();
        try {
            Debug.expect(!nowUndoing, "nested undo");
            nowUndoing = true;
            currentOption.undo();
        }
        finally {
            nowUndoing = false;
        }
        // /\/: We need to get the viewers to show the undone contents,
        // and the easiest way to so that is to call the current option's
        // makeYourselfTheCurrentOption method.
        currentOption.makeYourselfTheCurrentOption();
        checkContext();
        printUndoTrail(currentOption);
    }

    protected final boolean undoIsActive() {
        Debug.expect(inUndoableTransaction >= 0,
                     "exited too many undoable transactions");
        return noticePlanChangeEvents // we're not reloading viewers
            && !nowUndoing;
    }

    public void markUndoPoint(String note) {
        // If we're in a transaction, we don't need to make a new undo-point,
	// so in that case we do nothing.
        if (undoIsActive() && inUndoableTransaction == 0) {
            Debug.noteln("Mark undo point in", currentOption);
            checkContext();
            currentOption.markUndoPoint(note);
            checkContext();
            printUndoTrail(currentOption);
        }
    }

    public void saveUndoAction(UndoAction un) {
        if (undoIsActive()) {
            Debug.noteln("Save undo action " + un + " in " + currentOption);
            checkContext();     // just in case
            currentOption.saveUndoAction(un);
	    if (inUndoableTransaction == 0)
		printUndoTrail(currentOption);
        }
    }

    /**
     * Packages an undoable transaction as one method call.
     *
     * @see Ip2ModelManager#undoableTransaction(String, Runnable)
     */
    public void undoableTransaction(String note, Runnable r) {
        // Note that the "begin" call must be immediately before the "try".
        beginUndoableTransaction(note);
        try {
            r.run();
        }
        finally {
            endUndoableTransaction(note);
        }
    }

    public void beginUndoableTransaction(String note) {
        if (!undoIsActive())
            return;
        Debug.noteln("/-- Begin undoable transaction " +
                     "(" + (inUndoableTransaction+1) + ")" +
                     " for " + note + " in " + currentOption);
        // Nested transactions don't create a new undo-point.
        if (inUndoableTransaction == 0)
            markUndoPoint(note);    // one for the whole transaction
        // This ++ is what's reversed in the "finally".  So we need to be
        // sure it will have happened if we enter the "try".  Since
        // the call to beginUndoableTransaction is outside the "try",
        // the ++ needs to be last - making it right before the "try".
        inUndoableTransaction++;
    }

    public void endUndoableTransaction(String note) {
        if (!undoIsActive())
            return;
        inUndoableTransaction--;
        // Since this is used in a "finally", we want to be sure
        // it doesn't throw an exception.
        try {
            Debug.noteln("\\--> End undoable transaction " +
                         "(" + (inUndoableTransaction+1) + ")" +
                         " for " + note + " in " + currentOption);
            printUndoTrail(currentOption);
        }
        catch (Throwable t) {
            ;                   // ? /\/
        }
    }

    public List<UndoAction> getUndoTrail() {
	return currentOption.undoTrail.getContents();
    }

    public void printUndoTrail() {
	printUndoTrail(currentOption);
    }

    protected void printUndoTrail(Opt option) {
        Debug.noteln("Undo trail:");
        printUndoTrail(3, 5, option.undoTrail.getContents());
    }

    protected void printUndoTrail(int indent, int max, LinkedList trail) {
        for (Iterator i = trail.iterator(); i.hasNext();) {
            UndoAction un = (UndoAction)i.next();
	    if (max <= 0) {
		Debug.noteln(Strings.repeat(indent, " "), "...");
		break;
	    }
            Debug.noteln(Strings.repeat(indent, " "), un.getNote());
            if (un instanceof Opt.UndoPoint) {
                Opt.UndoPoint up = (Opt.UndoPoint)un;
                printUndoTrail(indent + 3, 10, up.undoActions);
            }
	    max--;
        }
    }

    /*
     * Options
     */

    public class Opt {

	String name;
	Plan plan;		// to load when opt next becomes current
	List delayedMessages;	// to receive when opt next becomes current
// 	Context baseContext;
	Context iplanContext;
        UndoTrail undoTrail;
	boolean hasChanged;
	PlanGen planGen = null;
	PlanStats stats = null;
	long lastChangeTimestamp = 0; // to compare with lastEvalTimestamp
	long lastEvalTimestamp = lastChangeTimestamp - 1;
	Map planEvaluations = Collections.EMPTY_MAP;

	/** Create an option with the specified name. */
	Opt(String name) {
	    this(name, null);
	}

	/** Create an option with the specified name and initial plan. */
	Opt(String name, Plan plan) {
	    // The current context is the one we're trying to "capture".
	    // It becomes this Opt's baseContext, and no changes should
	    // be made in that context.  Instead, modifications are made
	    // in a child of that context, the iplanContext.
	    this.name = name;
	    this.plan = plan;
	    this.delayedMessages = new LinkedList();
// 	    this.baseContext = Context.getContext();
	    this.iplanContext = Context.pushContext();
            this.undoTrail = new UndoTrail();
	    this.hasChanged = true; // so first split is down
	    if (nameToOptionMap.get(name) != null)
		throw new ConsistencyException
		    ("There is already an option named " + name);
	    nameToOptionMap.put(name, this);
	    fireOptionAdded(this);
	}

        // /\/: The "clear" methods in this class are late additions -
        // they came in as part of "undo" - and so it's not very clear
        // what they should do.  However, if clearing weren't only for
        // the current option, they would clearly have more to do.

        public void clear() {
            undoTrail.clear();
        }

        public void clearAllButState() {
            undoTrail.clear();
        }

	public String getName() {
	    return name;
	}

	public void setName(String newName) {
	    String oldName = name;
	    name = wantOptionName(newName);
	    nameToOptionMap.remove(oldName);
	    nameToOptionMap.put(newName, this);
	    fireOptionRenamed(this, oldName);
	}

	public boolean expectsContext() {
	    return Context.getContext() == iplanContext;
	}

        public Context getExpectedContext() {
            return iplanContext;
        }

	public PlanStats getStats() {
	    Debug.expect(stats != null, "no stats available from", this);
	    return stats;
	}

	public Map getPlanEvaluations() {
	    return planEvaluations;
	}

	public PlanEvaluation getPlanEvaluation(PlanEvaluator e) {
	    return (PlanEvaluation)getPlanEvaluations().get(e);
	}

	void noteChange(EventObject e) {
	    Debug.noteln(this + " has changed");
	    recordChangeTimestamp();
	    // /\/: Check that we're the current option, but
	    // continue anyway if we're not.  THE TRY-CATCH IS TEMPORARY.
	    try {
		Debug.expectSame(currentOption, this,
				 "Change when not current option");
	    }
	    catch (Throwable t) {
		Debug.displayException(t);
	    }
	    hasChanged = true;
	    if (planGen != null)
		// Replans aren't allowed after a change
		dropPlanGen();
	    evaluatePlan();
	    fireOptionContentsChanged(this, e);
	}

	private void recordChangeTimestamp() {
	    lastChangeTimestamp = System.currentTimeMillis();
	}

	public boolean hasChanged() {
	    return hasChanged;
	}

	public boolean allowsReplan() {
	    return planGen != null;
	}

	public void recordDelayedInput(IPC.InputMessage messsage) {
	    delayedMessages.add(messsage);
	}

	void makeYourselfTheCurrentOption() {
	    ip2.resetViewers();
	    Context.setContext(iplanContext);
	    currentOption = this;
	    reloadViewers();
	    // If we've delayed the loading of a plan, do it now.
	    if (plan != null) {
		Debug.noteln("Load plan for " + this + " in " + iplanContext);
		Debug.expect(noticePlanChangeEvents);
		boolean savedDebug = Debug.off();
		try {
		    noticePlanChangeEvents = false;
		    ip2.loadPlan(plan);
		    plan = null;
		}
		finally {
		    Debug.setDebug(savedDebug);
		    noticePlanChangeEvents = true;
		}
	    }
	    // Now any delayed messages
	    processMessages(delayedMessages, ip2);
	    delayedMessages.clear();
	    // Evaluate the plan
	    evaluatePlan();
	    // Tell listeners we're now current
	    fireOptionSet(currentOption);
	}

	private void processMessages(List messages, Ip2 modelHolder) {
	    for (Iterator i = messages.iterator(); i.hasNext();) {
		IPC.InputMessage m = (IPC.InputMessage)i.next();
		if (m instanceof PseudoMessage)
		    ((PseudoMessage)m).receivedBy(this, modelHolder);
		else
		    modelHolder.handleInputDirectly(m);
	    }
	}

	void evaluatePlan() {
	    Debug.noteln("Evaluating plan in", this);
	    if (lastChangeTimestamp == lastEvalTimestamp)
		Debug.noteln("No need to re-evaluate.");
	    else {
		planEvaluations = planEvalManager.evaluatePlan(ip2, name);
		lastEvalTimestamp = lastChangeTimestamp;
	    }
	}

        // A split copies the plan to a new top-level option
        // (top-level in the implementational sense, that is)
        // because some things - such as the fields of issues and
        // activities - are not context-layered.  Also, planning
        // made copies anyway, so that behaviour was different
        // depending on whether you'd planned or not.

	Opt splitYourself() {	// returns the new option
	    Opt newSibling = hasChanged ? splitDown() : splitAcross();
	    return newSibling;
	}

	private Opt splitDown() {
	    Debug.noteln("Splitting down", this);
	    // Make sure we can get the names we want.
	    String newName = wantOptionName(childName(name, 1));
	    String sibName = wantOptionName(childName(name, 2));
//  	    // Move down one context, so that the old iplanContext
//  	    // becomes the baseContext and can be shared with
//  	    // siblings.
//  	    baseContext = iplanContext;
//  	    iplanContext = new Context(baseContext);
//  	    if (this == currentOption)
//  		Context.setContext(iplanContext);
	    hasChanged = false;
	    setName(newName);
	    Opt sibling = splitAcross();
	    Debug.expectEquals(sibName, sibling.name);
	    return sibling;
	}

	private Opt splitAcross() {
	    Debug.noteln("Splitting across", this);
	    // Debug.expect(!this.hasChanged);
	    String parentName = parentName(name);
	    final String siblingName = nextSiblingName(parentName);
//  	    Context.inContext(baseContext, new Runnable() {
//  		public void run() {
//  		    new Opt(siblingName, null);
//  		}
//  	    });
//  	    Opt sibling = getOption(siblingName);
	    Opt sibling = makeTopLevelOption(siblingName, this.asPlan());
	    if (planGen != null)
		sibling.takePlanGen(planGen);
	    sibling.planEvaluations = planEvaluations;
	    sibling.hasChanged = false;
            // /\/: We can't give the sibling a copy of the undo-trail
            // because UndoActions can refer to specific issues, activities,
            // etc and separate options have copies, not the same objects.
            // It wouldn't work if they were the same objects either,
            // because then changing them in one context would change
            // them in all.  So we'd either have to use contexts for
            // everything undoable or else copy the undo-trail in a
            // clever way.
//  	    Debug.expectSame(baseContext, sibling.baseContext,
//  			     "sibling base contexts don't match");
	    return sibling;
	}

        void undo() {
            if (undoTrail.isEmpty())
                throw new UndoException.NoFurtherUndo();
            UndoAction un = undoTrail.pop();
            Debug.noteln("Undoing", un.getNote());
            un.undo();
            recordChangeTimestamp();
            // evaluatePlan();
            fireOptionContentsChanged(this, new UndoEvent(this));
        }

        void markUndoPoint(String note) {
            Debug.expect(inUndoableTransaction == 0);
            undoTrail.push(new UndoPoint(note));
        }

        void saveUndoAction(UndoAction un) {
            if (inUndoableTransaction > 0) {
                Debug.noteln("Saving in undoable transaction.");
                UndoPoint up = (UndoPoint)undoTrail.getFirst();
                up.saveUndoAction(un);
            }
            else
                undoTrail.push(un);
        }

        // /\/: The UndoPoint class exists in case we decide to save more than
        // just the context.  For example, we might want to save hasChanged.

        public class UndoPoint extends AbstractUndoAction {

            Context savedState;
            LinkedList undoActions = new LinkedList();

            public UndoPoint(String note) {
                super(note);
                Debug.expectSame(iplanContext, Context.getContext());
                this.savedState = iplanContext;
                iplanContext = Context.pushContext();
            }

            public void saveUndoAction(UndoAction un) {
                undoActions.addFirst(un);
            }

            public void undo() {
                Debug.noteln("Undoing " + Opt.this + " to " + savedState);
                Context.setContext(savedState);
                iplanContext = Context.pushContext();
                Debug.noteln("Pushed " + Opt.this + " to " + iplanContext);
                while (!undoActions.isEmpty()) {
                    UndoAction un = (UndoAction)undoActions.removeFirst();
                    Debug.noteln("In "+getNote()+", undoing "+un.getNote());
                    un.undo();
                }
            }

        }

        class UndoTrail {
            private LinkedList<UndoAction> trail =
		new LinkedList<UndoAction>();
            private boolean savedHasChanged;
            UndoTrail() {
                savedHasChanged = hasChanged;
            }
            void clear() {
                trail.clear();
            }
            boolean isEmpty() {
                return trail.isEmpty();
            }
            void push(UndoAction un) {
                if (trail.isEmpty()) {
                    Debug.noteln("Saving hasChanged == " + hasChanged);
                    savedHasChanged = hasChanged;
                }
                trail.addFirst(un);
            }
            UndoAction getFirst() {
                return trail.getFirst();
            }
            UndoAction pop() {
                UndoAction result = trail.removeFirst();
                if (trail.isEmpty()) {
                    Debug.noteln("Restoring hasChanged to " + savedHasChanged);
                    hasChanged = savedHasChanged;
                }
                return result;
            }
            LinkedList getContents() {
                return trail;
            }
        }

        // /\/: plan() and replan() don't call evaluatePlan()
        // because they're always done on the current option
        // and result in a call to makeYourselfTheCurrentOption()
        // - either directly or via setOption(Option) - which
        // calls evaluatePlan().  Ordinary changes in the
        // current option don't to that and so must call
        // evaluatePlan().  This is too complicated.

	Opt plan() {
	    // Make sure there is a plan before we discard any
	    // current ability to replan.
	    PlanGen gen = new PlanGen();
	    Plan p = gen.plan(this.asPlan());
	    PlanStats s = gen.getStats();
	    if (p == null) {
		// No plan was found.  Since we won't create another
		// option, show the stats here.
		stats = s;
		// Return, keeping any current planGen.
		return null;
	    }
	    // We have a plan.
	    if (!planSplitsOption) {
		// Replace the plan in this option.
		if (planGen != null)	// drop any old plan-generator
		    dropPlanGen();
		takePlanGen(gen); 	// take the new plan-generator
		replacePlanWith(p);
		stats = s;
		hasChanged = true;
		recordChangeTimestamp();
		fireOptionContentsChanged(this, new PlanEvent(this));
		return this;
	    }
	    // We're supposed to put the plan in a new option.
	    String childName = nextSiblingName(name);
	    Opt child = makeTopLevelOption(childName, p);
	    child.stats = s;
	    child.takePlanGen(gen);
	    return child;
	}

	Opt replan() {
	    Debug.expect(planGen != null, "lost plan generator for", this);
	    // Debug.expect(!hasChanged, "replan after change in", this);
	    Plan p = planGen.replan();
	    PlanStats s = planGen.getStats();
	    if (p == null) {
		// No plan was found.
		// Since we won't create another option, show stats here.
		stats = s;
		// Replanning is no longer possible.
		planGen.discardYourself();
		fireOptionContentsChanged(this, new NoPlanEvent(this));
		return null;
	    }
	    // We have a new plan.
	    if (useOnlyOneOption) {
		// Put the new plan in the current option.
		replacePlanWith(p);
		return this;
	    }
	    // Make an option to hold the new plan.
	    // [/\/: It might be better to make a new top-level
	    // option (but pick names as if split), because we don't
	    // want to inherit anything from the base context the
	    // sibling shares with this option.]
	    // The first replan splits down, later ones split across.
	    int nPlans = planGen.getNumberPlansReturned();
	    Opt sibling = (!planSplitsOption && nPlans == 2)
		? splitDown() : splitAcross();
	    this.hasChanged = true; // because we re/planned to get here
	    sibling.replacePlanWith(p);
	    sibling.stats = s;
	    // The sibling can replan, sharing our plan-generator,
	    // but counts as changed and should split down.
	    // sibling.takePlanGen(this.planGen); // now done in splitAcross()
	    sibling.hasChanged = true;
	    return sibling;
	}

	// The interaction between splitting and replanning is tricky.
	// Call the option in which plan() is called "the plan option".
	// plan() makes a change.  Think of a replan() as going back
	// to a sibling of the plan option (made before plan() was called)
	// and making a different change.  So a replan() should
	// normally split across and mark the new option as changed.
	// If you get a total of 3 plans, say, it should be as if
	// you had made 3 siblings to start with and then manually
	// changed each one.  They should all be siblings at the
	// same level and should all be marked as changed (so that
	// each will split down).

	// But what about the first replan?  Should it split across?
	// According to the as-if just sketched, it should depend on
	// how the plan option would have split before plan() was
	// called -- because that's what would have happened if we'd
	// made the siblings then.  Instead, we always split down
	// on the first replan().  Think of the "decision to plan"
	// as a kind of virtual change made just before plan()
	// was called.  This way, the plan-replan options are
	// set off as more closely related to each other than to
	// other options.

	void takePlanGen(PlanGen gen) {
	    Debug.expect(planGen == null);
	    planGen = gen;
	    planGen.addClient(this);
	}

	void dropPlanGen() {
	    planGen.removeClient(this);
	    planGen = null;
	}

	void replacePlanWith(Plan newPlan) {
	    if (!delayedMessages.isEmpty()) // should be Debug.expect /\/
		Debug.warn("Trying to replace plan in " + this +
			   " when there are delayed messages.");
	    Context.inContext(iplanContext, new Runnable() {
		public void run() {
		    ip2.clearModel();
		}
	    });
	    plan = newPlan; // will load when this option becomes current
	}

	void deleteYourself() {
	    nameToOptionMap.remove(name);
	    if (planGen != null)
		planGen.removeClient(this);
	    // /\/: Do stuff with contexts.
	    fireOptionDeleted(this);
	}

	public Plan asPlan() {
	    // Only top-level options should have delayed-loading plans,
	    // and if such a plan still hasn't been loaded, the option
	    // has never been the current option, and so all of the
	    // content should be in the delayed plan.  We don't want
	    // to load the plan now, because it would confuse the
	    // viewers.
	    // /\/: It's not only top-level options now (see the replan
	    // and replacePlanWith methods), but we should still be ok
	    // because we call ip2.clearModel() in the option's
	    // iplanContext.
	    // /\/: Delayed messages are a further complication.
	    Plan plan = getPlan();
	    if (delayedMessages.isEmpty())
		return plan;
	    Debug.noteln("Getting plan with delayed messages in", this);
	    // If this were the current option, any messages should
	    // have been processed.
	    Debug.expect(this != currentOption);
	    checkContext();
	    ModelHolder m = ModelHolder.instance();
	    checkContext();
	    m.clearModel();
	    m.loadPlan(plan);
	    processMessages(delayedMessages, m);
	    Plan result = m.getPlan();
	    m.clearModel();
	    checkContext();
	    return result;
	    // /\/: Have we left the delayed messages to be processed
	    // again when this option becomes current?  I think so;
	    // but probably we shouldn't.
	}

	private Plan getPlan() {
	    // /\/: We have to do this:
	    Parameters.setParameter("plan-state-to-save", "*"); //\/
	    if (plan != null)
		return plan;
	    Context saved = Context.getContext();
	    try {
		Context.setContext(iplanContext);
		return ip2.getPlan();
	    }
	    finally {
		Context.setContext(saved);
	    }
	}

	public String toString() {
	    return "option[" + name + "]";
	}

    }

    // "Events" for fireOptionContentsChanged.

    public static class PlanEvent extends EventObject {
	PlanEvent(Opt source) {
	    super(source);
	}
    }

    public static class NoPlanEvent extends EventObject {
	NoPlanEvent(Opt source) {
	    super(source);
	}
    }

    public static class UndoEvent extends EventObject {
        UndoEvent(Opt source) {
            super(source);
        }
    }

    public static class ModelHolder extends PlannerBase {
	// We need a very plain version of Ip2, and PlannerBase
	// is the closest. /\/
	private static ModelHolder instance;
	private ModelHolder() {
	    super(false);	// not stand-alone
	}
	static ModelHolder instance() {
	    if (instance == null)
		instance = newInstance();
	    return instance;
	}
	public static ModelHolder newInstance() {
	    // /\/: We're sometimes called when there's no main agent.
	    IXAgent agent = IXAgent.getAgent();
	    synchronized(agent != null ? agent : IXAgent.class) {
		Context saved = Context.getContext();
		try {
		    Context.setContext(Context.rootContext);
		    ModelHolder inst = new ModelHolder();
		    inst.mainStartup(new String[]{});
		    return inst;
		}
		finally {
		    Context.setContext(saved);
		}
	    }
	}
    }

    /*
     * Plan-generators
     */

    /**
     * Finds plans for options.  A PlanGen represents a replanning
     * capability and can be shared by several sibling options.
     */
    protected class PlanGen {

	String name = nameGen.nextString("PlanGen");
	List clients = new LinkedList();
	Context homeContext;
	Context initialPlanContext;
	Slip slip;
	PlanStats stats;
	int plansReturned = 0;
	Predicate1 planFilter = makePlanFilter();
	int rejectedPlans;

	PlanGen() {
	    Debug.noteln("Making", this);
	}

	PlanStats getStats() {
	    Debug.expect(stats != null, "no stats available from", this);
	    return stats;
	}

	int getNumberPlansReturned() {
	    return plansReturned;
	}

	void addClient(Opt opt) {
	    Debug.noteln(this + " adding client", opt);
	    clients.add(opt);
	}

	void removeClient(Opt opt) {
	    Debug.noteln(this + " removing client", opt);
	    Debug.expect(clients.contains(opt));
	    clients.remove(opt);
	    if (clients.isEmpty())
		vanish();
	}

	void vanish() {
	    Debug.expect(clients.isEmpty(), "vanishing too soon", this);
	    Debug.noteln("No longer need", this);
	    ; // /\/
	}

	void discardYourself() {
	    // We have to iterate over a copy, because the
	    // client list is modified as we go.
	    for (Iterator i = new ArrayList(clients).iterator()
		     ; i.hasNext();) {
		Opt client = (Opt)i.next();
		client.dropPlanGen();
		// vanish() will have been called if that was the last client.
	    }
	    Debug.expect(clients.isEmpty());
	}

	// /\/: Handling the stats object is tricky, becuase calling
	// slip.replan() makes Slip create a new one, and we set
	// slip = null when there's no plan, so we can't always
	// ask Slip for the stats object.  So we get the stats
	// object right after planning.

	Plan plan(Plan initialPlan) {
//  	    synchronized(ip2) {
//  		return do_plan(initialPlan);
//  	    }
//  	}

//  	private Plan do_plan(Plan initialPlan) {
	    Debug.noteln("Plan in", this);
	    homeContext = departContext("planning"); // save caller's context
	    try {
		rejectedPlans = 0;
		createPlanner(initialPlan);
		slip.plan();
		return filteredPlan();
	    }
	    catch (NoPlanException npe) {
		handleNoPlan();
		displayNoPlan();
		return null;
	    }
	    catch (Throwable t) {
		handleNoPlan();
		Debug.displayException(t);
		return null;
	    }
	    finally {
		restoreContext(homeContext);
	    }
	}

	void handleNoPlan() {
	    stats = slip.getStatistics();
	    recordRejectCount(stats);
	    discardPlanner();
	}

	Plan replan() {
//  	    synchronized(ip2) {
//  		return do_replan();
//  	    }
//  	}

//  	private Plan do_replan() {
	    Debug.noteln("Replan in", this);
	    Debug.expect(slip != null, "no planner in", this);
	    homeContext = departContext("replanning");
	    try {
		rejectedPlans = 0;
		slip.replan();
		return filteredPlan();
	    }
	    catch (NoPlanException npe) {
		handleNoPlan();
		displayNoPlan();
		return null;
	    }
	    catch (Throwable t) {
		handleNoPlan();
		Debug.displayException(t);
		return null;
	    }
	    finally {
		restoreContext(homeContext);
	    }
	}

	private void displayNoPlan() {
	    if (Parameters.isInteractive())
		Util.displayAndWait(IPlan.displayFrame(), "No plan was found");
	}

	private Plan filteredPlan() {
	    for (;;) {
		// slip.plan() or slip.replan() has just been called
		// and has found a plan.
		stats = slip.getStatistics();
		Plan p = slip.getPlan();
		if (planFilter.trueOf(p)) {
		    recordRejectCount(stats);
		    plansReturned++;
		    return p;
		}
		rejectPlan(p);
	        slip.replan();
	    }
	}

	private void rejectPlan(Plan p) {
	    Debug.noteln("Plan rejected by", planFilter);
	    rejectedPlans++;
	    if (rejectedPlans % 50 == 0 && Parameters.isInteractive()) {
		// This is taking a while.  Let's see if the
		// user wants to continue.
		String[] message = {
		    rejectedPlans + " duplicate plans have been rejected",
		    "Do you want to continue?"
		};
		if (!Util.dialogConfirms(IPlan.displayFrame(), message))
		    throw new NoPlanException();
	    }
	}

	private void recordRejectCount(PlanStats stats) {
	    if (rejectedPlans > 0)
		stats.recordStat("Number of rejected plans",
				 new Integer(rejectedPlans));
	}
	
	// We need to move plans between our MM which has listenters
	// and the Slip MM which doesn't.  Hence various context
	// manipulations. /\/

	void createPlanner(Plan initialPlan) {
	    Context.setContext(Context.rootContext);
	    initialPlanContext = Context.pushContext();
	    slip = new Slip(false);
	    slip.mainStartup(new String[]{});
	    slip.setDomain(ip2.getDomain());
	    slip.loadPlan(initialPlan);
	}

	void discardPlanner() {
	    initialPlanContext.discard();
	    initialPlanContext = null;
	    // slip.reset();
	    slip = null;
	}

	public String toString() {
	    return name + "[for " + clients + "]";
	}

    }

    /*
     * OptionListeners
     */

    public void addOptionListener(OptionListener listener) {
	Debug.noteln("Adding OptionListener", listener);
	optionListeners.add(listener);
    }

    void fireOptionSet(Opt opt) {
	OptionEvent e = new OptionEvent(this, opt);
	for (Iterator i = optionListeners.iterator(); i.hasNext();) {
	    OptionListener l = (OptionListener)i.next();
	    l.optionSet(e);
	}
    }

    void fireOptionAdded(Opt opt) {
	OptionEvent e = new OptionEvent(this, opt);
	for (Iterator i = optionListeners.iterator(); i.hasNext();) {
	    OptionListener l = (OptionListener)i.next();
	    l.optionAdded(e);
	}
    }

    void fireOptionRenamed(Opt opt, String oldName) {
	OptionEvent e = new OptionEvent(this, opt);
	for (Iterator i = optionListeners.iterator(); i.hasNext();) {
	    OptionListener l = (OptionListener)i.next();
	    l.optionRenamed(e, oldName);
	}
    }

    void fireOptionContentsChanged(Opt opt, EventObject how) {
	OptionEvent e = new OptionEvent(this, opt);
	for (Iterator i = optionListeners.iterator(); i.hasNext();) {
	    OptionListener l = (OptionListener)i.next();
	    l.optionContentsChanged(e, how);
	}
    }

    void fireOptionDeleted(Opt opt) {
	OptionEvent e = new OptionEvent(this, opt);
	for (Iterator i = optionListeners.iterator(); i.hasNext();) {
	    OptionListener l = (OptionListener)i.next();
	    l.optionDeleted(e);
	}
    }

    /*
     * Utilities
     */

    /**
     * Called to check that the current context is the right
     * context for the current option and issue a warning if
     * it is not.
     *
     * @see Debug#warn(String)
     */
    private void checkContext() {
	// See also Opt#expectsContext().
	Context fromOpt = currentOption.getExpectedContext();
	Context current = Context.getContext();
	if (fromOpt != current)
	    Debug.warn("The current option, " + currentOption + ", " +
		       "is for " + fromOpt + ", " +
		       "but the current context is " + current);
    }

    /**
     * Returns a map from (file / option) names to Plans.
     */
    public SortedMap readPlans(String directoryName) {
	// Find all the files in the directory that might contain plans.
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	Map filesToPlans = fsm.readAllObjects(Plan.class, directoryName);
	SortedMap plans = new TreeMap();
	for (Iterator i = filesToPlans.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    File f = (File)e.getKey();
	    Plan p = (Plan)e.getValue();
	    String name = Strings.beforeLast(".", f.getName());
	    if (plans.get(name) != null)
		throw new IllegalArgumentException
		    ("There are two plans named " + Strings.quote(name) +
		     " in " + directoryName);
	    plans.put(name, p);
	}
	return plans;
    }

    /**
     * Returns a map from option names to Plans.
     */
    SortedMap getOptionsAsPlans() {
	SortedMap plans = new TreeMap();
	for (Iterator i = nameToOptionMap.values().iterator(); i.hasNext();) {
	    Opt option = (Opt)i.next();
	    plans.put(option.getName(), option.asPlan());
	}
	return plans;
    }

    // /\/: This is a (hopefully) temporary expedient, because
    // the viewers don't have a proper reload function.
    // /\/: There will be a problem if viewers treat newBindings
    // events as something more than a reason to redisplay.
    // If any viewer tries to keep track of variables and their
    // values, it will probably become confused by context changes.
    void reloadViewers() {
	Debug.expect(noticePlanChangeEvents);
	boolean savedDebug = Debug.off();
	try {
	    noticePlanChangeEvents = false;
	    ip2.reloadViewers();
	}
	finally {
	    Debug.setDebug(savedDebug);
	    noticePlanChangeEvents = true;
	}
    }

}
