/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 23 04:42:19 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import ix.icore.domain.Domain;
import ix.icore.plan.Plan;

import ix.util.*;

/**
 * A Planner that returns only plans that pass a filter.
 * The filter is a {@link Predicate1} that should return true
 * for a plan that should be returned and false for one that
 * should not.  The predicate is allowed to have "state"
 * and so may not always return the same value for the same
 * plan.  That lets filtering be used for such things as
 * eliminating duplicate plans.
 */
public class FilteredPlanner implements Planner {

    protected Planner planner;
    protected Predicate1 filter;
    protected boolean filterPlans = true;

    protected Plan plan = null;

    public FilteredPlanner(Planner planner, Predicate1 filter) {
	this.planner = planner;
	this.filter = filter;
	this.filterPlans = Parameters.getBoolean("filter-plans", filterPlans);
    }

    public void setDomain(Domain domain) {
	planner.setDomain(domain);
    }

    public void loadPlan(Plan plan) {
	planner.loadPlan(plan);
    }

    public void plan() {
	planner.plan();
	plan = filteredPlan();
    }

    public void replan() {
	planner.replan();
	plan = filteredPlan();
    }

    public Plan getPlan() {
	if (plan != null)
	    return plan;
	else
	    throw new IllegalStateException
		("This planner has not yet found a plan.");
    }

    public PlanStats getStatistics() {
	return planner.getStatistics();
    }

    private Plan filteredPlan() {
	int rejectedPlans = 0;
	for (;;) {
	    // plan() or replan() has just been called and has found a plan.
	    PlanStats stats = planner.getStatistics();
	    Plan p = planner.getPlan();
	    if (!filterPlans)
		return p;
	    if (filter.trueOf(p)) {
		recordRejectCount(stats, rejectedPlans);
		return p;
	    }
	    rejectedPlans++;
	    rejectPlan(p, rejectedPlans);
	    planner.replan();
	}
    }

    private void rejectPlan(Plan p, int rejectedPlans) {
	Debug.noteln("Plan rejected by", filter);
	if (rejectedPlans % 50 == 0 && Parameters.isInteractive()) {
	    // This is taking a while.  Let's see if the
	    // user wants to continue.
	    String[] message = {
		rejectedPlans + " duplicate plans have been rejected",
		"Do you want to continue?"
	    };
	    if (!Util.dialogConfirms(IPlan.displayFrame(), message))
		throw new NoPlanException();
	}
    }

    private void recordRejectCount(PlanStats stats, int rejectedPlans) {
	if (rejectedPlans > 0)
	    stats.recordStat("Number of rejected plans",
			     new Integer(rejectedPlans));
    }
	
}
