/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Jun  3 02:03:05 2007 by Jeff Dalton
 * Copyright: (c) 1999, 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.icore.plan.Plan;
import ix.icore.process.*;

import ix.ip2.Ip2;		// model holder for testing

import ix.util.*;
import ix.util.xml.*;

/**
 * Sorts node-ends into execution-stages.
 *
 * <p>Each "stage" is a list of node-ends that could execute at the
 * same time (as each other).  The stages are returned as a list,
 * in the order in which they would have to execute.</p>
 *
 * <p>It's assumed that a node-end can execute at its earliest
 * start time, provided that all the node ends linked before
 * it have executed.</p>
 *
 * <p>Note that the stages are not derermined solely by the times
 * calculated by the TPN, because links are also taken into account.
 * There can therefore be more than one stage with the same
 * numeric start time.</p>
 *
 * <p>Use:
 * <pre>
 *   List<PNodeEnd> ends = ...
 *   List<List<PNodeEnd>> stages = new ExecutionStages(ends).getStages();
 * </pre></p>
 */
public class ExecutionStages {

    protected List<List<PNodeEnd>> stages;

    public ExecutionStages(List<PNodeEnd> ends) {
	stages = makeExecutionStages(ends);
    }

    public List<List<PNodeEnd>> getStages() {
	return stages;
    }

    protected List<List<PNodeEnd>> makeExecutionStages(List<PNodeEnd> ends) {
	List<List<PNodeEnd>> stages = new LinkedList<List<PNodeEnd>>();
	List<PNodeEnd> waiting = new LinkedList<PNodeEnd>(ends);
	Map marks = new IdentityHashMap();
	long count = 0;
	// Start by putting the node-ends in order by earliest start time.
	sortByEst(waiting);
	while (!waiting.isEmpty()) {
	    // Simulate the passage of time.
	    long now = waiting.get(0).getMinTime();
	    // Get node-ends that could execute now,
	    // and remove them from the "waiting" list.
	    List<PNodeEnd> ready = new LinkedList<PNodeEnd>();
	    for (Iterator<PNodeEnd> i = waiting.iterator(); i.hasNext();) {
		PNodeEnd e = i.next();
		long t = e.getMinTime();
		if (t > now)
		    break;
		else if (t == now) {
		    if (isReadyByLinks(e, marks)) {
			count++;
			ready.add(e);
			i.remove();
		    }
		}
		else
		    throw new ConsistencyException // t < now
			("Unexpected t < now", e);
	    }
	    // Check that something was actually ready.  If not,
	    // it means that a node-end with est = now had a predecessor
	    // that hadn't already executed.
	    if (ready.isEmpty())
		throw new ConsistencyException
		    ("Couldn't find any ready node-end among ", waiting);
	    // Record the ready ends as a stage and pretend to execute them.
	    stages.add(ready);
	    for (PNodeEnd r: ready)
		marks.put(r, Boolean.TRUE);
	}
	Debug.expect(count == ends.size(), "lost some node-ends");
	return stages;
    }

    private boolean isReadyByLinks(PNodeEnd e, Map marks) {
	for (Object pre: e.getPredecessors())
	    if (marks.get(pre) != Boolean.TRUE)
		return false;
	return true;
    }

    protected void sortByEst(List<PNodeEnd> ends) {
	Collections.sort(ends, new Comparator<PNodeEnd>() {
	    public int compare(PNodeEnd a, PNodeEnd b) {
		return Long.signum(a.getMinTime() - b.getMinTime());
	    }
	});
    }

    /**
     * Test program.
     */
    public static void main(String[] argv) {
	Debug.off();
	Ip2 ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);	// should load a plan
	Debug.on = true;
	List<PNodeEnd> ends = ip2.getIp2ModelManager().getNodeEnds();
	List<List<PNodeEnd>> stages = new ExecutionStages(ends).getStages();
	for (List<PNodeEnd> stage: stages) {
	    for (PNodeEnd end: stage) {
		System.out.println(end.getMinTime() + " " + end);
	    }
	    System.out.println("-----");
	}
    }

}
