/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Jun  3 16:22:07 2007 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2007, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.PrintStream;
import java.net.URL;
import java.util.*;

import ix.iplan.*;
import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.plan.build.*;
import ix.icore.domain.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;
import ix.util.context.Context;

/**
 * Runs a list of {@link PlanTest}s.
 *
 * <p><i>This class is based on parts of O-Plan.</i></p>
 *
 * @see PlanCheckingSimulator
 * @see SanityChecker
 */
public class AutoTester {

    protected PrintStream traceOut = Debug.out;

    protected int numberOfTests = 0;

    protected int numberOfPlans = 0;

    protected List failedTests = null;

    protected SlipStats statSummary = null;

    protected PlanTestDefaults planTestDefaults = null;

    protected boolean planFromPlans = false;

    protected Map<String,PlanTestGroup> nameToTestGroup =
	new LinkedHashMap<String,PlanTestGroup>();

    public AutoTester() {
    }

    public static void main(String[] argv) {
	Debug.off();
	Parameters.setIsInteractive(false); // change default.
	Parameters.processCommandLineArguments(argv);
	AutoTester auto = new AutoTester();
	auto.setPlanFromPlans(Parameters.getBoolean("plan-from-plans", false));
	String testListResource =
	    Parameters.getParameter("test-list",
				    "test-domains/standard-tests.xml");
	List tests = auto.readTestList(testListResource);
	auto.runTestList(tests);
    }

    void setPlanTestDefaults(PlanTestDefaults defaults) {
	this.planTestDefaults = defaults;
    }

    void setPlanFromPlans(boolean v) {
	this.planFromPlans = v;
    }

    void definePlanTestGroup(PlanTestGroup group) {
	traceln("\nDefining", group);
	nameToTestGroup.put(group.getName(), group);
    }

    PlanTestGroup getPlanTestGroup(String name) {
	return nameToTestGroup.get(name);
    }

    /*
     * Running tests
     */

    void runTestList(List tests) {
	Debug.noteln("Tests", XML.objectToXMLString(tests));
	failedTests = new LinkedList();
	statSummary = new SlipStats(null);
	planTestDefaults = null;
	numberOfTests = 0;
	numberOfPlans = 0;
	// Run the tests.
	testLoop(tests);
	// Output a summary of the results.
	if (failedTests.isEmpty())
	    traceln("No failed tests.");
	else {
	    traceln(failedTests.size() + " failed tests:");
	    for (Iterator i = failedTests.iterator(); i.hasNext();) {
		PlanTest failed = (PlanTest)i.next();
		traceln(failed.testDescription());
	    }
	}
	traceln("");
	traceln("Statistics totals for " + numberOfTests + " tests" +
		" and " + numberOfPlans + " plans:");
	statSummary.report(traceOut);
    }

    void testLoop(List tests) {
	for (Iterator i = tests.iterator(); i.hasNext();) {
	    PlanTest test = (PlanTest)i.next();
	    // Run the test.
	    try {
		test.makeTestRunner(this).runTest();
	    }
	    catch (Throwable t) {
		t.printStackTrace(traceOut);
		testFailure(test, Debug.describeException(t));
	    }
	    finally {
		// We drop the planner used for the test, and it
		// should be garbage-collected, but contexts are
		// global so we have to explicitly clear them.  /\/
		Context.clearContexts();
	    }
	    traceln("");
	}
    }

    void recordFailure(PlanTest failed) {
	if (!failedTests.contains(failed))
	    failedTests.add(failed);
    }

    /**
     * Conducts a plan-test.
     */
    public class TestRunner {

	PlanTest test;
	int planNumber;
	int remainingReplans;
	Map changedState;
	
	public TestRunner(PlanTest test) {
	    this.test = test;
	}

	public void runTest() {
	    if (planTestDefaults != null)
		test.takeDefaults(planTestDefaults);
	    describeTest();
	    numberOfTests++;
	    // Set up the planner.
	    Slip slip = makeTestPlanner(test);
	    Domain domain = readDomain();
	    Plan initialPlan = initialPlan();
	    Plan taskPlan = taskPlan();
	    if (initialPlan == null && taskPlan == null)
		throw new IllegalArgumentException
		    ("No task or initial plan specified in test " +
		     XML.objectToXMLString(test));
	    slip.setDomain(domain);
	    if (initialPlan != null)
		slip.loadPlan(initialPlan);
	    if (taskPlan != null)
		slip.loadPlan(taskPlan);
	    // Create plans and test them
	    // In the exhaustive case, we want to replan one more time
	    // after we get the desired number of plans to ensure there
	    // are no more.
	    remainingReplans = test.getIsExhaustive()
		? test.getPlans() : test.getPlans() - 1;
	    planNumber = 1;
	    try {
		slip.plan();
		havePlan(slip);
		while (!testFinished()) {
		    planNumber++;
		    remainingReplans--;
		    slip.replan();
		    havePlan(slip);
		}
	    }
	    catch (NoPlanException e) {
		checkNoPlan();
	    }
	}

	private boolean testFinished() {
	    // Called after each plan is found to determine whether
	    // we've done all the planning we're meant to.
	    if (remainingReplans > 0)
		return false;		// more plans are wanted
	    else if (remainingReplans == 0) {
		if (test.getIsExhaustive())
		    testFailure(test, "too many solutions");
		return true;		// no more plans wanted
	    }
	    else
		throw new ConsistencyException("remainingReplans < 0");
	}

	private void havePlan(Slip slip) {
	    numberOfPlans++;
	    describePlan(slip);
	    statSummary.addStats(slip.getStatistics());
	    savePlan(slip);
	    checkPlan(slip);
	    if (changedState != null) {
		saveChangedState(changedState);
	    }
	    // We can also try using the result plan as an initial plan.
	    if (planFromPlans)
		planFromPlan(slip);
	}

	private void checkNoPlan() {
	    // Called when we fail to get a plan, to see if that's ok.
	    if (remainingReplans == 0 && test.getIsExhaustive())
		traceln("No more plans, but that's ok");
	    else
		testFailure(test, "missing solution");
	}

	protected void describeTest() {
	    traceln(test.testDescription());
	}

	protected void describePlan(Slip slip) {
	    traceln("Plan " + planNumber);
	    slip.getStatistics().report(traceOut);
	}

	protected void savePlan(Slip slip) {
	    Plan plan = slip.getPlan();
	    String resultFile = "test-results/" + savedPlanName();
	    XML.writeObject(plan, resultFile);
	}

	protected String savedPlanName() {
	    // The task may be a multi-element pattern and hence contain
	    // spaces; so we replace each space with "-".
	    return
		test.getDomain() +
		"-" + Strings.replace(" ", "-", test.taskDescription()) +
		"-plan-" + planNumber + ".xml";
	}

	protected void saveChangedState(Map m) {
	    Plan plan = new Plan();
	    plan.setWorldState(m);
	    String resultFile = "test-results/" + savedChangedStateName();
	    XML.writeObject(plan, resultFile);
	}

	protected String savedChangedStateName() {
	    // The task may be a multi-element pattern and hence contain
	    // spaces; so we replace each space with "-".
	    return
		test.getDomain() +
		"-" + Strings.replace(" ", "-", test.taskDescription()) +
		"-changed-" + planNumber + ".init";
	}

	protected Domain readDomain() {
	    return (Domain)XML.readObject(Domain.class, fullDomainName(test));
	}

	protected Plan initialPlan() {
	    String initialPlan = test.getInitialPlan();
	    if (initialPlan != null) {
		initialPlan = "test-domains/" + initialPlan; //\/
		return (Plan)XML.readObject(Plan.class, initialPlan);
	    }
	    else
		return null;
	}

	protected Plan taskPlan() {
	    String task = test.getTask();
	    if (task != null) {
		PlanBuilder builder = new SimplePlanBuilder();
		LList taskPattern = Lisp.elementsFromString(task);
		builder.addActivity(new Activity(taskPattern));
		return builder.getPlan();
	    }
	    else
		return null;
	}

	protected void checkPlan(Slip slip) {
	    // Simulate plan execution.
	    PlanCheckingSimulator sim = new PlanCheckingSimulator(slip);
	    sim.setTrace(false);
	    // We always run one simulation using the natural
	    // order for the node-ends.
	    trace("Simulated execution: ");
	    changedState = runSimulation(sim);
	    // We may also run some simulations with permuted orders.
	    int r = test.getRandomSimulations();
	    if (r > 0) {
		sim.setShuffle(true);
		for (int i = 1; i <= r; i++) {
		    trace("Permutation test " + i + ": ");
		    Map changed = runSimulation(sim);
		    if (!changed.equals(changedState))
			trace("Different final state.");
		}
	    }
	}

	protected Map runSimulation(PlanCheckingSimulator sim) {
	    sim.run();
	    if (Debug.on) sim.describeChangedWorldState();
	    // Print a report that describes any problems found.
	    sim.report();
	    if (!sim.getProblems().isEmpty())
		testFailure(test, "problems during simulated execution");
	    return sim.getChangedWorldStateMap();
	}

	protected void planFromPlan(Slip slip) {
	    traceln("Planning from the plan");
	    Context savedContext = Context.getContext();
	    Context initialPlanContext = null;
	    Domain domain = slip.getDomain();
	    Plan plan = slip.getPlan();	// must get in current context /\/
	    try {
		Context.setContext(Context.rootContext);
		initialPlanContext = Context.pushContext();
		Slip slip2 = makeTestPlanner(test);
		slip2.setDomain(domain);
		slip2.loadPlan(plan);
		slip2.plan();
		Plan plan2 = slip2.getPlan();
		savePlanFromPlan(plan2);
		boolean same = new StructuralEquality().equal(plan, plan2);
		traceln("The plan " + (same ? "is" : "is not") +
			" the same as before");
		checkPlan(slip2);
	    }
	    catch(NoPlanException npe) {
		testFailure(test, "could not use the plan as an initial plan");
	    }
	    finally {
		try {
		    Context.setContext(savedContext);
		    if (initialPlanContext != null)
			initialPlanContext.discard();
		}
		catch (Throwable t) {
		    Debug.noteException(t);
		}
	    }
	}

	protected void savePlanFromPlan(Plan plan) {
	    String name = savedPlanName();
	    XML.writeObject(plan, "/tmp/plan-from-" + name);
	}

    }

    /*
     * Trace output
     */

    public void trace(String message) {
	traceOut.print(message);
    }

    public void traceln(String message) {
	traceOut.println(message);
    }

    public void traceln(String message, Object obj) {
	traceOut.println(message + " " + obj);
    }

    public void testFailure(PlanTest failed, String reason) {
	recordFailure(failed);
	traceln("Test failure because", reason);
    }

    /*
     * Utilities.
     */

    List readTestList(String resourceName) {
	// /\/: This should be easier to do.
	Debug.noteln("Loading a test list from", resourceName);
	try {
	    return (List)XML.readObject(List.class, resourceName);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot get a test list from " +
		    Strings.quote(resourceName) + ": " +
		    Debug.describeException(e));
	}
    }

    // /\/: Should this be in TestRunner?
    Slip makeTestPlanner(PlanTest test) {
	// Don't make the planner the main I-X agent and
	// don't give it any command-line arguments.
	// It will still see any parameters we get from the command-line. /\/
	Slip slip = new Slip(false);
	slip.mainStartup(new String[]{});
	if (test.getStepLimit() > 0)
	    slip.setStepLimit(test.getStepLimit());
	return slip;
    }

    String fullDomainName(PlanTest test) {
	String name = test.getDomain();
	String type = Strings.afterLast(".", name);
	// Default the type to ".lsp".
	return "test-domains/"
	       + (name.equals(type) // no type was given
		  ? name + ".lsp"
		  : name);
    }

}
