/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 24 17:01:22 2004 by Jeff Dalton
 * Copyright: (c) 2002 - 2003, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;
import java.util.*;

import ix.ip2.event.*;
import ix.icore.*;
import ix.util.*;
import ix.util.reflect.ClassDescr; // for ObjectCopier /\/
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.xml.XML;		// for debugging output

/**
 * Generates HandlerActions that send queries.
 */
public class QueryHandler extends ActivityHandler {

    // Syntax is: (query ?agent-name &rest ?pattern)

    static final Symbol
	S_QUERY = Symbol.intern("query"),
	S_ANSWER = Symbol.intern("answer");

    static final Object QUERY_BY_KEY = Symbol.intern("query by");
    static final Object QUERY_REF_KEY = Symbol.intern("query ref");
    static final Object ANSWER_KEY = Symbol.intern("query answer");

    public QueryHandler() {
	super("Query agent");
    }

    public List getSyntaxList() {
	return (LList)Lisp.readFromString
	    ("((query ?agent-name ?pattern-element ...))");
    }

    public boolean appliesTo(AgendaItem item) {
	LList pattern = item.getPattern();
	return pattern.length() >= 3 && pattern.get(0) == S_QUERY;
    }

    public void addHandlerActions(AgendaItem item) {
	// appliesTo will have already been called.
	item.addAction(new QueryAction(item));
    }

    /**
     * Sends a query to a specified agent.
     */
    static class QueryAction extends HandlerAction {

	AgendaItem item;

	QueryAction(AgendaItem item) {
	    this.item = item;
	    this.shortDescription = "Send Query";
	}

	public boolean isReady() {
	    return Variable.isFullyBound(getToName());
	}

	private Object getToName() {
	    return item.getPattern().get(1);
	}
	private LList getQueryPattern() {
	    return item.getPattern().drop(2);
	}

	public void handle(AgendaItem item) {
	    Debug.expect(item == this.item);

	    Object ipcName = IXAgent.getAgent().getAgentIPCName();
	    String toName = Variable.removeVars(getToName()).toString();
	    LList queryPattern = getQueryPattern();

	    // Add a listener that will notice any answer to the query
	    item.addItemListener(new AnswerListener());

	    // Make sure the item's activity has an id so that the
	    // answer will go to the right item.
	    item.getAbout().ensureId();

	    // Send activity (answer ?queryPattern) to toName.
	    Activity ans = new Activity(Lisp.list(S_ANSWER, queryPattern));
	    ans.setPattern((LList)Variable.removeVars(ans.getPattern()));
	    ans.setSenderId(Name.valueOf(ipcName));
	    ans.setReportBack(YesNo.YES);
	    ans.setRef(item.getAbout().getId());
	    // /\/: Need to remember the original asker in case
	    // the "answer" activity is passed to another agent.
	    ans.setAnnotation(QUERY_BY_KEY, ipcName);
	    ans.setAnnotation(QUERY_REF_KEY, item.getAbout().getId());
	    IPC.sendObject(toName, ans);

	    item.setStatus(Status.EXECUTING);
	}

    }

    /**
     * A listener that waits for a new report that contains an
     * answer to the query.
     */
    static class AnswerListener extends AgendaItemAdapter {

	AnswerListener() {
	}

	public void newReport(AgendaItemEvent event, Report report) {

	    AgendaItem item = (AgendaItem)event.getSource();
	    LList queryPattern = item.getPattern().drop(2);

	    // /\/: Class of answer may not be LList, because
	    // the XML (if used) may not contain an impl attribute.
	    List answer = (List)report.getAnnotation(ANSWER_KEY);
	    if (answer == null)
		return;

	    // /\/: Convert the answer to an LList, making sure
	    // to convert any sublists too.
	    LList answerPattern = (LList)new DeepCopier() {
		public Object mapList(List obj, ClassDescr cd) {
		    // /\/: Makes 2 copies, one to copy the subobjects,
		    // then another converting that copy to an LList.
		    return LList.newLList((List)super.mapList(obj, cd));
		}
	    }.copy(answer);

	    Debug.noteln("Have answer", answerPattern);

	    // /\/: Match data parameter is not allowed to contain ItemVars,
	    // so we have to use the answerPattern as the pattern parameter
	    // instead.
	    MatchEnv env = Matcher.match(answerPattern, queryPattern);
	    if (env == null) {
		Debug.noteln("Answer " + answerPattern
			     + " does not match query " + queryPattern);
		JOptionPane.showMessageDialog
		    (null, 
		     new String[] {
			"Answer does not match query",
			"Query: " + queryPattern,
			"Answer: " + answerPattern
		     },
		     "Warning",
		     JOptionPane.ERROR_MESSAGE);
		item.setStatus(Status.IMPOSSIBLE);
		return;
            }
	    Map bindings = new HashMap();
	    for (Iterator i = env.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		Object key = e.getKey();
		Object val = e.getValue();
		if (key instanceof Variable && !(val instanceof ItemVar))
		    bindings.put(key, val);
	    }
	    if (!bindings.isEmpty()) {
		Debug.noteln("New bindings", bindings);
		Ip2 ip2 = (Ip2)IXAgent.getAgent();
		ip2.getModelManager().bindVariables(bindings);
	    }
	    
	}

    }

    /**
     * Generates HandlerActions that send answers to queries.
     *
     * @see QueryHandler
     */
    public static class AnswerHandler extends ActivityHandler {

	public AnswerHandler() {
	    super("Answer query");
	}

	public List getSyntaxList() {
	    return (LList)Lisp.readFromString
		("((answer ?agent-name ?query-pattern))");
	}

	public boolean appliesTo(AgendaItem item) {
	    LList pattern = item.getPattern();
	    return pattern.length() == 2
		&& pattern.get(0) == S_ANSWER
		&& item.getAbout().getReportBack() == YesNo.YES;
	}

	public void addHandlerActions(AgendaItem item) {
	    TaskItem about = item.getAbout();
	    String toName = (String)about.getAnnotation(QUERY_BY_KEY);
	    Name toRef = (Name)about.getAnnotation(QUERY_REF_KEY);
	    item.addAction(new AnswerAction(toName, toRef, item));
	}

	/**
	 * Sends an answer to a query.
	 */
	static class AnswerAction extends HandlerAction {

	    String toName;
	    Name toRef;
	    AgendaItem item;
	    LList queryPattern;

	    AnswerAction(String toName, Name toRef, AgendaItem item) {
		this.toName = toName;
		this.toRef = toRef;
		this.item = item;
		this.queryPattern = (LList)item.getPattern().get(1);
		this.shortDescription = "Send answer to " + toName;
	    }

	    public void handle(AgendaItem item) {
		Debug.expect(item == this.item);

		Object ipcName = IXAgent.getAgent().getAgentIPCName();
		TaskItem about = item.getAbout();

		// Send a report containing the answer.
		// The answer is just the query pattern, with (we hope)
		// at least some variable values filled in.
		Report answer = new Report("Answer to query");
		answer.setSenderId(Name.valueOf(ipcName));
		answer.setAnnotation(ANSWER_KEY, 
				     Variable.removeVars(queryPattern));
		answer.setRef(toRef);
		IPC.sendObject(toName, answer);

		// N.B. Making the status COMPLETE will send a
		// SUCCESS report, though it may not go to the same
		// agent that was sent the answer.  /\/
		item.setStatus(Status.COMPLETE);
	    }

	}

    }

}

// Issues:
// * It used to have syntax (query ?agentName ?parameters = ?resultPattern)
//   and not be ready for execution until all variables in the parameters
//   were bound.
