/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Jun  3 19:12:39 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Constructs a plan from the current contents of the model.
 *
 * @see ix.icore.plan.Plan
 */
class PlanMaker {

    // /\/ Duplicates dcl in Ip2ModelManager
    public static final Symbol
	S_WORLD_STATE = Symbol.intern("world-state"),
	S_CONDITION   = Symbol.intern("condition"),
	S_EFFECT      = Symbol.intern("effect");

    Gensym.Generator nameGen = new Gensym.Generator();

    PanelController controller;
    Ip2ModelManager modelManager;
    VariableManager variableManager; // /\/

    Plan plan;
    ListOfPlanNode topNodes;
    ListOfPlanRefinement refinements;
    ListOfPlanIssue topIssues;
    ListOfPlanIssueRefinement issueRefinements;

    Map varToDclMap;
    Map itemToNodeMap = new HashMap(); // ActivityItem -> PlanNode

    PlanMaker(Ip2 ip2) {
	controller = ip2.getController();
	modelManager = (Ip2ModelManager)ip2.getModelManager();
	variableManager = modelManager.variableManager; // /\/
    }

    Plan getPlan() {
	buildPlan();
	return plan;
    }

    Plan buildPlan() {
	plan = new Plan();
	topNodes = new LinkedListOfPlanNode();
	refinements = new LinkedListOfPlanRefinement();
	topIssues = new LinkedListOfPlanIssue();
	issueRefinements = new LinkedListOfPlanIssueRefinement();
	varToDclMap = new HashMap();

	List variables = getVariables(modelManager.getVarEnv());
	ListOfPlanVariableDeclaration varDcls =
	    makeVariableDeclarations(variables);
	Collect.extendMap(varToDclMap, variables, varDcls);

	ListOfConstrainer topConstraints = getTopLevelConstraints();
	ListOfPatternAssignment worldState = getWorldState();

	walkNodes();
	walkIssues();

	topConstraints.addAll(buildNonSiblingOrderings());

	Annotations annotations = modelManager.getAnnotations();

	if (!varDcls.isEmpty())
	    plan.setPlanVariableDeclarations(varDcls);
	if (!topIssues.isEmpty())
	    plan.setPlanIssues(topIssues);
	if (!issueRefinements.isEmpty())
	    plan.setPlanIssueRefinements(issueRefinements);
	if (!topNodes.isEmpty())
	    plan.setPlanNodes(topNodes);
	if (!refinements.isEmpty())
	    plan.setPlanRefinements(refinements);
	if (!topConstraints.isEmpty())
	    plan.setConstraints(topConstraints);
	if (!worldState.isEmpty())
	    plan.setWorldState(worldState);
	if (annotations != null && !annotations.isEmpty())
	    // Safe to just set because we soon deep-copy the plan.
	    plan.setAnnotations(annotations);

	plan = (Plan)new Pass2().copy(plan);

	return plan;

    }

    /**
     * Constructs a deep copy of a plan in which each Variable
     * has been replaced by its value, if it has a value, or else
     * by a PlanVariable.
     */
    class Pass2 extends ObjectCopier {
	public Object mapElement(Object obj) {
	    return obj instanceof Variable
		? mapVariable((Variable)obj)
		: copy(obj);
	}
	Object mapVariable(Variable var) {
	    Object val = var.getValue();
	    return val != null ? val : getPV(var);
	}
	PlanVariable getPV(Variable var) {
	    PlanVariableDeclaration dcl =
		(PlanVariableDeclaration)varToDclMap.get(var);
	    return dcl.getPlanVariable();
	}
    }

    Name genId(String base) {
	return nameGen.nextName(base);
    }

    List getVariables(Map varEnv) {
	// Here we start with all variables and keep only the ones in
	// a specific env, thus retaining the order in which they were
	// created.
	List variables = new LinkedList(Variable.getAllVariables());
	variables.retainAll(varEnv.values());
	// Now remove variables that already have values.
	for (Iterator i = variables.iterator(); i.hasNext();) {
	    Variable v = (Variable)i.next();
	    if (v.getValue() != null)
		i.remove();
	}
	return variables;
    }

    ListOfPlanVariableDeclaration makeVariableDeclarations
				     (List variables) {
	ListOfPlanVariableDeclaration result =
	    new LinkedListOfPlanVariableDeclaration();
	Collect.map(result, variables, new Function1() {
	    public Object funcall(Object arg) {
		Variable v = (Variable)arg;
		return new PlanVariableDeclaration(genId("var"), v);
	    }
	});
	return result;
    }

    ListOfConstrainer getTopLevelConstraints() {
	ListOfConstrainer result = new LinkedListOfConstrainer();
	result.addAll(variableManager.getConstraints());
	result.addAll(modelManager.getOtherConstraints());
	return result;
    }

    ListOfPatternAssignment getWorldState() {
	final List savable = Parameters.getList("plan-state-to-save");
	Map worldStateMap = modelManager.getWorldStateMap();
	return PatternAssignment.mapToAssignments
	    (savable.contains("*") // keep everything?
	     ? worldStateMap
	     : Collect.filter(new StableHashMap(),
			      worldStateMap,
			      new Predicate2() {
		   public boolean trueOf(Object key, Object value) {
		       LList k = (LList)key;
		       return k != Lisp.NIL
			   && savable.contains(k.car().toString());
		   }
	     }));
    }

    void walkIssues() {
	List issueItems = controller.getIssueAgenda().getItems();
	for (Iterator i = issueItems.iterator(); i.hasNext();) {
	    IssueItem item = (IssueItem)i.next();
	    if (item.getLevel() == 0)
		walkTopIssue(item);
	}
    }

    void walkTopIssue(IssueItem item) {
	Issue issue = (Issue)item.getAbout();
	PlanIssue pi = new PlanIssue(genId("issue"), issue);
	topIssues.add(pi);
	if (item.isExpanded()) {
	    buildIssueExpansion(item, pi);
	}
    }

    void buildIssueExpansion(IssueItem item, PlanIssue pi) {

	// Construct the IssueRefinement.
	PlanIssueRefinement ref = new PlanIssueRefinement();
	ref.setId(genId("issue-refinement"));
	issueRefinements.add(ref);

	// Connect the PlanIssue and the IssueRefinement
	pi.setExpansion(ref.getId());
	ref.setExpands(pi.getId());

	// Variables
	List variables = getVariables(item.getVarTable());
	if (!variables.isEmpty()) {
	    ListOfPlanVariableDeclaration varDcls =
		makeVariableDeclarations(variables);
	    Collect.extendMap(varToDclMap, variables, varDcls);
	    ref.setPlanVariableDeclarations(varDcls);
	}

	// Walk children
	List childItems = item.getChildren();
	ListOfPlanIssue childPlanIssues = new LinkedListOfPlanIssue();
	for (Iterator i = childItems.iterator(); i.hasNext();) {
	    IssueItem childItem = (IssueItem)i.next();
	    Issue childIssue = (Issue)childItem.getAbout();
	    PlanIssue childPlanIssue =
		new PlanIssue(genId(pi.getId().toString()),
			      childIssue);
	    childPlanIssues.add(childPlanIssue);
	    if (childItem.isExpanded()) {
		// Finally, the recursion.
		buildIssueExpansion(childItem, childPlanIssue);
	    }
	}
	if (!childPlanIssues.isEmpty()) 
	    ref.setPlanIssues(childPlanIssues);
    }

    void walkNodes() {
	for (Iterator i = modelManager.getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    if (item.getLevel() == 0)
		walkTopNode(item);
	}
    }

    void walkTopNode(ActivityItem item) {
	PlanNode node = makePlanNode("node", item);
	topNodes.add(node);
	if (item.isExpanded()) {
	    buildExpansion(item, node);
	}
	else {
	    // There can be constraints even if it hasn't been expanded.
	    // Sometimes constraints are explicitly added, for instance
	    // to restrict resource use in I-Globe.
	    ListOfConstrainer constraints = getNodeConstraints(item);
	    if (!constraints.isEmpty()) {
		PlanRefinement ref = makePlanRefinement(node);
		ref.setConstraints(constraints);
		ref.setIsNotExpansion();
	    }
	}
    }

    PlanNode makePlanNode(String idBase, ActivityItem item) {
	Activity activity = (Activity)item.getAbout();
	PlanNode node = new PlanNode(genId(idBase), activity);
	itemToNodeMap.put(item, node);
	return node;
    }

    PlanNode getPlanNode(ActivityItem item) {
	PlanNode pn = (PlanNode)itemToNodeMap.get(item);
	Debug.expect(pn != null, "Can't find PlanNode for", item);
	return pn;
    }

    void buildExpansion(ActivityItem item, PlanNode node) {
	// /\/: Confusing because an ActivityItem is also sometimes
	// called a node in the model manager.

	// Construct the refinement that will describe the expansion
	// of the node.
	PlanRefinement ref = makePlanRefinement(node);

	// Variables
	List variables = getVariables(item.getVarTable());
	if (!variables.isEmpty()) {
	    ListOfPlanVariableDeclaration varDcls =
		makeVariableDeclarations(variables);
	    Collect.extendMap(varToDclMap, variables, varDcls);
	    ref.setPlanVariableDeclarations(varDcls);
	}

	// Walk children
	List childItems = item.getChildren();
	ListOfPlanNode childNodes = new LinkedListOfPlanNode();
	for (Iterator i = childItems.iterator(); i.hasNext();) {
	    ActivityItem childItem = (ActivityItem)i.next();
	    PlanNode childNode = makePlanNode(node.getId().toString(),
					      childItem);
	    childNodes.add(childNode);
	    if (childItem.isExpanded()) {
		// Finally, the recursion.
		buildExpansion(childItem, childNode);
	    }
	}
	checkItemToNodeMap(childItems, childNodes);
	if (!childNodes.isEmpty()) 
	    ref.setPlanNodes(childNodes);

	// Constraints
	ListOfConstrainer constraints = getNodeConstraints(item);
	if (!constraints.isEmpty())
	    ref.setConstraints(constraints);

    }

    PlanRefinement makePlanRefinement(PlanNode node) {
	PlanRefinement ref = new PlanRefinement();
	ref.setId(genId("refinement"));
	refinements.add(ref);
	// Connect the node and the refinement.
	node.setExpansion(ref.getId());
	ref.setExpands(node.getId());
	return ref;
    }

    ListOfConstrainer getNodeConstraints(ActivityItem item) {
	// /\/: For now, we have to (re)construct the constraints.
	ListOfConstrainer constraints = new LinkedListOfConstrainer();
	// Orderings
	constraints
	    .addAll(buildChildSiblingOrderings(item));
	// World-state constraints
	constraints
	    .addAll(modelManager.getNodeConditions(item));
	constraints
	    .addAll(buildConstraints(S_WORLD_STATE,
				     S_EFFECT,
				     modelManager.getNodeEffects(item)));
	constraints
	    .addAll(Collect.ensureList(modelManager
				       .getOtherNodeConstraints(item)));
	constraints
	    .addAll(Collect.ensureList(modelManager
				       .getNodeTimeConstraints(item)));
	return constraints;
    }

    void checkItemToNodeMap(List childItems, List childNodes) {
	Debug.expect(childItems.size() == childNodes.size());
	for (Iterator i = childItems.iterator(),
		      n = childNodes.iterator();
	     i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    PlanNode node = (PlanNode)n.next();
	    Debug.expect(getPlanNode(item) == node);
	}
    }

    List buildConstraints(Symbol type, Symbol relation, List assigns) {
	List result = new LinkedList();
	if (assigns == null) return result;
	for (Iterator i = assigns.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    result.add(new Constraint(type, relation, Lisp.list(pv)));
	}
	return result;
    }

    // /\/: Really shouldn't have to build the orderings here
    // rather than somewhere in the MM or CM.

    List buildChildSiblingOrderings(ActivityItem item) {
	// Returns only orderings between siblings.
	final List orderings = new LinkedList();
	walkChildOrderings(item, new OrdWalker() {
	    public void visit(PNodeEnd before, PNodeEnd after) {
		if (before.getNode().isSiblingOf(after.getNode()))
		    orderings.add(makeOrdering(before, after));
	    }
	});
	return orderings;
    }

    // This is a bit tricky.  We need to build the non-singling orderings
    // in the same order they'd be in if the Plan were loaded into a
    // clean model-manager and then turned into a new Plan, and that can
    // be different from the order we get if we just iterate over
    // the nodes that are now in the MM.  [/\/: Not absolutely clear
    // why, but probably because when a node is expanded, the children
    // are added at the end of the node list rather than inserted
    // after their parent, but when we load a Plan, children are
    // placed right after their parent.]  So we have to walk the
    // nodes in an order that visits children right after their
    // parent.  This doesn't matter for correctness, but it makes
    // it easier to compare plans.  For one thing, it depends on
    // the order of top-level nodes, but not on the order in
    // which they were expanded.  It also makes it so that loading
    // a Plan then getting a new Plan from the MM doesn't change
    // the order.

    List buildNonSiblingOrderings() {
	// Called after all PlanNodes have been created and there
	// is a complete itemToNodeMap.
	final List orderings = new LinkedList();
	modelManager.walkTopNodes(new Proc() {
	    public void call(Object node) {
		ActivityItem item = (ActivityItem)node;
		walkNodeOrderings(item, new OrdWalker() {
		    public void visit(PNodeEnd before, PNodeEnd after) {
			if (isNonSiblingOrdering(before, after))
			    orderings.add(makeOrdering(before, after));
		    }
		});
		modelManager.walkNodeChildren(item, this);
	    }
	});
	return orderings;
    }

    /*
    List buildNonSiblingOrderings() {
	// Called after all PlanNodes have been created and there
	// is a complete itemToNodeMap.
	final List orderings = new LinkedList();
	for (Iterator i = modelManager.getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    walkNodeOrderings(item, new OrdWalker() {
		public void visit(PNodeEnd before, PNodeEnd after) {
		    if (isNonSiblingOrdering(before, after))
			orderings.add(makeOrdering(before, after));
		}
	    });
	}
	return orderings;
    }
    */

    boolean isNonSiblingOrdering(PNodeEnd before, PNodeEnd after) {
	// The method name isn't quite right. /\/
	// We know the orderings are all from successor lists.
	ActivityItem b = (ActivityItem)before.getNode();
	ActivityItem a = (ActivityItem)after.getNode();
	End b_e = before.getEnd();
	End a_e = after.getEnd();
	// Skip links between parent node-ends and children.
	if ((b_e == End.BEGIN && b == a.getParent()) ||
	    (a_e == End.END && a == b.getParent()))
	    return false;
	// Skip links between siblings **except at the top level**.
	if (b.getParent() != null && b.isSiblingOf(a))
	    return false;
	// Oherwise, approve the ordering
	return true;
    }

    /*
     * Ordering utilities
     */

    interface OrdWalker {
	public void visit(PNodeEnd before, PNodeEnd after);
    }

    void walkChildOrderings(PNode node, OrdWalker w) {
	for (Iterator i = node.getChildren().iterator(); i.hasNext();) {
	    PNode child = (PNode)i.next();
	    walkNodeOrderings(child, w);
	}
    }

    void walkNodeOrderings(PNode node, OrdWalker w) {
	walkEndOrderings(node.getBegin(), w);
	walkEndOrderings(node.getEnd(), w);
    }

    void walkEndOrderings(PNodeEnd nodeEnd, OrdWalker w) {
	// Skip link from begin to end of the same node.
	PNodeEnd skip = null;
	if (nodeEnd.getEnd() == End.BEGIN)
	    skip = nodeEnd.getNode().getEnd();
	for (Iterator i = nodeEnd.getSuccessors().iterator(); i.hasNext();) {
	    PNodeEnd after = (PNodeEnd)i.next();
	    if (after != skip)
		w.visit(nodeEnd, after);
	}
    }

    Ordering makeOrdering(PNodeEnd before, PNodeEnd after) {
	ActivityItem beforeItem = (ActivityItem)before.getNode();
	ActivityItem afterItem = (ActivityItem)after.getNode();
	PlanNode beforePlanNode = getPlanNode(beforeItem);
	PlanNode afterPlanNode = getPlanNode(afterItem);
	return new Ordering
	    (new NodeEndRef(before.getEnd(), beforePlanNode.getId()),
	     new NodeEndRef(after.getEnd(), afterPlanNode.getId()));
    }

}
