/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue May 22 01:52:16 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.*;
import ix.icore.plan.inspect.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Loads a plan into the model.
 */
public class PlanInstaller {

    // "Node" is potentially confusing.  In the installer, we use "node"
    // only for PlanNodes; when talking of ActivityItems, we use "item"
    // even though ActivityItem is an indirect subclass of PNode and
    // some other parts of the model manager refer to such things as
    // "nodes".  /\/

    PanelController controller;
    Ip2ModelManager modelManager;
    VariableManager variableManager; // /\/

    Plan plan;

    // Maps a PlanVariable id to a Variable.
    // We have to map from the ids, because PlanVariable objects
    // will not be unique if the plan was read from XML.  /\/
    Map idToVarMap = new HashMap();

    // Maps a PlanNode id to an ActivityItem.
    // Doesn't currently apply to issues.
    Map idToItemMap = new HashMap();

    public PlanInstaller(Ip2 ip2, Plan plan) {
	this.plan = plan;
	this.controller = ip2.getController();
	this.modelManager = (Ip2ModelManager)ip2.getModelManager();
	this.variableManager = modelManager.variableManager; // /\/
    }

    public void installPlan() {
	// Check that the new plans fits with the corrent
	// contents of this I-P2.
	checkPlanCompatibility();

	// Turn off automatic status computation while installing
	// the plan.
	boolean savedComputeStatus = PNode.computeStatus;
	PNode.computeStatus = false;
	try {
	    do_installPlan();
	}
	finally { PNode.computeStatus = savedComputeStatus; }

	// Now deal with plans that didn't give us proper status
	// values - usually by leaving things BLANK.  /\/: We
	// probably don't handle all possible cases correctly,
	// perhaps not even all reasonable cases.
	walkInstalledPlan(new PlanWalker() {
	    public void visitAgendaItem(AgendaItem item) {
		Debug.noteln("Revisiting", item);
		item.computeStatus();
		// Don't recurse to children.
		// No, do.  [JD, May 3, 2004]
		walkInstalledChildren(item, this);
	    }
	});

	// /\/: Tell the var manager to recalculate.
	// Not clear where we ought to do this ...
	variableManager.recalculate();
	variableManager.showState();

	// Redo all handler-actions.  When a plan is loaded,
	// activities look "not expanded" when first added,
	// because the expansion (children etc) is added later.
	// So expand-actions are added even if they make no
	// sense.  This gets rid of them, and any others that
	// no longer belong.
	walkInstalledPlan(new PlanWalker() {
	    public void visitAgendaItem(AgendaItem item) {
		item.clearActions();
		// Put in handlers
		// This also shows how activities were expanded
		// (see CompletionHandler.NoAction).
		getAgendaFor(item).addHandlerActions(item);
		// Recurse to children.
		walkInstalledChildren(item, this);
	    }
	});

    }

    public static interface PlanWalker {
	void visitAgendaItem(AgendaItem item);
    }

    public void walkInstalledPlan(PlanWalker w) {
	// Activities
	for (Iterator i = Collect.iterator(plan.getPlanNodes());
	     i.hasNext();) {
	    PlanNode node = (PlanNode)i.next();
	    Activity activity = node.getActivity();
	    ActivityItem item = findItemForActivity(activity);
	    w.visitAgendaItem(item);
	}
	// Issues
	for (Iterator i = Collect.iterator(plan.getPlanIssues());
	     i.hasNext();) {
	    PlanIssue pi = (PlanIssue)i.next();
	    Issue issue = pi.getIssue();
	    IssueItem item = findItemForIssue(issue);
	    w.visitAgendaItem(item);
	}
    }

    public void walkInstalledChildren(AgendaItem item, PlanWalker w) {
	// Recursion-step utility
	for (Iterator i = item.getChildren().iterator(); i.hasNext();) {
	    AgendaItem child = (AgendaItem)i.next();
	    w.visitAgendaItem(child);
	}
    }

    Agenda getAgendaFor(AgendaItem item) {
	if (item instanceof ActivityItem)
	    return controller.getActivityAgenda();
	else if (item instanceof IssueItem)
	    return controller.getIssueAgenda();
	else
	    throw new ConsistencyException("No agenda for", item);
    }

    void checkPlanCompatibility() {
	PlanInspector pi = new PlanInspector(plan);
	List issues = controller.getIssueAgenda().getItems();
	List activities = controller.getActivityAgenda().getItems();
	if (Collect.haveCommonElements(pi.getIssueIdSet(),
				       idSet(issues))
	    || Collect.haveCommonElements(pi.getActivityIdSet(),
					  idSet(activities))) {
	    throw new UnsupportedOperationException
		("The plan contains issues or activities " +
		 "which are already present in this I-P2.");
	}
    }

    Set idSet(List agendaItems) {
	Set result = new HashSet();
	for (Iterator i = Collect.iterator(agendaItems); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    TaskItem about = item.getAbout();
	    if (about.getId() != null)
		result.add(about.getId());
	}
	return result;
    }

    /*
     * Main installer
     */

    void do_installPlan() {

	// Variables
	List allDcls = plan.getAllPlanVariableDeclarations();
	for (Iterator i = allDcls.iterator(); i.hasNext();) {
	    PlanVariableDeclaration dcl =
		(PlanVariableDeclaration)i.next();
	    idToVarMap.put(dcl.getId(), mapVariableDeclaration(dcl));
	}
	Debug.noteln("idToVarMap", idToVarMap);

	// Replace PlanVariables with Variables throughout the plan
	plan = (Plan)new Pass1().copy(plan);

	// World state
	if (plan.getWorldState() != null)
	    modelManager.handleEffects(plan.getWorldState());

	// Issues
	for (Iterator i = Collect.iterator(plan.getPlanIssues());
	     i.hasNext();) {
	    PlanIssue pi = (PlanIssue)i.next();
	    Issue issue = pi.getIssue();
	    controller.addIssue(issue);
	    if (pi.getExpansion() != null)
		installIssueExpansion(pi);
	}

	// Activities / nodes
	// For the top level, we have the controller create the
	// ActivityItem from each Activity.  One reason for this
	// is that the class of the ActivityItem may vary depending
	// on the class of the Activity, e.g. WaitForReportsActivity.
	for (Iterator i = Collect.iterator(plan.getPlanNodes());
	     i.hasNext();) {
	    PlanNode node = (PlanNode)i.next();
	    Activity activity = node.getActivity();
	    controller.addActivity(activity);
	    idToItemMap.put(node.getId(), findItemForActivity(activity));
	    if (node.getExpansion() != null)
		installExpansion(node);
	}

	// Make sure viewers know about the status of the intalled
	// activities and issues.
	walkInstalledPlan(new PlanWalker() {
	    public void visitAgendaItem(AgendaItem item) {
		Debug.noteln("Visiting", item);
		// Set status in node-ends
		item.assignStatus(item.getStatus());
		item.fireStatusChanged();
		walkInstalledChildren(item, this);
	    }
	});

	// Debug.noteln("idToItemMap", idToItemMap);

	// Constraints
	List cons = plan.getConstraints();
	if (cons != null) {
	    // Add any constraints not equal to existing ones.
	    // cons = (List)Collect.difference(cons, otherConstraints);
	    // otherConstraints.addAll(cons);

	    // Add all top-level constraints.
	    // modelManager.addConstraints(cons);
	    for (Iterator i = cons.iterator(); i.hasNext();) {
		Constrainer c = (Constrainer)i.next();
		modelManager.addConstraint(idToItemMap, c);
	    }
	}

	// Annotations
	Annotations ann = plan.getAnnotations();
	if (ann != null) {
	    modelManager.takeAnnotations(ann);
	}

    }

    public Variable mapVariableDeclaration(PlanVariableDeclaration dcl) {
	// Called when constructing the Plan that's given to the installer.
	// N.B. We see a given id only once; the caller has to remember
	// the Variable for that id.
	ItemVar varName = (ItemVar)dcl.getName();
	if (dcl.getScope() == VariableScope.GLOBAL) {
	    Variable v = modelManager.getVariable(varName);
	    if (v == null) {
		v = new Variable(varName);
		modelManager.addVariable(v);
	    }
	    return v;
	}
	else {
	    // Local var - here's where the result memo really matters,
	    // because here we always create a new Variable.
	    return new Variable(varName);
	}
    }

    /**
     * Constructs a copy of a plan in which each PlanVariable
     * has been replaced by a Variable.
     */
    class Pass1 extends ObjectCopier{
	public Object mapElement(Object obj) {
	    return obj instanceof PlanVariable
		? mapPlanVariable((PlanVariable)obj)
		: copy(obj);
	}
	Variable mapPlanVariable(PlanVariable pv) {
	    // The Variable should already be known.
	    Variable v = getVariableFromId(pv.getId());
	    Debug.expect(v != null, "Unknown variable", pv);
	    // Debug.noteln("Mapping " + pv + " to " + v);
	    return v;
	}
    }

    Variable getVariableFromId(Name id) {
	// The id can come from a PlanVariable or from a
	// PlanVariableDeclaration.
	return (Variable)idToVarMap.get(id);
    }

    void installIssueExpansion(PlanIssue pi) {
	Issue issue = pi.getIssue();
	IssueItem item = findItemForIssue(issue);
	PlanIssueRefinement refine =
	    plan.getPlanIssueRefinement(pi.getExpansion());
	Debug.expect(refine.getExpands().equals(pi.getId()));

	// Variables
	item.setVarTable
	    (makeVarTable(item, refine.getPlanVariableDeclarations()));

	// Subissues
	List childPlanIssues = refine.getPlanIssues();
	// Create and attach children
	if (childPlanIssues != null) {

	    // Make list of child items.
	    List childItems = new LinkedList();
	    for (Iterator i = childPlanIssues.iterator(); i.hasNext();) {
		PlanIssue childPlanIssue = (PlanIssue)i.next();
		Issue childIssue = childPlanIssue.getIssue();
		IssueItem childItem = new IssueItem(item, childIssue);
		childItems.add(childItem);
	    }

	    // Install item children.
	    Agenda issueAgenda = controller.getIssueAgenda();
	    installSubitems(issueAgenda, item,
			    childItems, childPlanIssues);
	}

	// Recursive expansion.
	if (childPlanIssues != null) {
	    for (Iterator i = childPlanIssues.iterator(); i.hasNext();) {
		PlanIssue childPlanIssue = (PlanIssue)i.next();
		if (childPlanIssue.getExpansion() != null)
		    installIssueExpansion(childPlanIssue);
	    }
	}

    }

    void installExpansion(PlanNode node) {
	Activity activity = node.getActivity();
	ActivityItem item = findItemForActivity(activity);
	PlanRefinement refine = 
	    plan.getPlanRefinement(node.getExpansion());
	Debug.expect(refine.getExpands().equals(node.getId()));

	// Variables
	// /\/: If the node isn't really expanded, because the
	// plan refinement is just providing some contstraints
	// that were added in another way, then we can't give
	// the item a var-table, because (/\/) that would
	// make item.isExpanded() == true.
	if (!refine.isNotExpansion())
	    item.setVarTable
		(makeVarTable(item, refine.getPlanVariableDeclarations()));

	// Subactivities
	List childNodes = refine.getPlanNodes();
	if (childNodes == null) {
	    // No subactivities, so no orderings.
	    Debug.expect(refine.getOrderings().isEmpty());
	}
	else {
	    // Create and attach children
	    List childItems = makeSubactivities(item, childNodes);
	    Agenda activityAgenda = controller.getActivityAgenda();
	    Map childMap = installSubitems(activityAgenda, item,
					   childItems, childNodes);
	    // Orderings
	    installOrderings(item, refine.getOrderings(), childMap);
	    item.ensureChildrenLinkedToParent();
	}

	// Other constraints
	installConstraints(item, refine.getConstraints(Constraint.class));

	// Notify anything that cares that the item was expanded.
	Debug.expect(refine.isNotExpansion() != item.isExpanded());
	item.fireAgendaItemEdited();

	// Recursive expansion.
	if (childNodes != null) {
	    for (Iterator i = childNodes.iterator(); i.hasNext();) {
		PlanNode childNode = (PlanNode)i.next();
		if (childNode.getExpansion() != null)
		    installExpansion(childNode);
	    }
	}

    }

    ActivityItem findItemForActivity(Activity activity) {
	List activityItems = controller.getActivityAgenda().getItems();
	return (ActivityItem)findItem(activity, activityItems);
    }

    IssueItem findItemForIssue(Issue issue) {
	List issueItems = controller.getIssueAgenda().getItems();
	return (IssueItem)findItem(issue, issueItems);
    }

    AgendaItem findItem(TaskItem target, List items) {
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (item.getAbout() == target)
		return item;
	}
	throw new IllegalArgumentException
	    ("Can't find item for " + target);
    }

    Map makeVarTable(AgendaItem item, List dclList) {
	Map result = new HashMap();
	for (Iterator i = Collect.iterator(dclList); i.hasNext();) {
	    PlanVariableDeclaration dcl =
		(PlanVariableDeclaration)i.next();
	    Variable v = getVariableFromId(dcl.getId());
	    result.put(v.getName(), v);
	    // /\/: The next line marks the var as local.
	    // See the Variable isGlobal method and the PNode
	    // MakeVarIfUnbound inner class.
	    v.setSourceNode(item);
	}
	return result;
    }

    List makeSubactivities(ActivityItem parentItem, List childNodes) {
	List result = new LinkedList();
	for (Iterator i = childNodes.iterator(); i.hasNext();) {
	    PlanNode childNode = (PlanNode)i.next();
	    Activity childActivity = childNode.getActivity();
	    ActivityItem childItem = new ActivityItem(parentItem, 
						      childActivity);
	    idToItemMap.put(childNode.getId(), childItem);
	    result.add(childItem);
	}
	return result;
    }

    Map installSubitems(Agenda agenda, // for activities or issues
			AgendaItem item,
			List childItems,
			List childNodes) {
	Map childMap = makeNameToChildMap(childItems, childNodes);
	item.setChildren(childItems);
	for (Iterator i = childItems.iterator(); i.hasNext();) {
	    AgendaItem childItem = (AgendaItem)i.next();
	    Debug.expect(childItem.getParent() == item);
	    Debug.expect(childItem.getLevel() == item.getLevel() + 1);
	    agenda.addItem(childItem);
	}
	return childMap;
    }

    Map makeNameToChildMap(List childItems, List childNodes) {
	// childNodes might be a list of PlanNodes or of PlanIssues.
	Debug.expect(childItems.size() == childNodes.size());

	Map result = new HashMap();
	for (Iterator i = childItems.iterator(),
		      n = childNodes.iterator();
	     i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    AbstractPlanItem node = (AbstractPlanItem)n.next();
	    result.put(node.getId(), item);
	}
	return result;
    }

    void installOrderings(ActivityItem item, ListOfOrdering orderings,
			  Map nameToChildMap) {
	if (!orderings.isEmpty())
	    item.processOrderings(orderings, nameToChildMap);
	modelManager.addOrderingsAsTimeConstraints
	                 (item, nameToChildMap, orderings);
    }

    void installConstraints(ActivityItem item, List constraints) {
	if (constraints != null)
	    modelManager.addConstraints(item, constraints);
    }

}
