/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Apr 23 15:15:41 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.event.*;

import java.lang.reflect.Constructor;

import java.io.File;		// for tests
import java.io.IOException;	// for tests
import java.net.URL;		// for tests

import java.util.*;

import ix.icore.IXAgent;
import ix.icore.Sendable;
import ix.icore.domain.Domain;
import ix.icore.event.AgentNameListener;
import ix.icore.event.AgentNameEvent;

import ix.iview.*;

import ix.iface.util.ToolManager;
import ix.iface.util.ToolController;
import ix.iface.util.ChainingMenu;
import ix.iface.util.CatchingActionListener;

import ix.ichat.ChatFrame;
import ix.itest.SendPanel;

import ix.ip2.test.*;

import ix.util.*;
import ix.util.xml.*;

/**
 * The main frame of an I-X panel user interface.
 */
public abstract class PanelFrame extends JFrame
    implements ActionListener, AgentNameListener {

    protected IXAgent agent;

    protected ToolManager toolManager = new ToolManager();

    protected JMenu testMenu =
	new ChainingMenu("Test", true); // tear-off "Test" menu

    public PanelFrame(IXAgent agent) {
	super(agent.getAgentDisplayName());

	this.agent = agent;

	agent.addAgentNameListener(this);

	// Make sure a "close" doesn't just leave it running in the
	// background...
	setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent e) {
		Debug.noteln("Main frame windowClosing event");
		if (Util.dialogConfirms(PanelFrame.this,
		       "Are you sure you want to exit?"))
		    System.exit(0);		// should call agent exit /\/
	    }
	});

    }

    public IXAgent getAgent() {
	return agent;
    }

    public ToolManager getToolManager() {
	return toolManager;
    }

    /**
     * Handles a change in the panel's symbol-name.  This method
     * changes the frame's title to me the possibly revised display
     * name.  Subclasses may need to take additional steps.
     */
    public void symbolNameChanged(AgentNameEvent e) {
	setTitle(agent.getAgentDisplayName());
    }

    /**
     * Constructs the domain editor for this panel.  If a
     * <code>domain-editor-class</code> parameter has been specified,
     * its value is taken as the name of a class that implements the
     * InternalDomainEditor interface and supplies an (IXAgent, Domain)
     * constructor.  If an exception is thrown when that constructor
     * is called, or if no class parameter was specified, a default
     * class is instantiated instead.
     */
    public InternalDomainEditor makeDomainEditor(IXAgent agent, Domain dom) {
	// New editor: ix.iview.DomainEditor
	// Old editor: ix.iview.SimpleDomainEditor
	String className =
	    Parameters.getParameter
	        ("domain-editor-class",
		 "ix.iview.DomainEditor"
		 /* was "ix.iview.SimpleDomainEditor" */);
	try {
	    Debug.noteln("Trying domain editor class", className);
	    Class c = Class.forName(className);
	    Class[] consClasses = {IXAgent.class, Domain.class};
	    Constructor cons = c.getConstructor(consClasses);
	    InternalDomainEditor ed =
		(InternalDomainEditor)
		    cons.newInstance(new Object[] {agent, dom});
	    // /\/: Special case to delete when possible:
	    if (ed instanceof JFrame)
		((JFrame)ed).setIconImage
		    (ix.iface.util.IconImage
		        .getIconImage("ip2-edit-icon.gif"));
	    return ed;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    Debug.noteln("Creating a SimpleDomainEditor instead.");
	    return new SimpleDomainEditor(agent, dom);
	}
    }

    // /\/: getDomainEditor() is called only when an agenda viewer wants
    // to call the domain-editor's saveExpansion method, when it's
    // told to do that by the ActivityEditor's "Save As Refinement"
    // command.  Unfortunately, nothing ensures that the domain-editor
    // becomes visible.  The agenda viewers ought to do it (or the
    // ActivityEditor should work directly with the domain-editor),
    // but for now at least, we'll make it visible here.
    public InternalDomainEditor getDomainEditor() {
	return (InternalDomainEditor)
	    toolManager.findToolElseError("Domain Editor")
	        .ensureToolVisible();
    }

    // /\/: Must find a better way to do this rather than have
    // this method everywhere.
    protected JMenuItem makeMenuItem(String text) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(this));
	return item;
    }

    public void addTool(ToolController tc) {
	toolManager.addTool(tc);
    }

    public SendPanel getSendPanelVisible() {
	return getChatFrameVisible().getSendPanel();
    }

    public SendPanel ensureSendPanelExists() {
	return ensureChatFrameExists().getSendPanel();
    }

    public ChatFrame getChatFrameVisible() {
	return (ChatFrame)
	    toolManager.findToolElseError("Messenger")
	        .ensureToolVisible();
    }

    public ChatFrame ensureChatFrameExists() {
	return (ChatFrame)
	    toolManager.findToolElseError("Messenger")
	        .ensureTool();
    }

    /*
     * The "Test" menu
     */

    /**
     * Adds a test by adding a menu item to the "Test" menu.
     * The action listener will be wrapped in a CatchingActionListener.
     * All test additions should go directly or indirectly
     * through this method.
     *
     * @see ix.iface.util.CatchingActionListener
     */
    public void addTest(String menuText, ActionListener listener) {
	Debug.noteln("Adding test", menuText);
	JMenuItem item = new JMenuItem(menuText);
	item.addActionListener(CatchingActionListener.listener(listener));
	testMenu.setEnabled(true);
	testMenu.add(item);
    }

    /**
     * Adds a separator to this panel's "Test" menu.
     */
    public void addTestSeparator() {
	testMenu.addSeparator();
    }

    public void addTestSubmenu(String menuText, List testItems) {
	JMenu submenu = new ChainingMenu(menuText);
	testMenu.add(submenu);
	testMenu.setEnabled(true); // normally done by addTest
	JMenu saved = testMenu;
	try {
	    testMenu = submenu;
	    addTests(testItems);
	}
	finally {
	    testMenu = saved;
	}
    }

    public void addTestResources(List resourceNames) {
	for (Iterator i = resourceNames.iterator(); i.hasNext();) {
	    String resourceName = (String)i.next();
	    try {
		addTests(resourceName);
	    }
	    catch (Throwable t) {
		Debug.displayException
		    ("Problem loading test-menu " +
		     Strings.quote(resourceName), t);
	    }
	}
    }

    public void addTests(String resourceName) throws IOException {
	Debug.noteln("Loading tests from", resourceName);
	URL url = XML.toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException
		("Can't find test menu " + Strings.quote(resourceName));
	List items = (List)XML.readObject(List.class, url);
	addTests(items);
    }

    public void addTests(List items) {
	if (items != null) {
	    for (Iterator i = items.iterator(); i.hasNext();) {
		TestElement test = (TestElement)i.next();
		test.addToMenu(this);
	    }
	}
    }

}

// Issues:
// * Not clear that this should exist, or if it does that it should
//   have this functionality.
